<?php
/*
  This code is part of FusionDirectory (http://www.fusiondirectory.org)
  Copyright (C) 2010 Antoine Gallavardin
  Copyright (C) 2011-2018 FusionDirectory project

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
*/

class dashboardSystems extends simplePlugin
{
  /* default values */
  var $default_start_computer_id = 0;

  static function plInfo()
  {
    return array(
      'plShortName'   => _('Systems'),
      'plDescription' => _('Statistics and information about systems'),
      'plObjectType'  => array('dashboard'),
      'plPriority'    => 20,

      'plProvidedAcls'  => array()
    );
  }

  static function getAttributesInfo()
  {
    return array(
      'stats' => array(
        'name'  => _('Statistics'),
        'attrs' => array(new FakeAttribute('stats')),
        'template' => get_template_path('systems_stats.tpl', TRUE, dirname(__FILE__)),
      ),
      'pc_ids' => array(
        'name'  => _('Computer name to use by unit'),
        'attrs' => array(new FakeAttribute('pc_ids')),
        'template' => get_template_path('systems_pcids.tpl', TRUE, dirname(__FILE__)),
      ),
    );
  }

  function __construct ($dn = NULL, $object = NULL, $parent = NULL, $mainTab = FALSE)
  {
    parent::__construct($dn, $object, $parent, $mainTab);

    $this->stats = array(
      'systems' => $this->systems_stats(),
      'argonaut' => $this->argonaut_stats(),
    );
    $this->pc_ids = $this->computer_ids_rules();
  }

  protected function systems_stats ()
  {
    global $config;
    $ldap = $config->get_ldap_link();

    /* Statistics */
    $stats = array(
      array('name'  => _('Workstations'),
            'type'  => 'workstation',
            'img'   => 'geticon.php?context=devices&icon=computer&size=16'
      ),
      array('name'  => _('Servers'),
            'type'  => 'server',
            'img'   => 'geticon.php?context=devices&icon=server&size=16'
      ),
      array('name'    => _('Windows Workstations'),
            'type'    => 'workstation',
            'filter'  => '(objectClass=sambaSamAccount)',
            'img'     => 'geticon.php?context=devices&icon=computer-windows&size=16'
      ),
      array('name'  => _('Terminals'),
            'type'  => 'terminal',
            'img'   => 'geticon.php?context=devices&icon=terminal&size=16'
      ),
      array('name'  => _('Printers'),
            'type'  => 'printer',
            'img'   => 'geticon.php?context=devices&icon=printer&size=16'
      ),
      array('name'  => _('Phones'),
            'type'  => 'phone',
            'img'   => 'geticon.php?context=devices&icon=telephone&size=16'
      ),
      array('name'  => _('Components'),
            'type'  => 'component',
            'img'   => 'geticon.php?context=devices&icon=network-device&size=16'
      ),
      array('name'  => _('Mobile phones'),
            'type'  => 'mobilePhone',
            'img'   => 'geticon.php?context=devices&icon=phone&size=16'),
    );

    foreach ($stats as $key => &$stat) {
      try {
        $stat['nb'] = count(objects::ls($stat['type'], NULL, NULL, (isset($stat['filter']) ? $stat['filter'] : ''), TRUE));
        if ($stat['nb'] == 0) {
          unset($stats[$key]);
        }
      } catch (FusionDirectoryException $e) {
        unset($stats[$key]);
        msg_dialog::display(
          _('LDAP error'),
          sprintf(_('Statistics for type "%s" could not be computed because of the following error: %s'), $stat['type'], $e->getMessage()),
          ERROR_DIALOG
        );
      }
    }
    unset($stat);

    return $stats;
  }

  protected function argonaut_stats ()
  {
    global $config, $ui;

    if (!class_available('argonautServer')) {
      return FALSE;
    }

    $argonaut_servers = objects::ls(
      'server',
      array('cn' => 'raw','ipHostNumber' => 'raw','argonautProtocol' => 'raw','argonautPort' => 'raw'),
      NULL,
      '(objectClass=argonautServer)',
      TRUE
    );
    $nb_argonaut_server = count($argonaut_servers);
    $argonaut_server    = array();
    if ($nb_argonaut_server == 1) {
      $attrs = reset($argonaut_servers);
      foreach (array('cn','ipHostNumber','argonautProtocol','argonautPort') as $key) {
        $argonaut_server[$key] = $attrs[$key][0];
      }
      $argonaut_server['link'] = objects::link(key($argonaut_servers), 'server', 'service_serviceArgonaut', $attrs);
    }

    if (strpos($ui->get_permissions($config->current['BASE'], 'server/argonautClient', ''), 'r') === FALSE) {
      $nb_argonaut_clients = 0;
    } else {
      $ldap = $config->get_ldap_link();
      $ldap->cd($config->current['BASE']);
      $ldap->search('(objectClass=argonautClient)', array('cn'));
      $nb_argonaut_clients = $ldap->count();
    }

    return array(
      'nb_servers'  => $nb_argonaut_server,
      'server'      => $argonaut_server,
      'nb_clients'  => $nb_argonaut_clients,
    );
  }

  protected function computer_ids_rules ()
  {
    global $config;
    $ldap = $config->get_ldap_link();
    $ldap->cd($config->current['BASE']);
    /* Begin of code for selecting next computer IDs
    * Global variable is board_next_computer_ids_rule
    * syntax is :
    * 7 : nb total of digit
    * LYP3 : prefix to use
    * each prefix is followed by the first item flagged by an equal sign
    * ex : "7;LYP0=3;LYP1=6;LYP2;LYP3"
    * if to item is doned it's 0 by default
    *
    */

    $output_next_computer_ids = "";
    $computer_ids_rules = $config->get_cfg_value('dashboardPrefix', array('PC'));
    if (!is_array($computer_ids_rules)) {
      $computer_ids_rules = array($computer_ids_rules);
    }

    $nb_digits = $config->get_cfg_value('dashboardNumberOfDigit', 3);

    /* running all the table */
    foreach ($computer_ids_rules as $rule) {
      /* aray initialization*/
      $array_complete_list  = array();
      $array_real_list      = array();
      $unused_computer_ids  = array();
      /* get computer ids configuration */
      $config_ids = explode("=", $rule);
      /* fist is is the prefix */
      $prefix = $config_ids[0];
      /* second (if specified) is the first item */
      if (!isset($config_ids[1])) {
        $start_id = $this->default_start_computer_id;
      } else {
        $start_id = $config_ids[1];
      }

      $output_next_computer_ids = $output_next_computer_ids.'<tr><th style="padding:4px;border:1px solid #BBB;">'.$prefix."</th>";
      $nb_digits_suffix         = $nb_digits - strlen($prefix);

      /* generation of list of suffixe */
      for ($d = $start_id;$d < pow(10, $nb_digits_suffix);$d++) {
        /* padding : 34 on 4 digit become : 0034*/
        $array_complete_list[] = str_pad($d, $nb_digits_suffix, "0", STR_PAD_LEFT);
      }

      /* request of all computer beginning by the prefix */
      $request = '(&(cn='.$prefix.'*)(ou:dn:=systems))';
      $ldap->search($request, array("cn"));
      while ($attrs = $ldap->fetch()) {
        /* if a computer is a windows host, we have to delete the $ at the end */
        $computer_id        = str_replace("$", "", $attrs["cn"][0]);
        $array_real_list[]  = substr($computer_id, -$nb_digits_suffix);
      }
      /* make dfference between real and complete list */
      $unused_computer_ids = array_diff($array_complete_list, $array_real_list);
      asort($unused_computer_ids);
      $unused_computer_ids = array_values($unused_computer_ids);
      /* we take the 5 first */

      for ($r = 0;$r <= 5;$r++) {
        $output_next_computer_ids .= '<td style="padding:4px;border-bottom:1px solid #BBB;">'.$prefix.$unused_computer_ids[$r]."</td>";
      }
      $output_next_computer_ids .= "</tr>";
    }

    return $output_next_computer_ids;
  }
}
?>
