use util:serialize;

/**
 * This class is the base class for all requests (sent from the client to the server). We rely on
 * the built-in serialization for sending and receiving them.
 */
class Request : serializable {
	// Called by the server to execute the request. Returns a suitable message, or "null" if the
	// client should be disconnected.
	Response? execute(ServerConn server, Database db) {
		error("Unknown message");
	}

	// Helper to create an error message.
	protected Response error(Str message) {
		ErrorResponse(message);
	}
}


/**
 * Authentication request. Expected to be the first message sent by a client.
 *
 * Returns either an AuthReply, AuthLoginReply, or an error.
 */
class AuthRequest : extends Request, serializable {
	// The client key.
	Str key;

	init(Str key) {
		init { key = key; }
	}

	// NOTE: This is handled as a special case, so we don't override "execute" here.
}

/**
 * Log out from this client. No additional data needed.
 */
class LogoutRequest : extends Request, serializable {
	Response? execute(ServerConn server, Database db) : override {
		db.logout(server.clientId);
		LogoutResponse();
	}
}

/**
 * Change nickname.
 */
class ChangeNicknameRequest : extends Request, serializable {
	Str newName;

	init(Str newName) {
		init { newName = newName; }
	}

	Response? execute(ServerConn server, Database db) : override {
		db.changeName(server.userId, newName);
		AuthResponse(newName);
	}
}

/**
 * Ask for the leaderboard.
 */
class LeaderboardRequest : extends Request, serializable {
	Response? execute(ServerConn server, Database db) : override {
		ScorePair[] scores;
		for (user, score in db.allScores()) {
			scores << ScorePair(user, score);
		}

		scores.sort();

		// Pick the top 10, and the current user.
		Bool currentFound = false;
		Score[] result;
		for (i, score in scores) {
			Bool add = (score.user == server.userId);
			currentFound |= add;
			add |= i < 10;

			if (currentFound & !add)
				break;

			if (add) {
				if (user = db.findUserName(score.user))
					result << Score(user, score.score, i.int + 1);
			}
		}

		LeaderboardResponse(result);
	}
}

// Pair for use when working with the scoreboard.
private value ScorePair {
	Int user;
	Int score;

	init(Int user, Int score) {
		init { user = user; score = score; }
	}

	Bool <(ScorePair o) {
		score > o.score;
	}
}

/**
 * Ask for problems..
 */
class UnsolvedRequest : extends Request, serializable {
	// Only problems with an error currently.
	Bool currentError;

	init(Bool currentError) {
		init {
			currentError = currentError;
		}
	}

	Response? execute(ServerConn server, Database db) : override {
		ProblemResponse(db.unsolvedProblems(server.userId, currentError));
	}
}

/**
 * Ask for solved problems.
 */
class SolvedProblemsRequest : extends Request, serializable {
	Response? execute(ServerConn server, Database db) : override {
		ProblemResponse(db.solvedProblems(server.userId));
	}
}

/**
 * Ask for a user's own problems.
 */
class OwnProblemsRequest : extends Request, serializable {
	Response? execute(ServerConn server, Database db) : override {
		ProblemResponse(db.ownProblems(server.userId));
	}
}

/**
 * Ask for more detailed stats of a problem.
 */
class StatsRequest : extends Request, serializable {
	// Problem we are interested in.
	Int problem;

	// Include improvments?
	Bool improvements;

	// Include solutions.
	Bool solutions;

	init(Int problem, Bool improvements, Bool solutions) {
		init {
			problem = problem;
			improvements = improvements;
			solutions = solutions;
		}
	}

	init(Problem problem, Bool improvements, Bool solutions) {
		init {
			problem = problem.id;
			improvements = improvements;
			solutions = solutions;
		}
	}

	Response? execute(ServerConn server, Database db) : override {
		ProblemInfo[] improved;
		if (improvements)
			improved = db.improvementsTo(problem, server.userId);
		Solution[] solution;
		if (solutions)
			solution = db.solutionsTo(problem);
		ProblemInfo? parent = db.parentTo(problem, server.userId);
		Bool solved = db.solvedProblem(server.userId, problem);
		StatsResponse(parent, solved, improved, solution);
	}
}

/**
 * Ask for details about a problem (including source).
 */
class DetailsRequest : extends Request, serializable {
	Int problemId;

	init(Int id) {
		init { problemId = id; }
	}

	Response? execute(ServerConn server, Database db) : override {
		DetailsResponse(db.problemDetails(problemId, server.userId));
	}
}

/**
 * Submit a solution to a problem.
 */
class PostSolutionRequest : extends Request, serializable {
	// Solution to this problem.
	Int problemId;

	// Type of solution.
	Str type;

	// Solution data.
	Str solution;

	init(Int problemId, Str type, Str solution) {
		init {
			problemId = problemId;
			type = type;
			solution = solution;
		}
	}

	Response? execute(ServerConn server, Database db) : override {
		var id = db.createSolution(server.userId, problemId, type, solution);
		PostSolutionResponse(id);
	}
}

/**
 * Publish a new problem.
 */
class NewProblemRequest : extends Request, serializable {
	Str title;
	Code main;
	Code impl;
	Code refImpl;

	// Is there a concurrency error in the combination main + impl currently?
	Bool currentError;

	init(Str title, Code main, Code impl, Code refImpl, Bool currentError) {
		init {
			title = title;
			main = main;
			impl = impl;
			refImpl = refImpl;
			currentError = currentError;
		}
	}

	Response? execute(ServerConn server, Database db) : override {
		Int id = db.createProblem(server.userId, title, main, impl, refImpl, currentError);
		NewProblemResponse(id);
	}
}

/**
 * Publish an improvement to an existing problem.
 */
class ImprovedProblemRequest : extends Request, serializable {
	Int originalId;
	Str title;
	Code? main;
	Code? impl;

	Bool currentError;

	init(Int originalId, Str title, Code? main, Code? impl, Bool currentError) {
		init {
			originalId = originalId;
			title = title;
			main = main;
			impl = impl;
			currentError = currentError;
		}
	}

	Response? execute(ServerConn server, Database db) : override {
		Int id = db.improvedProblem(server.userId, originalId, title, main, impl, currentError);
		NewProblemResponse(id);
	}
}
