// Copyright 2009 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package tls

import (
	"bytes"
	"crypto/ecdsa"
	"crypto/elliptic"
	"crypto/rsa"
	"crypto/x509"
	"encoding/hex"
	"encoding/pem"
	"flag"
	"fmt"
	"io"
	"log"
	"math/big"
	"net"
	"os"
	"strconv"
	"strings"
	"sync"
	"testing"
	"time"
)

type zeroSource struct{}

func (zeroSource) Read(b []byte) (n int, err error) {
	for i := range b {
		b[i] = 0
	}

	return len(b), nil
}

var testConfig *Config

func init() {
	testConfig = new(Config)
	testConfig.Time = func() time.Time { return time.Unix(0, 0) }
	testConfig.Rand = zeroSource{}
	testConfig.Certificates = make([]Certificate, 2)
	testConfig.Certificates[0].Certificate = [][]byte{testRSACertificate}
	testConfig.Certificates[0].PrivateKey = testRSAPrivateKey
	testConfig.Certificates[1].Certificate = [][]byte{testSNICertificate}
	testConfig.Certificates[1].PrivateKey = testRSAPrivateKey
	testConfig.BuildNameToCertificate()
	testConfig.CipherSuites = []uint16{TLS_RSA_WITH_RC4_128_SHA}
	testConfig.InsecureSkipVerify = true
	testConfig.MinVersion = VersionSSL30
	testConfig.MaxVersion = VersionTLS10
}

func testClientHelloFailure(t *testing.T, m handshakeMessage, expected error) {
	// Create in-memory network connection,
	// send message to server.  Should return
	// expected error.
	c, s := net.Pipe()
	go func() {
		cli := Client(c, testConfig)
		if ch, ok := m.(*clientHelloMsg); ok {
			cli.vers = ch.vers
		}
		cli.writeRecord(recordTypeHandshake, m.marshal())
		c.Close()
	}()
	err := Server(s, testConfig).Handshake()
	s.Close()
	if e, ok := err.(*net.OpError); !ok || e.Err != expected {
		t.Errorf("Got error: %s; expected: %s", err, expected)
	}
}

func TestSimpleError(t *testing.T) {
	testClientHelloFailure(t, &serverHelloDoneMsg{}, alertUnexpectedMessage)
}

var badProtocolVersions = []uint16{0x0000, 0x0005, 0x0100, 0x0105, 0x0200, 0x0205}

func TestRejectBadProtocolVersion(t *testing.T) {
	for _, v := range badProtocolVersions {
		testClientHelloFailure(t, &clientHelloMsg{vers: v}, alertProtocolVersion)
	}
}

func TestNoSuiteOverlap(t *testing.T) {
	clientHello := &clientHelloMsg{
		vers:               0x0301,
		cipherSuites:       []uint16{0xff00},
		compressionMethods: []uint8{0},
	}
	testClientHelloFailure(t, clientHello, alertHandshakeFailure)
}

func TestNoCompressionOverlap(t *testing.T) {
	clientHello := &clientHelloMsg{
		vers:               0x0301,
		cipherSuites:       []uint16{TLS_RSA_WITH_RC4_128_SHA},
		compressionMethods: []uint8{0xff},
	}
	testClientHelloFailure(t, clientHello, alertHandshakeFailure)
}

func TestTLS12OnlyCipherSuites(t *testing.T) {
	// Test that a Server doesn't select a TLS 1.2-only cipher suite when
	// the client negotiates TLS 1.1.
	var zeros [32]byte

	clientHello := &clientHelloMsg{
		vers:   VersionTLS11,
		random: zeros[:],
		cipherSuites: []uint16{
			// The Server, by default, will use the client's
			// preference order. So the GCM cipher suite
			// will be selected unless it's excluded because
			// of the version in this ClientHello.
			TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256,
			TLS_RSA_WITH_RC4_128_SHA,
		},
		compressionMethods: []uint8{compressionNone},
		supportedCurves:    []uint16{curveP256, curveP384, curveP521},
		supportedPoints:    []uint8{pointFormatUncompressed},
	}

	c, s := net.Pipe()
	var reply interface{}
	var clientErr error
	go func() {
		cli := Client(c, testConfig)
		cli.vers = clientHello.vers
		cli.writeRecord(recordTypeHandshake, clientHello.marshal())
		reply, clientErr = cli.readHandshake()
		c.Close()
	}()
	config := *testConfig
	config.CipherSuites = clientHello.cipherSuites
	Server(s, &config).Handshake()
	s.Close()
	if clientErr != nil {
		t.Fatal(clientErr)
	}
	serverHello, ok := reply.(*serverHelloMsg)
	if !ok {
		t.Fatalf("didn't get ServerHello message in reply. Got %v\n", reply)
	}
	if s := serverHello.cipherSuite; s != TLS_RSA_WITH_RC4_128_SHA {
		t.Fatalf("bad cipher suite from server: %x", s)
	}
}

func TestAlertForwarding(t *testing.T) {
	c, s := net.Pipe()
	go func() {
		Client(c, testConfig).sendAlert(alertUnknownCA)
		c.Close()
	}()

	err := Server(s, testConfig).Handshake()
	s.Close()
	if e, ok := err.(*net.OpError); !ok || e.Err != error(alertUnknownCA) {
		t.Errorf("Got error: %s; expected: %s", err, error(alertUnknownCA))
	}
}

func TestClose(t *testing.T) {
	c, s := net.Pipe()
	go c.Close()

	err := Server(s, testConfig).Handshake()
	s.Close()
	if err != io.EOF {
		t.Errorf("Got error: %s; expected: %s", err, io.EOF)
	}
}

func testHandshake(clientConfig, serverConfig *Config) (state ConnectionState, err error) {
	c, s := net.Pipe()
	go func() {
		cli := Client(c, clientConfig)
		cli.Handshake()
		c.Close()
	}()
	server := Server(s, serverConfig)
	err = server.Handshake()
	if err == nil {
		state = server.ConnectionState()
	}
	s.Close()
	return
}

func TestCipherSuitePreference(t *testing.T) {
	serverConfig := &Config{
		CipherSuites: []uint16{TLS_RSA_WITH_RC4_128_SHA, TLS_RSA_WITH_AES_128_CBC_SHA, TLS_ECDHE_RSA_WITH_RC4_128_SHA},
		Certificates: testConfig.Certificates,
		MaxVersion:   VersionTLS11,
	}
	clientConfig := &Config{
		CipherSuites:       []uint16{TLS_RSA_WITH_AES_128_CBC_SHA, TLS_RSA_WITH_RC4_128_SHA},
		InsecureSkipVerify: true,
	}
	state, err := testHandshake(clientConfig, serverConfig)
	if err != nil {
		t.Fatalf("handshake failed: %s", err)
	}
	if state.CipherSuite != TLS_RSA_WITH_AES_128_CBC_SHA {
		// By default the server should use the client's preference.
		t.Fatalf("Client's preference was not used, got %x", state.CipherSuite)
	}

	serverConfig.PreferServerCipherSuites = true
	state, err = testHandshake(clientConfig, serverConfig)
	if err != nil {
		t.Fatalf("handshake failed: %s", err)
	}
	if state.CipherSuite != TLS_RSA_WITH_RC4_128_SHA {
		t.Fatalf("Server's preference was not used, got %x", state.CipherSuite)
	}
}

func testServerScript(t *testing.T, name string, serverScript [][]byte, config *Config, peers []*x509.Certificate) {
	c, s := net.Pipe()
	srv := Server(s, config)
	pchan := make(chan []*x509.Certificate, 1)
	go func() {
		srv.Write([]byte("hello, world\n"))
		srv.Close()
		s.Close()
		st := srv.ConnectionState()
		pchan <- st.PeerCertificates
	}()

	for i, b := range serverScript {
		if i%2 == 0 {
			c.Write(b)
			continue
		}
		bb := make([]byte, len(b))
		n, err := io.ReadFull(c, bb)
		if err != nil {
			t.Fatalf("%s #%d: %s\nRead %d, wanted %d, got %x, wanted %x\n", name, i, err, n, len(bb), bb[:n], b)
		}
		if !bytes.Equal(b, bb) {
			t.Fatalf("%s #%d: mismatch on read: got:%x want:%x", name, i, bb, b)
		}
	}
	c.Close()

	if peers != nil {
		gotpeers := <-pchan
		if len(peers) == len(gotpeers) {
			for i := range peers {
				if !peers[i].Equal(gotpeers[i]) {
					t.Fatalf("%s: mismatch on peer cert %d", name, i)
				}
			}
		} else {
			t.Fatalf("%s: mismatch on peer list length: %d (wanted) != %d (got)", name, len(peers), len(gotpeers))
		}
	}
}

func TestHandshakeServerRSARC4(t *testing.T) {
	testServerScript(t, "RSA-RC4", rsaRC4ServerScript, testConfig, nil)
}

func TestHandshakeServerRSA3DES(t *testing.T) {
	des3Config := new(Config)
	*des3Config = *testConfig
	des3Config.CipherSuites = []uint16{TLS_RSA_WITH_3DES_EDE_CBC_SHA}
	testServerScript(t, "RSA-3DES", rsaDES3ServerScript, des3Config, nil)
}

func TestHandshakeServerRSAAES(t *testing.T) {
	aesConfig := new(Config)
	*aesConfig = *testConfig
	aesConfig.CipherSuites = []uint16{TLS_RSA_WITH_AES_128_CBC_SHA}
	testServerScript(t, "RSA-AES", rsaAESServerScript, aesConfig, nil)
}

func TestHandshakeServerECDHEECDSAAES(t *testing.T) {
	ecdsaConfig := new(Config)
	*ecdsaConfig = *testConfig
	ecdsaConfig.Certificates = make([]Certificate, 1)
	ecdsaConfig.Certificates[0].Certificate = [][]byte{testECDSACertificate}
	ecdsaConfig.Certificates[0].PrivateKey = testECDSAPrivateKey
	ecdsaConfig.BuildNameToCertificate()
	ecdsaConfig.CipherSuites = []uint16{TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA}
	testServerScript(t, "ECDHE-ECDSA-AES", ecdheECDSAAESServerScript, ecdsaConfig, nil)
}

func TestHandshakeServerSSLv3(t *testing.T) {
	testServerScript(t, "SSLv3", sslv3ServerScript, testConfig, nil)
}

// TestHandshakeServerSNI involves a client sending an SNI extension of
// "snitest.com", which happens to match the CN of testSNICertificate. The test
// verifies that the server correctly selects that certificate.
func TestHandshakeServerSNI(t *testing.T) {
	testServerScript(t, "SNI", selectCertificateBySNIScript, testConfig, nil)
}

func TestResumption(t *testing.T) {
	testServerScript(t, "IssueTicket", issueSessionTicketTest, testConfig, nil)
	testServerScript(t, "Resume", serverResumeTest, testConfig, nil)
}

func TestTLS12ClientCertServer(t *testing.T) {
	config := *testConfig
	config.MaxVersion = VersionTLS12
	config.ClientAuth = RequireAnyClientCert
	config.CipherSuites = []uint16{TLS_ECDHE_RSA_WITH_RC4_128_SHA}

	testServerScript(t, "TLS12", tls12ServerScript, &config, nil)
}

type clientauthTest struct {
	name       string
	clientauth ClientAuthType
	peers      []*x509.Certificate
	script     [][]byte
}

func TestClientAuthRSA(t *testing.T) {
	for _, cat := range clientauthRSATests {
		t.Log("running", cat.name)
		cfg := new(Config)
		*cfg = *testConfig
		cfg.ClientAuth = cat.clientauth
		testServerScript(t, cat.name, cat.script, cfg, cat.peers)
	}
}

func TestClientAuthECDSA(t *testing.T) {
	for _, cat := range clientauthECDSATests {
		t.Log("running", cat.name)
		cfg := new(Config)
		*cfg = *testConfig
		cfg.Certificates = make([]Certificate, 1)
		cfg.Certificates[0].Certificate = [][]byte{testECDSACertificate}
		cfg.Certificates[0].PrivateKey = testECDSAPrivateKey
		cfg.BuildNameToCertificate()
		cfg.CipherSuites = []uint16{TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA}
		cfg.ClientAuth = cat.clientauth
		testServerScript(t, cat.name, cat.script, cfg, cat.peers)
	}
}

// TestCipherSuiteCertPreferance ensures that we select an RSA ciphersuite with
// an RSA certificate and an ECDSA ciphersuite with an ECDSA certificate.
func TestCipherSuiteCertPreferance(t *testing.T) {
	var config = *testConfig
	config.CipherSuites = []uint16{TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA, TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA}
	config.MaxVersion = VersionTLS11
	config.PreferServerCipherSuites = true
	testServerScript(t, "CipherSuiteCertPreference", tls11ECDHEAESServerScript, &config, nil)

	config = *testConfig
	config.CipherSuites = []uint16{TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA, TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA}
	config.Certificates = []Certificate{
		Certificate{
			Certificate: [][]byte{testECDSACertificate},
			PrivateKey:  testECDSAPrivateKey,
		},
	}
	config.BuildNameToCertificate()
	config.PreferServerCipherSuites = true
	testServerScript(t, "CipherSuiteCertPreference2", ecdheECDSAAESServerScript, &config, nil)
}

func TestTLS11Server(t *testing.T) {
	var config = *testConfig
	config.CipherSuites = []uint16{TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA}
	config.MaxVersion = VersionTLS11
	testServerScript(t, "TLS11", tls11ECDHEAESServerScript, &config, nil)
}

func TestAESGCM(t *testing.T) {
	var config = *testConfig
	config.CipherSuites = []uint16{TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256}
	config.MaxVersion = VersionTLS12
	testServerScript(t, "AES-GCM", aesGCMServerScript, &config, nil)
}

// recordingConn is a net.Conn that records the traffic that passes through it.
// WriteTo can be used to produce Go code that contains the recorded traffic.
type recordingConn struct {
	net.Conn
	lock             sync.Mutex
	flows            [][]byte
	currentlyReading bool
}

func (r *recordingConn) Read(b []byte) (n int, err error) {
	if n, err = r.Conn.Read(b); n == 0 {
		return
	}
	b = b[:n]

	r.lock.Lock()
	defer r.lock.Unlock()

	if l := len(r.flows); l == 0 || !r.currentlyReading {
		buf := make([]byte, len(b))
		copy(buf, b)
		r.flows = append(r.flows, buf)
	} else {
		r.flows[l-1] = append(r.flows[l-1], b[:n]...)
	}
	r.currentlyReading = true
	return
}

func (r *recordingConn) Write(b []byte) (n int, err error) {
	if n, err = r.Conn.Write(b); n == 0 {
		return
	}
	b = b[:n]

	r.lock.Lock()
	defer r.lock.Unlock()

	if l := len(r.flows); l == 0 || r.currentlyReading {
		buf := make([]byte, len(b))
		copy(buf, b)
		r.flows = append(r.flows, buf)
	} else {
		r.flows[l-1] = append(r.flows[l-1], b[:n]...)
	}
	r.currentlyReading = false
	return
}

// WriteTo writes Go source code to w that contains the recorded traffic.
func (r *recordingConn) WriteTo(w io.Writer) {
	fmt.Fprintf(w, "var changeMe = [][]byte {\n")
	for _, buf := range r.flows {
		fmt.Fprintf(w, "\t{")
		for i, b := range buf {
			if i%8 == 0 {
				fmt.Fprintf(w, "\n\t\t")
			}
			fmt.Fprintf(w, "0x%02x, ", b)
		}
		fmt.Fprintf(w, "\n\t},\n")
	}
	fmt.Fprintf(w, "}\n")
}

var serve = flag.Bool("serve", false, "run a TLS server on :10443")
var testCipherSuites = flag.String("ciphersuites",
	"0x"+strconv.FormatInt(int64(TLS_RSA_WITH_RC4_128_SHA), 16),
	"cipher suites to accept in serving mode")
var testMinVersion = flag.String("minversion",
	"0x"+strconv.FormatInt(int64(VersionSSL30), 16),
	"minimum version to negotiate")
var testMaxVersion = flag.String("maxversion",
	"0x"+strconv.FormatInt(int64(VersionTLS10), 16),
	"maximum version to negotiate")
var testClientAuth = flag.Int("clientauth", 0, "value for tls.Config.ClientAuth")

func GetTestConfig() *Config {
	var config = *testConfig

	minVersion, err := strconv.ParseUint(*testMinVersion, 0, 64)
	if err != nil {
		panic(err)
	}
	config.MinVersion = uint16(minVersion)
	maxVersion, err := strconv.ParseUint(*testMaxVersion, 0, 64)
	if err != nil {
		panic(err)
	}
	config.MaxVersion = uint16(maxVersion)

	suites := strings.Split(*testCipherSuites, ",")
	config.CipherSuites = make([]uint16, len(suites))
	for i := range suites {
		suite, err := strconv.ParseUint(suites[i], 0, 64)
		if err != nil {
			panic(err)
		}
		config.CipherSuites[i] = uint16(suite)
	}

	ecdsa := false
	for _, suite := range config.CipherSuites {
		switch suite {
		case TLS_ECDHE_ECDSA_WITH_RC4_128_SHA,
			TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA,
			TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA:
			ecdsa = true
		}
	}
	if ecdsa {
		config.Certificates = nil
		if !*connect {
			config.Certificates = make([]Certificate, 1)
			config.Certificates[0].Certificate = [][]byte{testECDSACertificate}
			config.Certificates[0].PrivateKey = testECDSAPrivateKey
		}
		config.BuildNameToCertificate()
	}

	config.ClientAuth = ClientAuthType(*testClientAuth)
	return &config
}

func TestRunServer(t *testing.T) {
	if !*serve {
		return
	}

	config := GetTestConfig()

	const addr = ":10443"
	l, err := net.Listen("tcp", addr)
	if err != nil {
		t.Fatal(err)
	}
	log.Printf("Now listening for connections on %s", addr)

	for {
		tcpConn, err := l.Accept()
		if err != nil {
			log.Printf("error accepting connection: %s", err)
			break
		}

		record := &recordingConn{
			Conn: tcpConn,
		}

		conn := Server(record, config)
		if err := conn.Handshake(); err != nil {
			log.Printf("error from TLS handshake: %s", err)
			break
		}

		_, err = conn.Write([]byte("hello, world\n"))
		if err != nil {
			log.Printf("error from Write: %s", err)
			continue
		}

		conn.Close()

		record.WriteTo(os.Stdout)
	}
}

func bigFromString(s string) *big.Int {
	ret := new(big.Int)
	ret.SetString(s, 10)
	return ret
}

func fromHex(s string) []byte {
	b, _ := hex.DecodeString(s)
	return b
}

var testRSACertificate = fromHex("308202b030820219a00302010202090085b0bba48a7fb8ca300d06092a864886f70d01010505003045310b3009060355040613024155311330110603550408130a536f6d652d53746174653121301f060355040a1318496e7465726e6574205769646769747320507479204c7464301e170d3130303432343039303933385a170d3131303432343039303933385a3045310b3009060355040613024155311330110603550408130a536f6d652d53746174653121301f060355040a1318496e7465726e6574205769646769747320507479204c746430819f300d06092a864886f70d010101050003818d0030818902818100bb79d6f517b5e5bf4610d0dc69bee62b07435ad0032d8a7a4385b71452e7a5654c2c78b8238cb5b482e5de1f953b7e62a52ca533d6fe125c7a56fcf506bffa587b263fb5cd04d3d0c921964ac7f4549f5abfef427100fe1899077f7e887d7df10439c4a22edb51c97ce3c04c3b326601cfafb11db8719a1ddbdb896baeda2d790203010001a381a73081a4301d0603551d0e04160414b1ade2855acfcb28db69ce2369ded3268e18883930750603551d23046e306c8014b1ade2855acfcb28db69ce2369ded3268e188839a149a4473045310b3009060355040613024155311330110603550408130a536f6d652d53746174653121301f060355040a1318496e7465726e6574205769646769747320507479204c746482090085b0bba48a7fb8ca300c0603551d13040530030101ff300d06092a864886f70d010105050003818100086c4524c76bb159ab0c52ccf2b014d7879d7a6475b55a9566e4c52b8eae12661feb4f38b36e60d392fdf74108b52513b1187a24fb301dbaed98b917ece7d73159db95d31d78ea50565cd5825a2d5a5f33c4b6d8c97590968c0f5298b5cd981f89205ff2a01ca31b9694dda9fd57e970e8266d71999b266e3850296c90a7bdd9")

var testECDSACertificate = fromHex("3082020030820162020900b8bf2d47a0d2ebf4300906072a8648ce3d04013045310b3009060355040613024155311330110603550408130a536f6d652d53746174653121301f060355040a1318496e7465726e6574205769646769747320507479204c7464301e170d3132313132323135303633325a170d3232313132303135303633325a3045310b3009060355040613024155311330110603550408130a536f6d652d53746174653121301f060355040a1318496e7465726e6574205769646769747320507479204c746430819b301006072a8648ce3d020106052b81040023038186000400c4a1edbe98f90b4873367ec316561122f23d53c33b4d213dcd6b75e6f6b0dc9adf26c1bcb287f072327cb3642f1c90bcea6823107efee325c0483a69e0286dd33700ef0462dd0da09c706283d881d36431aa9e9731bd96b068c09b23de76643f1a5c7fe9120e5858b65f70dd9bd8ead5d7f5d5ccb9b69f30665b669a20e227e5bffe3b300906072a8648ce3d040103818c0030818802420188a24febe245c5487d1bacf5ed989dae4770c05e1bb62fbdf1b64db76140d311a2ceee0b7e927eff769dc33b7ea53fcefa10e259ec472d7cacda4e970e15a06fd00242014dfcbe67139c2d050ebd3fa38c25c13313830d9406bbd4377af6ec7ac9862eddd711697f857c56defb31782be4c7780daecbbe9e4e3624317b6a0f399512078f2a")

var testSNICertificate = fromHex("308201f23082015da003020102020100300b06092a864886f70d01010530283110300e060355040a130741636d6520436f311430120603550403130b736e69746573742e636f6d301e170d3132303431313137343033355a170d3133303431313137343533355a30283110300e060355040a130741636d6520436f311430120603550403130b736e69746573742e636f6d30819d300b06092a864886f70d01010103818d0030818902818100bb79d6f517b5e5bf4610d0dc69bee62b07435ad0032d8a7a4385b71452e7a5654c2c78b8238cb5b482e5de1f953b7e62a52ca533d6fe125c7a56fcf506bffa587b263fb5cd04d3d0c921964ac7f4549f5abfef427100fe1899077f7e887d7df10439c4a22edb51c97ce3c04c3b326601cfafb11db8719a1ddbdb896baeda2d790203010001a3323030300e0603551d0f0101ff0404030200a0300d0603551d0e0406040401020304300f0603551d2304083006800401020304300b06092a864886f70d0101050381810089c6455f1c1f5ef8eb1ab174ee2439059f5c4259bb1a8d86cdb1d056f56a717da40e95ab90f59e8deaf627c157995094db0802266eb34fc6842dea8a4b68d9c1389103ab84fb9e1f85d9b5d23ff2312c8670fbb540148245a4ebafe264d90c8a4cf4f85b0fac12ac2fc4a3154bad52462868af96c62c6525d652b6e31845bdcc")

var testRSAPrivateKey = &rsa.PrivateKey{
	PublicKey: rsa.PublicKey{
		N: bigFromString("131650079503776001033793877885499001334664249354723305978524647182322416328664556247316495448366990052837680518067798333412266673813370895702118944398081598789828837447552603077848001020611640547221687072142537202428102790818451901395596882588063427854225330436740647715202971973145151161964464812406232198521"),
		E: 65537,
	},
	D: bigFromString("29354450337804273969007277378287027274721892607543397931919078829901848876371746653677097639302788129485893852488285045793268732234230875671682624082413996177431586734171663258657462237320300610850244186316880055243099640544518318093544057213190320837094958164973959123058337475052510833916491060913053867729"),
	Primes: []*big.Int{
		bigFromString("11969277782311800166562047708379380720136961987713178380670422671426759650127150688426177829077494755200794297055316163155755835813760102405344560929062149"),
		bigFromString("10998999429884441391899182616418192492905073053684657075974935218461686523870125521822756579792315215543092255516093840728890783887287417039645833477273829"),
	},
}

var testECDSAPrivateKey = &ecdsa.PrivateKey{
	PublicKey: ecdsa.PublicKey{
		Curve: &elliptic.CurveParams{
			P:       bigFromString("6864797660130609714981900799081393217269435300143305409394463459185543183397656052122559640661454554977296311391480858037121987999716643812574028291115057151"),
			N:       bigFromString("6864797660130609714981900799081393217269435300143305409394463459185543183397655394245057746333217197532963996371363321113864768612440380340372808892707005449"),
			B:       bigFromString("1093849038073734274511112390766805569936207598951683748994586394495953116150735016013708737573759623248592132296706313309438452531591012912142327488478985984"),
			Gx:      bigFromString("2661740802050217063228768716723360960729859168756973147706671368418802944996427808491545080627771902352094241225065558662157113545570916814161637315895999846"),
			Gy:      bigFromString("3757180025770020463545507224491183603594455134769762486694567779615544477440556316691234405012945539562144444537289428522585666729196580810124344277578376784"),
			BitSize: 521,
		},
		X: bigFromString("2636411247892461147287360222306590634450676461695221912739908880441342231985950069527906976759812296359387337367668045707086543273113073382714101597903639351"),
		Y: bigFromString("3204695818431246682253994090650952614555094516658732116404513121125038617915183037601737180082382202488628239201196033284060130040574800684774115478859677243"),
	},
	D: bigFromString("5477294338614160138026852784385529180817726002953041720191098180813046231640184669647735805135001309477695746518160084669446643325196003346204701381388769751"),
}

func loadPEMCert(in string) *x509.Certificate {
	block, _ := pem.Decode([]byte(in))
	if block.Type == "CERTIFICATE" && len(block.Headers) == 0 {
		cert, err := x509.ParseCertificate(block.Bytes)
		if err == nil {
			return cert
		}
		panic("error parsing cert")
	}
	panic("error parsing PEM")
}

// Script of interaction with gnutls implementation.
// The values for this test are obtained by building and running in server mode:
//   % go test -test.run "TestRunServer" -serve
// The recorded bytes are written to stdout.
var rsaRC4ServerScript = [][]byte{
	{
		0x16, 0x03, 0x01, 0x00, 0x54, 0x01, 0x00, 0x00,
		0x50, 0x03, 0x01, 0x50, 0x77, 0x3d, 0xbd, 0x32,
		0x13, 0xd7, 0xea, 0x33, 0x65, 0x02, 0xb8, 0x70,
		0xb7, 0x84, 0xc4, 0x05, 0x1f, 0xa4, 0x24, 0xc4,
		0x91, 0x69, 0x04, 0x32, 0x96, 0xfe, 0x5b, 0x49,
		0x71, 0x60, 0x9a, 0x00, 0x00, 0x28, 0x00, 0x39,
		0x00, 0x38, 0x00, 0x35, 0x00, 0x16, 0x00, 0x13,
		0x00, 0x0a, 0x00, 0x33, 0x00, 0x32, 0x00, 0x2f,
		0x00, 0x05, 0x00, 0x04, 0x00, 0x15, 0x00, 0x12,
		0x00, 0x09, 0x00, 0x14, 0x00, 0x11, 0x00, 0x08,
		0x00, 0x06, 0x00, 0x03, 0x00, 0xff, 0x02, 0x01,
		0x00,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x2a, 0x02, 0x00, 0x00,
		0x26, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x16,
		0x03, 0x01, 0x02, 0xbe, 0x0b, 0x00, 0x02, 0xba,
		0x00, 0x02, 0xb7, 0x00, 0x02, 0xb4, 0x30, 0x82,
		0x02, 0xb0, 0x30, 0x82, 0x02, 0x19, 0xa0, 0x03,
		0x02, 0x01, 0x02, 0x02, 0x09, 0x00, 0x85, 0xb0,
		0xbb, 0xa4, 0x8a, 0x7f, 0xb8, 0xca, 0x30, 0x0d,
		0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
		0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x45, 0x31,
		0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
		0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11,
		0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53,
		0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74,
		0x65, 0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55,
		0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65,
		0x72, 0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64,
		0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79,
		0x20, 0x4c, 0x74, 0x64, 0x30, 0x1e, 0x17, 0x0d,
		0x31, 0x30, 0x30, 0x34, 0x32, 0x34, 0x30, 0x39,
		0x30, 0x39, 0x33, 0x38, 0x5a, 0x17, 0x0d, 0x31,
		0x31, 0x30, 0x34, 0x32, 0x34, 0x30, 0x39, 0x30,
		0x39, 0x33, 0x38, 0x5a, 0x30, 0x45, 0x31, 0x0b,
		0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
		0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11, 0x06,
		0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f,
		0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x65,
		0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x04,
		0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65, 0x72,
		0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64, 0x67,
		0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20,
		0x4c, 0x74, 0x64, 0x30, 0x81, 0x9f, 0x30, 0x0d,
		0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
		0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8d,
		0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00,
		0xbb, 0x79, 0xd6, 0xf5, 0x17, 0xb5, 0xe5, 0xbf,
		0x46, 0x10, 0xd0, 0xdc, 0x69, 0xbe, 0xe6, 0x2b,
		0x07, 0x43, 0x5a, 0xd0, 0x03, 0x2d, 0x8a, 0x7a,
		0x43, 0x85, 0xb7, 0x14, 0x52, 0xe7, 0xa5, 0x65,
		0x4c, 0x2c, 0x78, 0xb8, 0x23, 0x8c, 0xb5, 0xb4,
		0x82, 0xe5, 0xde, 0x1f, 0x95, 0x3b, 0x7e, 0x62,
		0xa5, 0x2c, 0xa5, 0x33, 0xd6, 0xfe, 0x12, 0x5c,
		0x7a, 0x56, 0xfc, 0xf5, 0x06, 0xbf, 0xfa, 0x58,
		0x7b, 0x26, 0x3f, 0xb5, 0xcd, 0x04, 0xd3, 0xd0,
		0xc9, 0x21, 0x96, 0x4a, 0xc7, 0xf4, 0x54, 0x9f,
		0x5a, 0xbf, 0xef, 0x42, 0x71, 0x00, 0xfe, 0x18,
		0x99, 0x07, 0x7f, 0x7e, 0x88, 0x7d, 0x7d, 0xf1,
		0x04, 0x39, 0xc4, 0xa2, 0x2e, 0xdb, 0x51, 0xc9,
		0x7c, 0xe3, 0xc0, 0x4c, 0x3b, 0x32, 0x66, 0x01,
		0xcf, 0xaf, 0xb1, 0x1d, 0xb8, 0x71, 0x9a, 0x1d,
		0xdb, 0xdb, 0x89, 0x6b, 0xae, 0xda, 0x2d, 0x79,
		0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x81, 0xa7,
		0x30, 0x81, 0xa4, 0x30, 0x1d, 0x06, 0x03, 0x55,
		0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xb1, 0xad,
		0xe2, 0x85, 0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69,
		0xce, 0x23, 0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18,
		0x88, 0x39, 0x30, 0x75, 0x06, 0x03, 0x55, 0x1d,
		0x23, 0x04, 0x6e, 0x30, 0x6c, 0x80, 0x14, 0xb1,
		0xad, 0xe2, 0x85, 0x5a, 0xcf, 0xcb, 0x28, 0xdb,
		0x69, 0xce, 0x23, 0x69, 0xde, 0xd3, 0x26, 0x8e,
		0x18, 0x88, 0x39, 0xa1, 0x49, 0xa4, 0x47, 0x30,
		0x45, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55,
		0x04, 0x06, 0x13, 0x02, 0x41, 0x55, 0x31, 0x13,
		0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x13,
		0x0a, 0x53, 0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74,
		0x61, 0x74, 0x65, 0x31, 0x21, 0x30, 0x1f, 0x06,
		0x03, 0x55, 0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e,
		0x74, 0x65, 0x72, 0x6e, 0x65, 0x74, 0x20, 0x57,
		0x69, 0x64, 0x67, 0x69, 0x74, 0x73, 0x20, 0x50,
		0x74, 0x79, 0x20, 0x4c, 0x74, 0x64, 0x82, 0x09,
		0x00, 0x85, 0xb0, 0xbb, 0xa4, 0x8a, 0x7f, 0xb8,
		0xca, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13,
		0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xff, 0x30,
		0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
		0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x81,
		0x81, 0x00, 0x08, 0x6c, 0x45, 0x24, 0xc7, 0x6b,
		0xb1, 0x59, 0xab, 0x0c, 0x52, 0xcc, 0xf2, 0xb0,
		0x14, 0xd7, 0x87, 0x9d, 0x7a, 0x64, 0x75, 0xb5,
		0x5a, 0x95, 0x66, 0xe4, 0xc5, 0x2b, 0x8e, 0xae,
		0x12, 0x66, 0x1f, 0xeb, 0x4f, 0x38, 0xb3, 0x6e,
		0x60, 0xd3, 0x92, 0xfd, 0xf7, 0x41, 0x08, 0xb5,
		0x25, 0x13, 0xb1, 0x18, 0x7a, 0x24, 0xfb, 0x30,
		0x1d, 0xba, 0xed, 0x98, 0xb9, 0x17, 0xec, 0xe7,
		0xd7, 0x31, 0x59, 0xdb, 0x95, 0xd3, 0x1d, 0x78,
		0xea, 0x50, 0x56, 0x5c, 0xd5, 0x82, 0x5a, 0x2d,
		0x5a, 0x5f, 0x33, 0xc4, 0xb6, 0xd8, 0xc9, 0x75,
		0x90, 0x96, 0x8c, 0x0f, 0x52, 0x98, 0xb5, 0xcd,
		0x98, 0x1f, 0x89, 0x20, 0x5f, 0xf2, 0xa0, 0x1c,
		0xa3, 0x1b, 0x96, 0x94, 0xdd, 0xa9, 0xfd, 0x57,
		0xe9, 0x70, 0xe8, 0x26, 0x6d, 0x71, 0x99, 0x9b,
		0x26, 0x6e, 0x38, 0x50, 0x29, 0x6c, 0x90, 0xa7,
		0xbd, 0xd9, 0x16, 0x03, 0x01, 0x00, 0x04, 0x0e,
		0x00, 0x00, 0x00,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x86, 0x10, 0x00, 0x00,
		0x82, 0x00, 0x80, 0x2d, 0x09, 0x7c, 0x7f, 0xfc,
		0x84, 0xce, 0xb3, 0x30, 0x9b, 0xf9, 0xb7, 0xc8,
		0xc3, 0xff, 0xee, 0x6f, 0x20, 0x8a, 0xf4, 0xfb,
		0x86, 0x55, 0x1f, 0x6a, 0xb4, 0x81, 0x50, 0x3a,
		0x46, 0x1b, 0xd3, 0xca, 0x4b, 0x11, 0xff, 0xef,
		0x02, 0xbc, 0x18, 0xb8, 0x4a, 0x7d, 0x43, 0x23,
		0x96, 0x92, 0x27, 0x7c, 0xca, 0xcf, 0xe6, 0x91,
		0xe8, 0x14, 0x97, 0x68, 0xb4, 0xe5, 0xc0, 0xc9,
		0x23, 0xdd, 0x54, 0x07, 0xa6, 0x2e, 0x8c, 0x98,
		0xfc, 0xc6, 0x8c, 0x04, 0x6b, 0x1b, 0x5f, 0xd5,
		0x3d, 0x8b, 0x6c, 0x55, 0x4f, 0x7a, 0xe6, 0x6c,
		0x74, 0x2c, 0x1e, 0x34, 0xdb, 0xfb, 0x00, 0xb1,
		0x4e, 0x10, 0x21, 0x16, 0xe0, 0x3e, 0xc5, 0x64,
		0x84, 0x28, 0x2b, 0x2b, 0x29, 0x47, 0x51, 0x34,
		0x76, 0x15, 0x20, 0x71, 0x0b, 0x30, 0xa1, 0x85,
		0xd5, 0x15, 0x18, 0x14, 0x64, 0x4b, 0x40, 0x7c,
		0x4f, 0xb3, 0x7b, 0x14, 0x03, 0x01, 0x00, 0x01,
		0x01, 0x16, 0x03, 0x01, 0x00, 0x24, 0xab, 0xee,
		0xf5, 0x97, 0x5f, 0xc6, 0x78, 0xf3, 0xc6, 0x83,
		0x5b, 0x55, 0x4f, 0xcb, 0x45, 0x3f, 0xfa, 0xf7,
		0x05, 0x02, 0xc2, 0x63, 0x87, 0x18, 0xb5, 0x9a,
		0x62, 0xe2, 0x3f, 0x88, 0x5a, 0x60, 0x61, 0x72,
		0xfa, 0x9c,
	},
	{
		0x14, 0x03, 0x01, 0x00, 0x01, 0x01, 0x16, 0x03,
		0x01, 0x00, 0x24, 0x72, 0xa4, 0xe4, 0xaa, 0xd2,
		0xc4, 0x39, 0x7e, 0x2a, 0xc1, 0x6f, 0x34, 0x42,
		0x28, 0xcb, 0x9d, 0x7a, 0x09, 0xca, 0x96, 0xad,
		0x0e, 0x11, 0x51, 0x8a, 0x06, 0xb0, 0xe9, 0xca,
		0xeb, 0xce, 0xe2, 0xd5, 0x2e, 0xc1, 0x8d, 0x17,
		0x03, 0x01, 0x00, 0x21, 0x2e, 0x61, 0x86, 0x17,
		0xdb, 0xa6, 0x30, 0xe2, 0x62, 0x06, 0x2a, 0x8b,
		0x75, 0x2c, 0x2d, 0xcf, 0xf5, 0x01, 0x11, 0x52,
		0x81, 0x38, 0xcf, 0xd5, 0xf7, 0xdc, 0x52, 0x31,
		0x1f, 0x97, 0x43, 0xc2, 0x71, 0x15, 0x03, 0x01,
		0x00, 0x16, 0xe0, 0x21, 0xfe, 0x36, 0x2e, 0x68,
		0x2c, 0xf1, 0xbe, 0x04, 0xec, 0xd4, 0xc6, 0xdd,
		0xac, 0x6f, 0x4c, 0x85, 0x32, 0x3f, 0x87, 0x1b,
	},
}

var rsaDES3ServerScript = [][]byte{
	{
		0x16, 0x03, 0x00, 0x00, 0xc5, 0x01, 0x00, 0x00,
		0xc1, 0x03, 0x03, 0x50, 0xae, 0x5d, 0x38, 0xec,
		0xaa, 0x2f, 0x41, 0xf9, 0xd2, 0x7b, 0xa1, 0xfd,
		0x0f, 0xff, 0x4e, 0x54, 0x0e, 0x15, 0x57, 0xaf,
		0x2c, 0x91, 0xb5, 0x35, 0x5b, 0x2e, 0xb0, 0xec,
		0x20, 0xe5, 0xd2, 0x00, 0x00, 0x50, 0xc0, 0x09,
		0xc0, 0x23, 0xc0, 0x2b, 0xc0, 0x0a, 0xc0, 0x24,
		0xc0, 0x2c, 0xc0, 0x08, 0xc0, 0x13, 0xc0, 0x27,
		0xc0, 0x2f, 0xc0, 0x14, 0xc0, 0x30, 0xc0, 0x12,
		0x00, 0x33, 0x00, 0x67, 0x00, 0x45, 0x00, 0x9e,
		0x00, 0x39, 0x00, 0x6b, 0x00, 0x88, 0x00, 0x16,
		0x00, 0x32, 0x00, 0x40, 0x00, 0x44, 0x00, 0xa2,
		0x00, 0x38, 0x00, 0x6a, 0x00, 0x87, 0x00, 0x13,
		0x00, 0x66, 0x00, 0x2f, 0x00, 0x3c, 0x00, 0x41,
		0x00, 0x9c, 0x00, 0x35, 0x00, 0x3d, 0x00, 0x84,
		0x00, 0x0a, 0x00, 0x05, 0x00, 0x04, 0x01, 0x00,
		0x00, 0x48, 0x00, 0x05, 0x00, 0x05, 0x01, 0x00,
		0x00, 0x00, 0x00, 0xff, 0x01, 0x00, 0x01, 0x00,
		0x00, 0x23, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x0c,
		0x00, 0x0a, 0x00, 0x13, 0x00, 0x15, 0x00, 0x17,
		0x00, 0x18, 0x00, 0x19, 0x00, 0x0b, 0x00, 0x02,
		0x01, 0x00, 0x00, 0x0d, 0x00, 0x1c, 0x00, 0x1a,
		0x04, 0x01, 0x04, 0x02, 0x04, 0x03, 0x05, 0x01,
		0x05, 0x03, 0x06, 0x01, 0x06, 0x03, 0x03, 0x01,
		0x03, 0x02, 0x03, 0x03, 0x02, 0x01, 0x02, 0x02,
		0x02, 0x03,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x30, 0x02, 0x00, 0x00,
		0x2c, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00,
		0x04, 0x00, 0x23, 0x00, 0x00, 0x16, 0x03, 0x01,
		0x02, 0xbe, 0x0b, 0x00, 0x02, 0xba, 0x00, 0x02,
		0xb7, 0x00, 0x02, 0xb4, 0x30, 0x82, 0x02, 0xb0,
		0x30, 0x82, 0x02, 0x19, 0xa0, 0x03, 0x02, 0x01,
		0x02, 0x02, 0x09, 0x00, 0x85, 0xb0, 0xbb, 0xa4,
		0x8a, 0x7f, 0xb8, 0xca, 0x30, 0x0d, 0x06, 0x09,
		0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
		0x05, 0x05, 0x00, 0x30, 0x45, 0x31, 0x0b, 0x30,
		0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
		0x41, 0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03,
		0x55, 0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d,
		0x65, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31,
		0x21, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a,
		0x13, 0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e,
		0x65, 0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69,
		0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c,
		0x74, 0x64, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x30,
		0x30, 0x34, 0x32, 0x34, 0x30, 0x39, 0x30, 0x39,
		0x33, 0x38, 0x5a, 0x17, 0x0d, 0x31, 0x31, 0x30,
		0x34, 0x32, 0x34, 0x30, 0x39, 0x30, 0x39, 0x33,
		0x38, 0x5a, 0x30, 0x45, 0x31, 0x0b, 0x30, 0x09,
		0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x41,
		0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55,
		0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d, 0x65,
		0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x21,
		0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13,
		0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65,
		0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69, 0x74,
		0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c, 0x74,
		0x64, 0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09,
		0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
		0x01, 0x05, 0x00, 0x03, 0x81, 0x8d, 0x00, 0x30,
		0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xbb, 0x79,
		0xd6, 0xf5, 0x17, 0xb5, 0xe5, 0xbf, 0x46, 0x10,
		0xd0, 0xdc, 0x69, 0xbe, 0xe6, 0x2b, 0x07, 0x43,
		0x5a, 0xd0, 0x03, 0x2d, 0x8a, 0x7a, 0x43, 0x85,
		0xb7, 0x14, 0x52, 0xe7, 0xa5, 0x65, 0x4c, 0x2c,
		0x78, 0xb8, 0x23, 0x8c, 0xb5, 0xb4, 0x82, 0xe5,
		0xde, 0x1f, 0x95, 0x3b, 0x7e, 0x62, 0xa5, 0x2c,
		0xa5, 0x33, 0xd6, 0xfe, 0x12, 0x5c, 0x7a, 0x56,
		0xfc, 0xf5, 0x06, 0xbf, 0xfa, 0x58, 0x7b, 0x26,
		0x3f, 0xb5, 0xcd, 0x04, 0xd3, 0xd0, 0xc9, 0x21,
		0x96, 0x4a, 0xc7, 0xf4, 0x54, 0x9f, 0x5a, 0xbf,
		0xef, 0x42, 0x71, 0x00, 0xfe, 0x18, 0x99, 0x07,
		0x7f, 0x7e, 0x88, 0x7d, 0x7d, 0xf1, 0x04, 0x39,
		0xc4, 0xa2, 0x2e, 0xdb, 0x51, 0xc9, 0x7c, 0xe3,
		0xc0, 0x4c, 0x3b, 0x32, 0x66, 0x01, 0xcf, 0xaf,
		0xb1, 0x1d, 0xb8, 0x71, 0x9a, 0x1d, 0xdb, 0xdb,
		0x89, 0x6b, 0xae, 0xda, 0x2d, 0x79, 0x02, 0x03,
		0x01, 0x00, 0x01, 0xa3, 0x81, 0xa7, 0x30, 0x81,
		0xa4, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e,
		0x04, 0x16, 0x04, 0x14, 0xb1, 0xad, 0xe2, 0x85,
		0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69, 0xce, 0x23,
		0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18, 0x88, 0x39,
		0x30, 0x75, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04,
		0x6e, 0x30, 0x6c, 0x80, 0x14, 0xb1, 0xad, 0xe2,
		0x85, 0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69, 0xce,
		0x23, 0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18, 0x88,
		0x39, 0xa1, 0x49, 0xa4, 0x47, 0x30, 0x45, 0x31,
		0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
		0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11,
		0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53,
		0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74,
		0x65, 0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55,
		0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65,
		0x72, 0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64,
		0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79,
		0x20, 0x4c, 0x74, 0x64, 0x82, 0x09, 0x00, 0x85,
		0xb0, 0xbb, 0xa4, 0x8a, 0x7f, 0xb8, 0xca, 0x30,
		0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x05,
		0x30, 0x03, 0x01, 0x01, 0xff, 0x30, 0x0d, 0x06,
		0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
		0x01, 0x05, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00,
		0x08, 0x6c, 0x45, 0x24, 0xc7, 0x6b, 0xb1, 0x59,
		0xab, 0x0c, 0x52, 0xcc, 0xf2, 0xb0, 0x14, 0xd7,
		0x87, 0x9d, 0x7a, 0x64, 0x75, 0xb5, 0x5a, 0x95,
		0x66, 0xe4, 0xc5, 0x2b, 0x8e, 0xae, 0x12, 0x66,
		0x1f, 0xeb, 0x4f, 0x38, 0xb3, 0x6e, 0x60, 0xd3,
		0x92, 0xfd, 0xf7, 0x41, 0x08, 0xb5, 0x25, 0x13,
		0xb1, 0x18, 0x7a, 0x24, 0xfb, 0x30, 0x1d, 0xba,
		0xed, 0x98, 0xb9, 0x17, 0xec, 0xe7, 0xd7, 0x31,
		0x59, 0xdb, 0x95, 0xd3, 0x1d, 0x78, 0xea, 0x50,
		0x56, 0x5c, 0xd5, 0x82, 0x5a, 0x2d, 0x5a, 0x5f,
		0x33, 0xc4, 0xb6, 0xd8, 0xc9, 0x75, 0x90, 0x96,
		0x8c, 0x0f, 0x52, 0x98, 0xb5, 0xcd, 0x98, 0x1f,
		0x89, 0x20, 0x5f, 0xf2, 0xa0, 0x1c, 0xa3, 0x1b,
		0x96, 0x94, 0xdd, 0xa9, 0xfd, 0x57, 0xe9, 0x70,
		0xe8, 0x26, 0x6d, 0x71, 0x99, 0x9b, 0x26, 0x6e,
		0x38, 0x50, 0x29, 0x6c, 0x90, 0xa7, 0xbd, 0xd9,
		0x16, 0x03, 0x01, 0x00, 0x04, 0x0e, 0x00, 0x00,
		0x00,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x86, 0x10, 0x00, 0x00,
		0x82, 0x00, 0x80, 0x51, 0x04, 0xf1, 0x7a, 0xbf,
		0xe8, 0xa5, 0x86, 0x09, 0xa7, 0xf3, 0xcc, 0x93,
		0x00, 0x10, 0x5b, 0xb8, 0xc1, 0x51, 0x0d, 0x5b,
		0xcd, 0xed, 0x26, 0x01, 0x69, 0x73, 0xf4, 0x05,
		0x8a, 0x6a, 0xc3, 0xb1, 0x9e, 0x84, 0x4e, 0x39,
		0xcf, 0x5e, 0x55, 0xa9, 0x70, 0x19, 0x96, 0x91,
		0xcd, 0x2c, 0x78, 0x3c, 0xa2, 0x6d, 0xb0, 0x49,
		0x86, 0xf6, 0xd1, 0x3a, 0xde, 0x00, 0x4b, 0xa6,
		0x25, 0xbf, 0x85, 0x39, 0xce, 0xb1, 0xcf, 0xbc,
		0x16, 0xc7, 0x66, 0xac, 0xf8, 0xd2, 0x3b, 0xd1,
		0xcc, 0x16, 0xac, 0x63, 0x3c, 0xbe, 0xd9, 0xb6,
		0x6a, 0xe4, 0x13, 0x8a, 0xf4, 0x56, 0x2f, 0x92,
		0x54, 0xd8, 0xf0, 0x84, 0x01, 0x32, 0x1a, 0xa9,
		0x2d, 0xaf, 0x82, 0x0e, 0x00, 0xfa, 0x07, 0x88,
		0xd9, 0x87, 0xe7, 0xdc, 0x9e, 0xe9, 0x72, 0x49,
		0xb8, 0xfa, 0x8c, 0x7b, 0x07, 0x0b, 0x03, 0x7c,
		0x10, 0x8c, 0x8a, 0x14, 0x03, 0x01, 0x00, 0x01,
		0x01, 0x16, 0x03, 0x01, 0x00, 0xa8, 0x61, 0xa4,
		0xf4, 0x5f, 0x8a, 0x1f, 0x5c, 0x92, 0x3f, 0x8c,
		0xdb, 0xd6, 0x10, 0xcd, 0x9e, 0xe7, 0xf0, 0xc4,
		0x3c, 0xb6, 0x1c, 0x9a, 0x56, 0x73, 0x7f, 0xa6,
		0x14, 0x24, 0xcb, 0x96, 0x1f, 0xe0, 0xaf, 0xcd,
		0x3c, 0x66, 0x43, 0xb7, 0x37, 0x65, 0x34, 0x47,
		0xf8, 0x43, 0xf1, 0xcc, 0x15, 0xb8, 0xdc, 0x35,
		0xe0, 0xa4, 0x2d, 0x78, 0x94, 0xe0, 0x02, 0xf3,
		0x76, 0x46, 0xf7, 0x9b, 0x8d, 0x0d, 0x5d, 0x0b,
		0xd3, 0xdd, 0x9a, 0x9e, 0x62, 0x2e, 0xc5, 0x98,
		0x75, 0x63, 0x0c, 0xbf, 0x8e, 0x49, 0x33, 0x23,
		0x7c, 0x00, 0xcf, 0xfb, 0xcf, 0xba, 0x0f, 0x41,
		0x39, 0x89, 0xb9, 0xcc, 0x59, 0xd0, 0x2b, 0xb6,
		0xec, 0x04, 0xe2, 0xc0, 0x52, 0xc7, 0xcf, 0x71,
		0x47, 0xff, 0x70, 0x7e, 0xa9, 0xbd, 0x1c, 0xdd,
		0x17, 0xa5, 0x6c, 0xb7, 0x10, 0x4f, 0x42, 0x18,
		0x37, 0x69, 0xa9, 0xd2, 0xb3, 0x18, 0x84, 0x92,
		0xa7, 0x47, 0x21, 0xf6, 0x95, 0x63, 0x29, 0xd6,
		0xa5, 0xb6, 0xda, 0x65, 0x67, 0x69, 0xc4, 0x26,
		0xac, 0x8b, 0x08, 0x58, 0xdd, 0x3c, 0x31, 0x20,
		0xd5, 0x0c, 0x88, 0x72, 0x18, 0x16, 0x88, 0x1e,
		0x4a, 0x0f, 0xe1, 0xcf, 0x95, 0x24,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x72, 0x04, 0x00, 0x00,
		0x6e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x65,
		0xe8, 0x4b, 0xde, 0xef, 0xba, 0x3e, 0x18, 0x1c,
		0x1e, 0x5e, 0xbc, 0x87, 0xf1, 0x87, 0x8d, 0x72,
		0xe3, 0xbe, 0x0f, 0xdf, 0xfd, 0xd0, 0xb2, 0x89,
		0xf8, 0x05, 0x9a, 0x52, 0x47, 0x77, 0x9e, 0xe8,
		0xb1, 0x1d, 0x18, 0xed, 0x6a, 0x4b, 0x63, 0x1d,
		0xf1, 0x62, 0xd2, 0x65, 0x21, 0x26, 0x73, 0xd4,
		0x35, 0x5b, 0x95, 0x89, 0x12, 0x59, 0x23, 0x8c,
		0xc3, 0xfc, 0xf9, 0x4d, 0x21, 0x79, 0xa0, 0xbd,
		0xff, 0x33, 0xa2, 0x3d, 0x0b, 0x6f, 0x89, 0xc9,
		0x23, 0xe4, 0xe7, 0x9f, 0x1d, 0x98, 0xf6, 0xed,
		0x02, 0x8d, 0xac, 0x1a, 0xf9, 0xcb, 0xa5, 0x14,
		0x03, 0x01, 0x00, 0x01, 0x01, 0x16, 0x03, 0x01,
		0x00, 0x28, 0x91, 0x56, 0x80, 0xe2, 0x6d, 0x51,
		0x88, 0x03, 0xf8, 0x49, 0xe6, 0x6a, 0x5a, 0xfb,
		0x2f, 0x0b, 0xb5, 0xa1, 0x0d, 0x63, 0x83, 0xae,
		0xb9, 0xbc, 0x05, 0xf0, 0x81, 0x00, 0x61, 0x83,
		0x38, 0xda, 0x14, 0xf6, 0xea, 0xd8, 0x78, 0x65,
		0xc7, 0x26, 0x17, 0x03, 0x01, 0x00, 0x18, 0x81,
		0x30, 0x8b, 0x22, 0x5a, 0xd3, 0x7f, 0xc8, 0xf2,
		0x8a, 0x6b, 0xa3, 0xba, 0x4d, 0xe7, 0x6e, 0xd2,
		0xfd, 0xbf, 0xf2, 0xc5, 0x28, 0xa0, 0x62, 0x17,
		0x03, 0x01, 0x00, 0x28, 0x17, 0x83, 0x3c, 0x78,
		0x18, 0xfa, 0x8d, 0x58, 0x5c, 0xaa, 0x05, 0x7d,
		0x67, 0x96, 0x11, 0x60, 0x11, 0xc0, 0x1e, 0x0d,
		0x6a, 0x6e, 0x5f, 0x1d, 0x98, 0x4b, 0xff, 0x82,
		0xee, 0x21, 0x06, 0x29, 0xd3, 0x8b, 0x80, 0x78,
		0x39, 0x05, 0x34, 0x9b, 0x15, 0x03, 0x01, 0x00,
		0x18, 0xa9, 0x38, 0x18, 0x4f, 0x9d, 0x84, 0x75,
		0x88, 0x53, 0xd6, 0x85, 0xc2, 0x15, 0x4b, 0xe3,
		0xe3, 0x35, 0x9a, 0x74, 0xc9, 0x3e, 0x13, 0xc1,
		0x8c,
	},
}

var rsaAESServerScript = [][]byte{
	{
		0x16, 0x03, 0x00, 0x00, 0xc5, 0x01, 0x00, 0x00,
		0xc1, 0x03, 0x03, 0x50, 0xae, 0x5c, 0xe9, 0x5e,
		0x31, 0x93, 0x82, 0xa5, 0x6f, 0x51, 0x82, 0xc8,
		0x55, 0x4f, 0x1f, 0x2e, 0x90, 0x98, 0x81, 0x13,
		0x27, 0x80, 0x68, 0xb4, 0x2d, 0xba, 0x3a, 0x76,
		0xd8, 0xd7, 0x2c, 0x00, 0x00, 0x50, 0xc0, 0x09,
		0xc0, 0x23, 0xc0, 0x2b, 0xc0, 0x0a, 0xc0, 0x24,
		0xc0, 0x2c, 0xc0, 0x08, 0xc0, 0x13, 0xc0, 0x27,
		0xc0, 0x2f, 0xc0, 0x14, 0xc0, 0x30, 0xc0, 0x12,
		0x00, 0x33, 0x00, 0x67, 0x00, 0x45, 0x00, 0x9e,
		0x00, 0x39, 0x00, 0x6b, 0x00, 0x88, 0x00, 0x16,
		0x00, 0x32, 0x00, 0x40, 0x00, 0x44, 0x00, 0xa2,
		0x00, 0x38, 0x00, 0x6a, 0x00, 0x87, 0x00, 0x13,
		0x00, 0x66, 0x00, 0x2f, 0x00, 0x3c, 0x00, 0x41,
		0x00, 0x9c, 0x00, 0x35, 0x00, 0x3d, 0x00, 0x84,
		0x00, 0x0a, 0x00, 0x05, 0x00, 0x04, 0x01, 0x00,
		0x00, 0x48, 0x00, 0x05, 0x00, 0x05, 0x01, 0x00,
		0x00, 0x00, 0x00, 0xff, 0x01, 0x00, 0x01, 0x00,
		0x00, 0x23, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x0c,
		0x00, 0x0a, 0x00, 0x13, 0x00, 0x15, 0x00, 0x17,
		0x00, 0x18, 0x00, 0x19, 0x00, 0x0b, 0x00, 0x02,
		0x01, 0x00, 0x00, 0x0d, 0x00, 0x1c, 0x00, 0x1a,
		0x04, 0x01, 0x04, 0x02, 0x04, 0x03, 0x05, 0x01,
		0x05, 0x03, 0x06, 0x01, 0x06, 0x03, 0x03, 0x01,
		0x03, 0x02, 0x03, 0x03, 0x02, 0x01, 0x02, 0x02,
		0x02, 0x03,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x30, 0x02, 0x00, 0x00,
		0x2c, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x2f, 0x00, 0x00,
		0x04, 0x00, 0x23, 0x00, 0x00, 0x16, 0x03, 0x01,
		0x02, 0xbe, 0x0b, 0x00, 0x02, 0xba, 0x00, 0x02,
		0xb7, 0x00, 0x02, 0xb4, 0x30, 0x82, 0x02, 0xb0,
		0x30, 0x82, 0x02, 0x19, 0xa0, 0x03, 0x02, 0x01,
		0x02, 0x02, 0x09, 0x00, 0x85, 0xb0, 0xbb, 0xa4,
		0x8a, 0x7f, 0xb8, 0xca, 0x30, 0x0d, 0x06, 0x09,
		0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
		0x05, 0x05, 0x00, 0x30, 0x45, 0x31, 0x0b, 0x30,
		0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
		0x41, 0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03,
		0x55, 0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d,
		0x65, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31,
		0x21, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a,
		0x13, 0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e,
		0x65, 0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69,
		0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c,
		0x74, 0x64, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x30,
		0x30, 0x34, 0x32, 0x34, 0x30, 0x39, 0x30, 0x39,
		0x33, 0x38, 0x5a, 0x17, 0x0d, 0x31, 0x31, 0x30,
		0x34, 0x32, 0x34, 0x30, 0x39, 0x30, 0x39, 0x33,
		0x38, 0x5a, 0x30, 0x45, 0x31, 0x0b, 0x30, 0x09,
		0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x41,
		0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55,
		0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d, 0x65,
		0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x21,
		0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13,
		0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65,
		0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69, 0x74,
		0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c, 0x74,
		0x64, 0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09,
		0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
		0x01, 0x05, 0x00, 0x03, 0x81, 0x8d, 0x00, 0x30,
		0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xbb, 0x79,
		0xd6, 0xf5, 0x17, 0xb5, 0xe5, 0xbf, 0x46, 0x10,
		0xd0, 0xdc, 0x69, 0xbe, 0xe6, 0x2b, 0x07, 0x43,
		0x5a, 0xd0, 0x03, 0x2d, 0x8a, 0x7a, 0x43, 0x85,
		0xb7, 0x14, 0x52, 0xe7, 0xa5, 0x65, 0x4c, 0x2c,
		0x78, 0xb8, 0x23, 0x8c, 0xb5, 0xb4, 0x82, 0xe5,
		0xde, 0x1f, 0x95, 0x3b, 0x7e, 0x62, 0xa5, 0x2c,
		0xa5, 0x33, 0xd6, 0xfe, 0x12, 0x5c, 0x7a, 0x56,
		0xfc, 0xf5, 0x06, 0xbf, 0xfa, 0x58, 0x7b, 0x26,
		0x3f, 0xb5, 0xcd, 0x04, 0xd3, 0xd0, 0xc9, 0x21,
		0x96, 0x4a, 0xc7, 0xf4, 0x54, 0x9f, 0x5a, 0xbf,
		0xef, 0x42, 0x71, 0x00, 0xfe, 0x18, 0x99, 0x07,
		0x7f, 0x7e, 0x88, 0x7d, 0x7d, 0xf1, 0x04, 0x39,
		0xc4, 0xa2, 0x2e, 0xdb, 0x51, 0xc9, 0x7c, 0xe3,
		0xc0, 0x4c, 0x3b, 0x32, 0x66, 0x01, 0xcf, 0xaf,
		0xb1, 0x1d, 0xb8, 0x71, 0x9a, 0x1d, 0xdb, 0xdb,
		0x89, 0x6b, 0xae, 0xda, 0x2d, 0x79, 0x02, 0x03,
		0x01, 0x00, 0x01, 0xa3, 0x81, 0xa7, 0x30, 0x81,
		0xa4, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e,
		0x04, 0x16, 0x04, 0x14, 0xb1, 0xad, 0xe2, 0x85,
		0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69, 0xce, 0x23,
		0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18, 0x88, 0x39,
		0x30, 0x75, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04,
		0x6e, 0x30, 0x6c, 0x80, 0x14, 0xb1, 0xad, 0xe2,
		0x85, 0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69, 0xce,
		0x23, 0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18, 0x88,
		0x39, 0xa1, 0x49, 0xa4, 0x47, 0x30, 0x45, 0x31,
		0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
		0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11,
		0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53,
		0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74,
		0x65, 0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55,
		0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65,
		0x72, 0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64,
		0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79,
		0x20, 0x4c, 0x74, 0x64, 0x82, 0x09, 0x00, 0x85,
		0xb0, 0xbb, 0xa4, 0x8a, 0x7f, 0xb8, 0xca, 0x30,
		0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x05,
		0x30, 0x03, 0x01, 0x01, 0xff, 0x30, 0x0d, 0x06,
		0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
		0x01, 0x05, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00,
		0x08, 0x6c, 0x45, 0x24, 0xc7, 0x6b, 0xb1, 0x59,
		0xab, 0x0c, 0x52, 0xcc, 0xf2, 0xb0, 0x14, 0xd7,
		0x87, 0x9d, 0x7a, 0x64, 0x75, 0xb5, 0x5a, 0x95,
		0x66, 0xe4, 0xc5, 0x2b, 0x8e, 0xae, 0x12, 0x66,
		0x1f, 0xeb, 0x4f, 0x38, 0xb3, 0x6e, 0x60, 0xd3,
		0x92, 0xfd, 0xf7, 0x41, 0x08, 0xb5, 0x25, 0x13,
		0xb1, 0x18, 0x7a, 0x24, 0xfb, 0x30, 0x1d, 0xba,
		0xed, 0x98, 0xb9, 0x17, 0xec, 0xe7, 0xd7, 0x31,
		0x59, 0xdb, 0x95, 0xd3, 0x1d, 0x78, 0xea, 0x50,
		0x56, 0x5c, 0xd5, 0x82, 0x5a, 0x2d, 0x5a, 0x5f,
		0x33, 0xc4, 0xb6, 0xd8, 0xc9, 0x75, 0x90, 0x96,
		0x8c, 0x0f, 0x52, 0x98, 0xb5, 0xcd, 0x98, 0x1f,
		0x89, 0x20, 0x5f, 0xf2, 0xa0, 0x1c, 0xa3, 0x1b,
		0x96, 0x94, 0xdd, 0xa9, 0xfd, 0x57, 0xe9, 0x70,
		0xe8, 0x26, 0x6d, 0x71, 0x99, 0x9b, 0x26, 0x6e,
		0x38, 0x50, 0x29, 0x6c, 0x90, 0xa7, 0xbd, 0xd9,
		0x16, 0x03, 0x01, 0x00, 0x04, 0x0e, 0x00, 0x00,
		0x00,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x86, 0x10, 0x00, 0x00,
		0x82, 0x00, 0x80, 0x51, 0x2e, 0xec, 0x0d, 0x86,
		0xf3, 0x9f, 0xf2, 0x77, 0x04, 0x27, 0x2b, 0x0e,
		0x9c, 0xab, 0x35, 0x84, 0x65, 0xff, 0x36, 0xef,
		0xc0, 0x08, 0xc9, 0x1d, 0x9f, 0x29, 0xae, 0x8d,
		0xc5, 0x66, 0x81, 0x31, 0x92, 0x5e, 0x3d, 0xac,
		0xaa, 0x37, 0x28, 0x2c, 0x06, 0x91, 0xa6, 0xc2,
		0xd0, 0x83, 0x34, 0x24, 0x1c, 0x88, 0xfc, 0x0a,
		0xcf, 0xbf, 0xc2, 0x94, 0xe2, 0xed, 0xa7, 0x6a,
		0xa8, 0x8d, 0x3d, 0xf7, 0x06, 0x7d, 0x69, 0xf8,
		0x0d, 0xb2, 0xf7, 0xe4, 0x45, 0xcb, 0x0a, 0x25,
		0xcb, 0xb2, 0x2e, 0x38, 0x9a, 0x84, 0x75, 0xe8,
		0xe1, 0x42, 0x39, 0xa2, 0x18, 0x0e, 0x48, 0xca,
		0x33, 0x16, 0x4e, 0xf6, 0x2f, 0xec, 0x07, 0xe7,
		0x57, 0xe1, 0x20, 0x40, 0x40, 0x6d, 0x4e, 0x29,
		0x04, 0x1a, 0x8c, 0x99, 0xfb, 0x19, 0x3c, 0xaa,
		0x75, 0x64, 0xd3, 0xa6, 0xe6, 0xed, 0x3f, 0x5a,
		0xd2, 0xc9, 0x80, 0x14, 0x03, 0x01, 0x00, 0x01,
		0x01, 0x16, 0x03, 0x01, 0x01, 0x10, 0xe9, 0x9e,
		0x06, 0x92, 0x18, 0xbf, 0x5e, 0xaf, 0x33, 0xc1,
		0xbf, 0x0e, 0x12, 0x07, 0x48, 0x4f, 0x6b, 0x6c,
		0xf5, 0x23, 0x5e, 0x87, 0xa7, 0xd3, 0x50, 0x79,
		0x38, 0xdc, 0xe0, 0x49, 0xd3, 0x81, 0x21, 0x12,
		0xd0, 0x3d, 0x9a, 0xfb, 0x83, 0xc1, 0x8b, 0xfc,
		0x14, 0xd5, 0xd5, 0xa7, 0xa3, 0x34, 0x14, 0x71,
		0xbe, 0xea, 0x37, 0x18, 0x12, 0x7f, 0x41, 0xfb,
		0xc5, 0x51, 0x17, 0x9d, 0x96, 0x58, 0x14, 0xfb,
		0x4f, 0xd7, 0xd3, 0x15, 0x0f, 0xec, 0x5a, 0x0d,
		0x35, 0xbb, 0x3c, 0x81, 0x5b, 0x3f, 0xdf, 0x52,
		0xa4, 0x4c, 0xcd, 0x13, 0xe1, 0x10, 0x37, 0x34,
		0xbf, 0xb4, 0x80, 0x1e, 0x8d, 0xe2, 0xc3, 0x7a,
		0x0f, 0x7b, 0x7d, 0x23, 0xeb, 0xd0, 0x99, 0x69,
		0xad, 0x0a, 0x2d, 0xb3, 0x6c, 0xd6, 0x80, 0x11,
		0x7f, 0x6c, 0xed, 0x1b, 0xcd, 0x08, 0x22, 0x56,
		0x90, 0x0e, 0xa4, 0xc3, 0x29, 0x33, 0x96, 0x30,
		0x34, 0x94, 0xa1, 0xeb, 0x9c, 0x1b, 0x5a, 0xd1,
		0x03, 0x61, 0xf9, 0xdd, 0xf3, 0x64, 0x8a, 0xfd,
		0x5f, 0x44, 0xdb, 0x2e, 0xa7, 0xfd, 0xe1, 0x1a,
		0x66, 0xc5, 0x01, 0x9c, 0xc7, 0xd1, 0xc4, 0xd3,
		0xea, 0x14, 0x3c, 0xed, 0x74, 0xbb, 0x1b, 0x97,
		0x8f, 0xf1, 0x29, 0x39, 0x33, 0x92, 0x93, 0x4e,
		0xf5, 0x87, 0x91, 0x61, 0x65, 0x8d, 0x27, 0x8d,
		0x76, 0xc1, 0xfa, 0x6a, 0x99, 0x80, 0xb1, 0x9b,
		0x29, 0x53, 0xce, 0x3e, 0xb6, 0x9a, 0xce, 0x3c,
		0x19, 0x5e, 0x48, 0x83, 0xaa, 0xa7, 0x66, 0x98,
		0x59, 0xf4, 0xbb, 0xf2, 0xbc, 0xd9, 0xc5, 0x9a,
		0xc8, 0x2c, 0x63, 0x58, 0xd5, 0xd4, 0xbc, 0x03,
		0xa9, 0x06, 0xa9, 0x80, 0x0d, 0xb3, 0x46, 0x2d,
		0xe3, 0xc6, 0xaf, 0x1a, 0x39, 0x18, 0x7e, 0x1e,
		0x83, 0x80, 0x46, 0x11, 0xd2, 0x13, 0x9f, 0xda,
		0xfc, 0x2d, 0x42, 0xaa, 0x5a, 0x1d, 0x4c, 0x31,
		0xe5, 0x58, 0x78, 0x5e, 0xe2, 0x04, 0xd6, 0x23,
		0x7f, 0x3f, 0x06, 0xc0, 0x54, 0xf8,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x72, 0x04, 0x00, 0x00,
		0x6e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x65,
		0xe8, 0x4b, 0xfb, 0xef, 0xba, 0xed, 0xc5, 0x36,
		0xc8, 0x5a, 0x41, 0x3f, 0x05, 0xfa, 0xfe, 0x48,
		0xc3, 0x91, 0x12, 0x8b, 0xe8, 0x32, 0x6a, 0x9f,
		0xdc, 0x97, 0xe2, 0x77, 0xb9, 0x96, 0x2d, 0xd4,
		0xe5, 0xbd, 0xa1, 0xfd, 0x94, 0xbb, 0x74, 0x63,
		0xb1, 0x0c, 0x38, 0xbc, 0x6f, 0x69, 0xaf, 0xa3,
		0x46, 0x9c, 0x96, 0x41, 0xde, 0x59, 0x23, 0xff,
		0x15, 0x6b, 0x3a, 0xef, 0x91, 0x6d, 0x92, 0x44,
		0xdc, 0x72, 0x1f, 0x40, 0x3d, 0xb5, 0x34, 0x8f,
		0x2a, 0xac, 0x21, 0x69, 0x05, 0x6f, 0xb2, 0x60,
		0x32, 0x5d, 0x3d, 0x97, 0xb4, 0x24, 0x99, 0x14,
		0x03, 0x01, 0x00, 0x01, 0x01, 0x16, 0x03, 0x01,
		0x00, 0x30, 0x68, 0x27, 0x97, 0xca, 0x63, 0x09,
		0x22, 0xed, 0x0e, 0x61, 0x7c, 0x76, 0x31, 0x9c,
		0xbe, 0x27, 0xc9, 0xe6, 0x09, 0xc3, 0xc3, 0xc2,
		0xf4, 0xa2, 0x32, 0xba, 0x7c, 0xf2, 0x0f, 0xb8,
		0x3d, 0xcb, 0xe2, 0x4c, 0xc0, 0x7d, 0x8e, 0x5b,
		0x5a, 0xed, 0x05, 0x5c, 0x15, 0x96, 0x69, 0xc2,
		0x6f, 0x5f, 0x17, 0x03, 0x01, 0x00, 0x20, 0x5a,
		0xfe, 0x0b, 0xe1, 0x6f, 0xa8, 0x54, 0x19, 0x78,
		0xca, 0xba, 0x2e, 0x1e, 0x2e, 0xe1, 0x5d, 0x17,
		0xe5, 0x97, 0x05, 0x2c, 0x08, 0x0c, 0xff, 0xa8,
		0x59, 0xa9, 0xde, 0x5e, 0x21, 0x34, 0x04, 0x17,
		0x03, 0x01, 0x00, 0x30, 0x86, 0xb1, 0x3f, 0x88,
		0x43, 0xf0, 0x07, 0xee, 0xa8, 0xf4, 0xbc, 0xe7,
		0x5f, 0xc6, 0x8c, 0x86, 0x4c, 0xca, 0x70, 0x88,
		0xcc, 0x6a, 0xb4, 0x3d, 0x40, 0xe8, 0x54, 0x89,
		0x19, 0x43, 0x1f, 0x76, 0xe2, 0xac, 0xb2, 0x5b,
		0x92, 0xf8, 0x57, 0x39, 0x2a, 0xc3, 0x6d, 0x13,
		0x45, 0xfa, 0x36, 0x9e, 0x15, 0x03, 0x01, 0x00,
		0x20, 0x6d, 0xed, 0x7b, 0x59, 0x28, 0x2a, 0x27,
		0x04, 0x15, 0x07, 0x4e, 0xeb, 0x13, 0x00, 0xe3,
		0x3a, 0x3f, 0xf8, 0xaa, 0x2b, 0x3b, 0x1a, 0x8c,
		0x12, 0xd6, 0x4c, 0xec, 0x2a, 0xaf, 0x33, 0x60,
		0xaf,
	},
}

// Generated using:
// $ go test -test.run TestRunServer -serve -ciphersuites=0xc00a
// $ openssl s_client -host 127.0.0.1 -port 10443 -cipher ECDHE-ECDSA-AES256-SHA
var ecdheECDSAAESServerScript = [][]byte{
	{
		0x16, 0x03, 0x01, 0x00, 0xa0, 0x01, 0x00, 0x00,
		0x9c, 0x03, 0x03, 0x50, 0xd7, 0x18, 0x31, 0x49,
		0xde, 0x19, 0x8d, 0x08, 0x5c, 0x4b, 0x60, 0x67,
		0x0f, 0xfe, 0xd0, 0x62, 0xf9, 0x31, 0x48, 0x17,
		0x9e, 0x50, 0xc1, 0xd8, 0x35, 0x24, 0x0e, 0xa6,
		0x09, 0x06, 0x51, 0x00, 0x00, 0x04, 0xc0, 0x0a,
		0x00, 0xff, 0x01, 0x00, 0x00, 0x6f, 0x00, 0x0b,
		0x00, 0x04, 0x03, 0x00, 0x01, 0x02, 0x00, 0x0a,
		0x00, 0x34, 0x00, 0x32, 0x00, 0x0e, 0x00, 0x0d,
		0x00, 0x19, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x18,
		0x00, 0x09, 0x00, 0x0a, 0x00, 0x16, 0x00, 0x17,
		0x00, 0x08, 0x00, 0x06, 0x00, 0x07, 0x00, 0x14,
		0x00, 0x15, 0x00, 0x04, 0x00, 0x05, 0x00, 0x12,
		0x00, 0x13, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03,
		0x00, 0x0f, 0x00, 0x10, 0x00, 0x11, 0x00, 0x23,
		0x00, 0x00, 0x00, 0x0d, 0x00, 0x22, 0x00, 0x20,
		0x06, 0x01, 0x06, 0x02, 0x06, 0x03, 0x05, 0x01,
		0x05, 0x02, 0x05, 0x03, 0x04, 0x01, 0x04, 0x02,
		0x04, 0x03, 0x03, 0x01, 0x03, 0x02, 0x03, 0x03,
		0x02, 0x01, 0x02, 0x02, 0x02, 0x03, 0x01, 0x01,
		0x00, 0x0f, 0x00, 0x01, 0x01,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x30, 0x02, 0x00, 0x00,
		0x2c, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0xc0, 0x0a, 0x00, 0x00,
		0x04, 0x00, 0x23, 0x00, 0x00, 0x16, 0x03, 0x01,
		0x02, 0x0e, 0x0b, 0x00, 0x02, 0x0a, 0x00, 0x02,
		0x07, 0x00, 0x02, 0x04, 0x30, 0x82, 0x02, 0x00,
		0x30, 0x82, 0x01, 0x62, 0x02, 0x09, 0x00, 0xb8,
		0xbf, 0x2d, 0x47, 0xa0, 0xd2, 0xeb, 0xf4, 0x30,
		0x09, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d,
		0x04, 0x01, 0x30, 0x45, 0x31, 0x0b, 0x30, 0x09,
		0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x41,
		0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55,
		0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d, 0x65,
		0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x21,
		0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13,
		0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65,
		0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69, 0x74,
		0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c, 0x74,
		0x64, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x32, 0x31,
		0x31, 0x32, 0x32, 0x31, 0x35, 0x30, 0x36, 0x33,
		0x32, 0x5a, 0x17, 0x0d, 0x32, 0x32, 0x31, 0x31,
		0x32, 0x30, 0x31, 0x35, 0x30, 0x36, 0x33, 0x32,
		0x5a, 0x30, 0x45, 0x31, 0x0b, 0x30, 0x09, 0x06,
		0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x41, 0x55,
		0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04,
		0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d, 0x65, 0x2d,
		0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x21, 0x30,
		0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x18,
		0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65, 0x74,
		0x20, 0x57, 0x69, 0x64, 0x67, 0x69, 0x74, 0x73,
		0x20, 0x50, 0x74, 0x79, 0x20, 0x4c, 0x74, 0x64,
		0x30, 0x81, 0x9b, 0x30, 0x10, 0x06, 0x07, 0x2a,
		0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x05,
		0x2b, 0x81, 0x04, 0x00, 0x23, 0x03, 0x81, 0x86,
		0x00, 0x04, 0x00, 0xc4, 0xa1, 0xed, 0xbe, 0x98,
		0xf9, 0x0b, 0x48, 0x73, 0x36, 0x7e, 0xc3, 0x16,
		0x56, 0x11, 0x22, 0xf2, 0x3d, 0x53, 0xc3, 0x3b,
		0x4d, 0x21, 0x3d, 0xcd, 0x6b, 0x75, 0xe6, 0xf6,
		0xb0, 0xdc, 0x9a, 0xdf, 0x26, 0xc1, 0xbc, 0xb2,
		0x87, 0xf0, 0x72, 0x32, 0x7c, 0xb3, 0x64, 0x2f,
		0x1c, 0x90, 0xbc, 0xea, 0x68, 0x23, 0x10, 0x7e,
		0xfe, 0xe3, 0x25, 0xc0, 0x48, 0x3a, 0x69, 0xe0,
		0x28, 0x6d, 0xd3, 0x37, 0x00, 0xef, 0x04, 0x62,
		0xdd, 0x0d, 0xa0, 0x9c, 0x70, 0x62, 0x83, 0xd8,
		0x81, 0xd3, 0x64, 0x31, 0xaa, 0x9e, 0x97, 0x31,
		0xbd, 0x96, 0xb0, 0x68, 0xc0, 0x9b, 0x23, 0xde,
		0x76, 0x64, 0x3f, 0x1a, 0x5c, 0x7f, 0xe9, 0x12,
		0x0e, 0x58, 0x58, 0xb6, 0x5f, 0x70, 0xdd, 0x9b,
		0xd8, 0xea, 0xd5, 0xd7, 0xf5, 0xd5, 0xcc, 0xb9,
		0xb6, 0x9f, 0x30, 0x66, 0x5b, 0x66, 0x9a, 0x20,
		0xe2, 0x27, 0xe5, 0xbf, 0xfe, 0x3b, 0x30, 0x09,
		0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04,
		0x01, 0x03, 0x81, 0x8c, 0x00, 0x30, 0x81, 0x88,
		0x02, 0x42, 0x01, 0x88, 0xa2, 0x4f, 0xeb, 0xe2,
		0x45, 0xc5, 0x48, 0x7d, 0x1b, 0xac, 0xf5, 0xed,
		0x98, 0x9d, 0xae, 0x47, 0x70, 0xc0, 0x5e, 0x1b,
		0xb6, 0x2f, 0xbd, 0xf1, 0xb6, 0x4d, 0xb7, 0x61,
		0x40, 0xd3, 0x11, 0xa2, 0xce, 0xee, 0x0b, 0x7e,
		0x92, 0x7e, 0xff, 0x76, 0x9d, 0xc3, 0x3b, 0x7e,
		0xa5, 0x3f, 0xce, 0xfa, 0x10, 0xe2, 0x59, 0xec,
		0x47, 0x2d, 0x7c, 0xac, 0xda, 0x4e, 0x97, 0x0e,
		0x15, 0xa0, 0x6f, 0xd0, 0x02, 0x42, 0x01, 0x4d,
		0xfc, 0xbe, 0x67, 0x13, 0x9c, 0x2d, 0x05, 0x0e,
		0xbd, 0x3f, 0xa3, 0x8c, 0x25, 0xc1, 0x33, 0x13,
		0x83, 0x0d, 0x94, 0x06, 0xbb, 0xd4, 0x37, 0x7a,
		0xf6, 0xec, 0x7a, 0xc9, 0x86, 0x2e, 0xdd, 0xd7,
		0x11, 0x69, 0x7f, 0x85, 0x7c, 0x56, 0xde, 0xfb,
		0x31, 0x78, 0x2b, 0xe4, 0xc7, 0x78, 0x0d, 0xae,
		0xcb, 0xbe, 0x9e, 0x4e, 0x36, 0x24, 0x31, 0x7b,
		0x6a, 0x0f, 0x39, 0x95, 0x12, 0x07, 0x8f, 0x2a,
		0x16, 0x03, 0x01, 0x01, 0x1a, 0x0c, 0x00, 0x01,
		0x16, 0x03, 0x00, 0x19, 0x85, 0x04, 0x01, 0x39,
		0xdc, 0xee, 0x44, 0x17, 0x5e, 0xdb, 0xd7, 0x27,
		0xaf, 0xb6, 0x56, 0xd9, 0xb4, 0x43, 0x5a, 0x99,
		0xcf, 0xaa, 0x31, 0x37, 0x0c, 0x6f, 0x3a, 0xa0,
		0xf8, 0x53, 0xc4, 0x74, 0xd1, 0x91, 0x0a, 0x46,
		0xf5, 0x38, 0x3b, 0x5c, 0x09, 0xd8, 0x97, 0xdc,
		0x4b, 0xaa, 0x70, 0x26, 0x48, 0xf2, 0xd6, 0x0b,
		0x31, 0xc9, 0xf8, 0xd4, 0x98, 0x43, 0xe1, 0x6c,
		0xd5, 0xc7, 0xb2, 0x8e, 0x0b, 0x01, 0xe6, 0xb6,
		0x00, 0x28, 0x80, 0x7b, 0xfc, 0x96, 0x8f, 0x0d,
		0xa2, 0x4f, 0xb0, 0x79, 0xaf, 0xdc, 0x61, 0x28,
		0x63, 0x33, 0x78, 0xf6, 0x31, 0x39, 0xfd, 0x8a,
		0xf4, 0x15, 0x18, 0x11, 0xfe, 0xdb, 0xd5, 0x07,
		0xda, 0x2c, 0xed, 0x49, 0xa0, 0x23, 0xbf, 0xd0,
		0x3a, 0x38, 0x1d, 0x54, 0xae, 0x1c, 0x7b, 0xea,
		0x29, 0xee, 0xd0, 0x38, 0xc1, 0x76, 0xa7, 0x7f,
		0x2a, 0xf4, 0xce, 0x1e, 0xac, 0xcc, 0x94, 0x79,
		0x90, 0x33, 0x00, 0x8b, 0x30, 0x81, 0x88, 0x02,
		0x42, 0x00, 0xc6, 0x85, 0x8e, 0x06, 0xb7, 0x04,
		0x04, 0xe9, 0xcd, 0x9e, 0x3e, 0xcb, 0x66, 0x23,
		0x95, 0xb4, 0x42, 0x9c, 0x64, 0x81, 0x39, 0x05,
		0x3f, 0xb5, 0x21, 0xf8, 0x28, 0xaf, 0x60, 0x6b,
		0x4d, 0x3d, 0xba, 0xa1, 0x4b, 0x5e, 0x77, 0xef,
		0xe7, 0x59, 0x28, 0xfe, 0x1d, 0xc1, 0x27, 0xa2,
		0xff, 0xa8, 0xde, 0x33, 0x48, 0xb3, 0xc1, 0x85,
		0x6a, 0x42, 0x9b, 0xf9, 0x7e, 0x7e, 0x31, 0xc2,
		0xe5, 0xbd, 0x66, 0x02, 0x42, 0x00, 0xad, 0x7d,
		0x06, 0x35, 0xab, 0xec, 0x8d, 0xac, 0xd4, 0xba,
		0x1b, 0x49, 0x5e, 0x05, 0x5f, 0xf0, 0x97, 0x93,
		0x82, 0xb8, 0x2b, 0x8d, 0x91, 0x98, 0x63, 0x8e,
		0xb4, 0x14, 0x62, 0xdb, 0x1e, 0xc9, 0x2b, 0x30,
		0xf8, 0x41, 0x9b, 0xa6, 0xe6, 0xbc, 0xde, 0x0e,
		0x68, 0x30, 0x22, 0x50, 0xe6, 0x98, 0x97, 0x7b,
		0x69, 0xf7, 0x93, 0xed, 0xcd, 0x19, 0x2f, 0x44,
		0x6c, 0x2e, 0xdf, 0x25, 0xee, 0xcc, 0x46, 0x16,
		0x03, 0x01, 0x00, 0x04, 0x0e, 0x00, 0x00, 0x00,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x8a, 0x10, 0x00, 0x00,
		0x86, 0x85, 0x04, 0x00, 0x1c, 0xc5, 0xe8, 0xb3,
		0x42, 0xb4, 0xad, 0xca, 0x45, 0xcd, 0x42, 0x7b,
		0xfb, 0x0c, 0xea, 0x32, 0x26, 0xd4, 0x8a, 0xef,
		0xdf, 0xc9, 0xff, 0xd2, 0xe0, 0x36, 0xea, 0x4e,
		0xbb, 0x3e, 0xf4, 0x9c, 0x76, 0x4f, 0x44, 0xbd,
		0x84, 0x72, 0xdd, 0xcb, 0xe5, 0x28, 0x8d, 0x31,
		0x72, 0x3b, 0xd3, 0xf2, 0x9a, 0x13, 0xfb, 0x8a,
		0xa7, 0x72, 0xca, 0x21, 0x6c, 0xea, 0xbf, 0xe9,
		0x8c, 0x0a, 0xcc, 0x8f, 0xd6, 0x00, 0x20, 0x87,
		0xf3, 0x7d, 0x18, 0xc5, 0xfd, 0x9e, 0xdd, 0x6b,
		0x06, 0xdc, 0x52, 0xeb, 0x14, 0xc0, 0x67, 0x5a,
		0x06, 0xd8, 0x98, 0x19, 0x14, 0xe7, 0xd4, 0x36,
		0x32, 0xee, 0xb7, 0xfa, 0xe2, 0x85, 0x4a, 0x16,
		0x42, 0x0c, 0xa6, 0x21, 0xcf, 0x1f, 0xae, 0x10,
		0x8b, 0x28, 0x32, 0x19, 0xa4, 0x0a, 0xd7, 0xce,
		0xe6, 0xe1, 0x93, 0xfb, 0x5f, 0x08, 0x8b, 0x42,
		0xa2, 0x20, 0xed, 0x0d, 0x62, 0xca, 0xed, 0x14,
		0x03, 0x01, 0x00, 0x01, 0x01, 0x16, 0x03, 0x01,
		0x00, 0x30, 0x2e, 0x33, 0xc0, 0x57, 0x6c, 0xb4,
		0x1b, 0xd2, 0x63, 0xe8, 0x67, 0x10, 0x2d, 0x87,
		0x71, 0x6e, 0x19, 0x60, 0xf4, 0xa4, 0x10, 0x52,
		0x73, 0x2d, 0x09, 0x5e, 0xdb, 0x6c, 0xdc, 0xcf,
		0x2d, 0xff, 0x03, 0x11, 0x95, 0x76, 0x90, 0xd7,
		0x87, 0x54, 0x43, 0xed, 0xc2, 0x36, 0x69, 0x14,
		0x72, 0x4a,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x72, 0x04, 0x00, 0x00,
		0x6e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x65,
		0xe8, 0x8b, 0xde, 0xef, 0xba, 0xc5, 0x7e, 0x04,
		0xab, 0xfd, 0x79, 0x56, 0xf3, 0xe1, 0xa5, 0x3e,
		0x02, 0xdf, 0x69, 0x6d, 0x1f, 0x41, 0x9f, 0xbc,
		0x93, 0xe2, 0x6c, 0xf1, 0xb1, 0x38, 0xf5, 0x2b,
		0x8c, 0x4c, 0xf4, 0x74, 0xe1, 0x79, 0x35, 0x34,
		0x97, 0x9b, 0xd5, 0xba, 0xfd, 0xf7, 0x2f, 0x2d,
		0x9e, 0x84, 0x54, 0xee, 0x77, 0x59, 0x23, 0x8f,
		0xc8, 0x84, 0xb4, 0xd6, 0xea, 0x4c, 0x44, 0x8a,
		0xc6, 0x9c, 0xf9, 0x9b, 0x27, 0xea, 0x4f, 0x28,
		0x72, 0x33, 0x12, 0x20, 0x7c, 0xd7, 0x3f, 0x56,
		0xa6, 0x76, 0xc7, 0x48, 0xe4, 0x2d, 0x6f, 0x14,
		0x03, 0x01, 0x00, 0x01, 0x01, 0x16, 0x03, 0x01,
		0x00, 0x30, 0x36, 0xe3, 0xd4, 0xf7, 0xb1, 0x69,
		0x18, 0x8d, 0x09, 0xba, 0x52, 0x1e, 0xd5, 0x7d,
		0x2c, 0x15, 0x3a, 0xd6, 0xe3, 0x99, 0x30, 0x2c,
		0x99, 0x97, 0xbc, 0x19, 0x3c, 0x63, 0xa1, 0x25,
		0x68, 0xbc, 0x8a, 0x16, 0x47, 0xec, 0xae, 0x13,
		0xa4, 0x03, 0x96, 0x29, 0x11, 0x92, 0x90, 0x1a,
		0xc8, 0xa4, 0x17, 0x03, 0x01, 0x00, 0x20, 0xc1,
		0x10, 0x1d, 0xa6, 0xf1, 0xe2, 0x8a, 0xcc, 0x37,
		0x7d, 0x8e, 0x05, 0x00, 0xfb, 0xd1, 0x9f, 0xc7,
		0x11, 0xd2, 0x00, 0xb4, 0x27, 0x0a, 0x25, 0x14,
		0xd9, 0x79, 0x1b, 0xcb, 0x4d, 0x81, 0x61, 0x17,
		0x03, 0x01, 0x00, 0x30, 0x5c, 0x7c, 0x2d, 0xc0,
		0x9e, 0xa6, 0xc4, 0x8e, 0xfd, 0xf4, 0xe2, 0xe5,
		0xe4, 0xe6, 0x56, 0x9f, 0x7d, 0x4c, 0x4c, 0x2d,
		0xb7, 0xa9, 0xac, 0xfa, 0x9f, 0x12, 0x7f, 0x2d,
		0x30, 0x57, 0xe4, 0x8e, 0x30, 0x86, 0x65, 0x59,
		0xcd, 0x24, 0xda, 0xe2, 0x8a, 0x7b, 0x0c, 0x5e,
		0x86, 0x05, 0x06, 0x2a, 0x15, 0x03, 0x01, 0x00,
		0x20, 0xd6, 0xb7, 0x70, 0xf8, 0x47, 0xbc, 0x0f,
		0xf4, 0x66, 0x98, 0x1b, 0x1e, 0x8a, 0x8c, 0x0b,
		0xa1, 0x4a, 0x04, 0x29, 0x60, 0x72, 0x8b, 0xc4,
		0x73, 0xc1, 0xd6, 0x41, 0x72, 0xb7, 0x17, 0x39,
		0xda,
	},
}

var sslv3ServerScript = [][]byte{
	{
		0x16, 0x03, 0x00, 0x00, 0x54, 0x01, 0x00, 0x00,
		0x50, 0x03, 0x00, 0x50, 0x77, 0x3d, 0x42, 0xae,
		0x84, 0xbd, 0xc5, 0x07, 0xa5, 0xc4, 0xd6, 0x16,
		0x4e, 0xd5, 0xc5, 0xfa, 0x02, 0x7a, 0x0f, 0x1d,
		0xc1, 0xe1, 0xaa, 0xe3, 0x3b, 0x4b, 0x6f, 0x11,
		0xfa, 0x1a, 0xa4, 0x00, 0x00, 0x28, 0x00, 0x39,
		0x00, 0x38, 0x00, 0x35, 0x00, 0x16, 0x00, 0x13,
		0x00, 0x0a, 0x00, 0x33, 0x00, 0x32, 0x00, 0x2f,
		0x00, 0x05, 0x00, 0x04, 0x00, 0x15, 0x00, 0x12,
		0x00, 0x09, 0x00, 0x14, 0x00, 0x11, 0x00, 0x08,
		0x00, 0x06, 0x00, 0x03, 0x00, 0xff, 0x02, 0x01,
		0x00,
	},
	{
		0x16, 0x03, 0x00, 0x00, 0x2a, 0x02, 0x00, 0x00,
		0x26, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x16,
		0x03, 0x00, 0x02, 0xbe, 0x0b, 0x00, 0x02, 0xba,
		0x00, 0x02, 0xb7, 0x00, 0x02, 0xb4, 0x30, 0x82,
		0x02, 0xb0, 0x30, 0x82, 0x02, 0x19, 0xa0, 0x03,
		0x02, 0x01, 0x02, 0x02, 0x09, 0x00, 0x85, 0xb0,
		0xbb, 0xa4, 0x8a, 0x7f, 0xb8, 0xca, 0x30, 0x0d,
		0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
		0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x45, 0x31,
		0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
		0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11,
		0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53,
		0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74,
		0x65, 0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55,
		0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65,
		0x72, 0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64,
		0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79,
		0x20, 0x4c, 0x74, 0x64, 0x30, 0x1e, 0x17, 0x0d,
		0x31, 0x30, 0x30, 0x34, 0x32, 0x34, 0x30, 0x39,
		0x30, 0x39, 0x33, 0x38, 0x5a, 0x17, 0x0d, 0x31,
		0x31, 0x30, 0x34, 0x32, 0x34, 0x30, 0x39, 0x30,
		0x39, 0x33, 0x38, 0x5a, 0x30, 0x45, 0x31, 0x0b,
		0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
		0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11, 0x06,
		0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f,
		0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x65,
		0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x04,
		0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65, 0x72,
		0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64, 0x67,
		0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20,
		0x4c, 0x74, 0x64, 0x30, 0x81, 0x9f, 0x30, 0x0d,
		0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
		0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8d,
		0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00,
		0xbb, 0x79, 0xd6, 0xf5, 0x17, 0xb5, 0xe5, 0xbf,
		0x46, 0x10, 0xd0, 0xdc, 0x69, 0xbe, 0xe6, 0x2b,
		0x07, 0x43, 0x5a, 0xd0, 0x03, 0x2d, 0x8a, 0x7a,
		0x43, 0x85, 0xb7, 0x14, 0x52, 0xe7, 0xa5, 0x65,
		0x4c, 0x2c, 0x78, 0xb8, 0x23, 0x8c, 0xb5, 0xb4,
		0x82, 0xe5, 0xde, 0x1f, 0x95, 0x3b, 0x7e, 0x62,
		0xa5, 0x2c, 0xa5, 0x33, 0xd6, 0xfe, 0x12, 0x5c,
		0x7a, 0x56, 0xfc, 0xf5, 0x06, 0xbf, 0xfa, 0x58,
		0x7b, 0x26, 0x3f, 0xb5, 0xcd, 0x04, 0xd3, 0xd0,
		0xc9, 0x21, 0x96, 0x4a, 0xc7, 0xf4, 0x54, 0x9f,
		0x5a, 0xbf, 0xef, 0x42, 0x71, 0x00, 0xfe, 0x18,
		0x99, 0x07, 0x7f, 0x7e, 0x88, 0x7d, 0x7d, 0xf1,
		0x04, 0x39, 0xc4, 0xa2, 0x2e, 0xdb, 0x51, 0xc9,
		0x7c, 0xe3, 0xc0, 0x4c, 0x3b, 0x32, 0x66, 0x01,
		0xcf, 0xaf, 0xb1, 0x1d, 0xb8, 0x71, 0x9a, 0x1d,
		0xdb, 0xdb, 0x89, 0x6b, 0xae, 0xda, 0x2d, 0x79,
		0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x81, 0xa7,
		0x30, 0x81, 0xa4, 0x30, 0x1d, 0x06, 0x03, 0x55,
		0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xb1, 0xad,
		0xe2, 0x85, 0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69,
		0xce, 0x23, 0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18,
		0x88, 0x39, 0x30, 0x75, 0x06, 0x03, 0x55, 0x1d,
		0x23, 0x04, 0x6e, 0x30, 0x6c, 0x80, 0x14, 0xb1,
		0xad, 0xe2, 0x85, 0x5a, 0xcf, 0xcb, 0x28, 0xdb,
		0x69, 0xce, 0x23, 0x69, 0xde, 0xd3, 0x26, 0x8e,
		0x18, 0x88, 0x39, 0xa1, 0x49, 0xa4, 0x47, 0x30,
		0x45, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55,
		0x04, 0x06, 0x13, 0x02, 0x41, 0x55, 0x31, 0x13,
		0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x13,
		0x0a, 0x53, 0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74,
		0x61, 0x74, 0x65, 0x31, 0x21, 0x30, 0x1f, 0x06,
		0x03, 0x55, 0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e,
		0x74, 0x65, 0x72, 0x6e, 0x65, 0x74, 0x20, 0x57,
		0x69, 0x64, 0x67, 0x69, 0x74, 0x73, 0x20, 0x50,
		0x74, 0x79, 0x20, 0x4c, 0x74, 0x64, 0x82, 0x09,
		0x00, 0x85, 0xb0, 0xbb, 0xa4, 0x8a, 0x7f, 0xb8,
		0xca, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13,
		0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xff, 0x30,
		0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
		0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x81,
		0x81, 0x00, 0x08, 0x6c, 0x45, 0x24, 0xc7, 0x6b,
		0xb1, 0x59, 0xab, 0x0c, 0x52, 0xcc, 0xf2, 0xb0,
		0x14, 0xd7, 0x87, 0x9d, 0x7a, 0x64, 0x75, 0xb5,
		0x5a, 0x95, 0x66, 0xe4, 0xc5, 0x2b, 0x8e, 0xae,
		0x12, 0x66, 0x1f, 0xeb, 0x4f, 0x38, 0xb3, 0x6e,
		0x60, 0xd3, 0x92, 0xfd, 0xf7, 0x41, 0x08, 0xb5,
		0x25, 0x13, 0xb1, 0x18, 0x7a, 0x24, 0xfb, 0x30,
		0x1d, 0xba, 0xed, 0x98, 0xb9, 0x17, 0xec, 0xe7,
		0xd7, 0x31, 0x59, 0xdb, 0x95, 0xd3, 0x1d, 0x78,
		0xea, 0x50, 0x56, 0x5c, 0xd5, 0x82, 0x5a, 0x2d,
		0x5a, 0x5f, 0x33, 0xc4, 0xb6, 0xd8, 0xc9, 0x75,
		0x90, 0x96, 0x8c, 0x0f, 0x52, 0x98, 0xb5, 0xcd,
		0x98, 0x1f, 0x89, 0x20, 0x5f, 0xf2, 0xa0, 0x1c,
		0xa3, 0x1b, 0x96, 0x94, 0xdd, 0xa9, 0xfd, 0x57,
		0xe9, 0x70, 0xe8, 0x26, 0x6d, 0x71, 0x99, 0x9b,
		0x26, 0x6e, 0x38, 0x50, 0x29, 0x6c, 0x90, 0xa7,
		0xbd, 0xd9, 0x16, 0x03, 0x00, 0x00, 0x04, 0x0e,
		0x00, 0x00, 0x00,
	},
	{
		0x16, 0x03, 0x00, 0x00, 0x84, 0x10, 0x00, 0x00,
		0x80, 0x4a, 0x8d, 0xc4, 0x38, 0x7a, 0x9c, 0xd6,
		0xe8, 0x72, 0x9e, 0xa3, 0xdf, 0x37, 0xb4, 0x6c,
		0x58, 0x33, 0x59, 0xd9, 0xc9, 0x4b, 0x50, 0x33,
		0x6c, 0xed, 0x73, 0x38, 0x2a, 0x46, 0x55, 0x31,
		0xa9, 0x8e, 0x8e, 0xfc, 0x0b, 0x5d, 0x5f, 0x3c,
		0x88, 0x28, 0x3f, 0x60, 0x51, 0x13, 0xf1, 0x59,
		0x0c, 0xa3, 0x5e, 0xe0, 0xa3, 0x35, 0x06, 0xb1,
		0x71, 0x59, 0x24, 0x4e, 0xed, 0x07, 0x15, 0x88,
		0x50, 0xef, 0xc2, 0xb2, 0x2a, 0x52, 0x30, 0x6a,
		0x7c, 0xbe, 0x2f, 0xc6, 0x8f, 0xa8, 0x83, 0xc5,
		0x80, 0x14, 0x62, 0x74, 0x7f, 0x96, 0x9f, 0x41,
		0x32, 0x74, 0xdd, 0x76, 0x2d, 0x7b, 0xeb, 0x7b,
		0xea, 0xd0, 0x4f, 0x0c, 0xcf, 0x9a, 0x9c, 0xc5,
		0x7a, 0xe4, 0xbc, 0xf8, 0xa6, 0xe1, 0x09, 0x8e,
		0x7c, 0x53, 0x3a, 0xe3, 0x30, 0x8f, 0x76, 0xee,
		0x58, 0xbb, 0xfd, 0x0b, 0x06, 0xb8, 0xdf, 0xb7,
		0x31, 0x14, 0x03, 0x00, 0x00, 0x01, 0x01, 0x16,
		0x03, 0x00, 0x00, 0x3c, 0x13, 0x91, 0xc6, 0x4a,
		0x0c, 0x59, 0x25, 0xce, 0x54, 0xc0, 0x1d, 0xb9,
		0x2a, 0xff, 0x4d, 0xca, 0x26, 0x0c, 0x8c, 0x04,
		0x98, 0x7c, 0x7c, 0x38, 0xa3, 0xf5, 0xf9, 0x36,
		0x1c, 0x04, 0x32, 0x47, 0x2d, 0x48, 0x0e, 0x96,
		0xe8, 0x2b, 0x5e, 0x5a, 0xc6, 0x0a, 0x48, 0x41,
		0x34, 0x5e, 0x62, 0xd5, 0x68, 0x4e, 0x44, 0x1d,
		0xb2, 0xa1, 0x11, 0xad, 0x6e, 0x14, 0x85, 0x61,
	},
	{
		0x14, 0x03, 0x00, 0x00, 0x01, 0x01, 0x16, 0x03,
		0x00, 0x00, 0x3c, 0x88, 0xae, 0xa9, 0xd4, 0xa8,
		0x10, 0x8d, 0x65, 0xa6, 0x3e, 0x1e, 0xed, 0xd2,
		0xfc, 0xc4, 0x7c, 0xa8, 0x94, 0x4f, 0x11, 0xaf,
		0xa6, 0x87, 0x09, 0x37, 0x54, 0xf7, 0x69, 0xd1,
		0xb5, 0x25, 0x6b, 0xb5, 0xed, 0xcb, 0x25, 0x39,
		0x73, 0xeb, 0x53, 0x6c, 0xc7, 0xb4, 0x29, 0x8f,
		0xd6, 0x49, 0xd1, 0x95, 0x59, 0x80, 0x9a, 0x67,
		0x5c, 0xb2, 0xe0, 0xbd, 0x1e, 0xff, 0xaa, 0x17,
		0x03, 0x00, 0x00, 0x21, 0x65, 0x7b, 0x99, 0x09,
		0x02, 0xc3, 0x9d, 0x54, 0xd6, 0xe7, 0x32, 0x62,
		0xab, 0xc1, 0x09, 0x91, 0x30, 0x0a, 0xc9, 0xfa,
		0x70, 0xec, 0x06, 0x7b, 0xa3, 0xe1, 0x5f, 0xb4,
		0x63, 0xe6, 0x5c, 0xba, 0x1f, 0x15, 0x03, 0x00,
		0x00, 0x16, 0x40, 0x70, 0xbe, 0xe6, 0xa6, 0xee,
		0x8f, 0xd0, 0x87, 0xa0, 0x43, 0xa1, 0x92, 0xd7,
		0xd0, 0x1a, 0x0c, 0x20, 0x7c, 0xbf, 0xa2, 0xb5,
	},
}

var selectCertificateBySNIScript = [][]byte{
	{
		0x16, 0x03, 0x01, 0x00, 0x6a, 0x01, 0x00, 0x00,
		0x66, 0x03, 0x01, 0x50, 0x77, 0x3d, 0xfe, 0xfb,
		0x8d, 0xc2, 0x68, 0xeb, 0xf9, 0xfa, 0x54, 0x97,
		0x86, 0x45, 0xa2, 0xa3, 0xed, 0xb1, 0x91, 0xb8,
		0x28, 0xc0, 0x47, 0xaf, 0xfb, 0xcd, 0xdc, 0x0e,
		0xb3, 0xea, 0xa5, 0x00, 0x00, 0x28, 0x00, 0x39,
		0x00, 0x38, 0x00, 0x35, 0x00, 0x16, 0x00, 0x13,
		0x00, 0x0a, 0x00, 0x33, 0x00, 0x32, 0x00, 0x2f,
		0x00, 0x05, 0x00, 0x04, 0x00, 0x15, 0x00, 0x12,
		0x00, 0x09, 0x00, 0x14, 0x00, 0x11, 0x00, 0x08,
		0x00, 0x06, 0x00, 0x03, 0x00, 0xff, 0x02, 0x01,
		0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x10, 0x00,
		0x0e, 0x00, 0x00, 0x0b, 0x73, 0x6e, 0x69, 0x74,
		0x65, 0x73, 0x74, 0x2e, 0x63, 0x6f, 0x6d,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x2a, 0x02, 0x00, 0x00,
		0x26, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x16,
		0x03, 0x01, 0x02, 0x00, 0x0b, 0x00, 0x01, 0xfc,
		0x00, 0x01, 0xf9, 0x00, 0x01, 0xf6, 0x30, 0x82,
		0x01, 0xf2, 0x30, 0x82, 0x01, 0x5d, 0xa0, 0x03,
		0x02, 0x01, 0x02, 0x02, 0x01, 0x00, 0x30, 0x0b,
		0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
		0x01, 0x01, 0x05, 0x30, 0x28, 0x31, 0x10, 0x30,
		0x0e, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x07,
		0x41, 0x63, 0x6d, 0x65, 0x20, 0x43, 0x6f, 0x31,
		0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x03,
		0x13, 0x0b, 0x73, 0x6e, 0x69, 0x74, 0x65, 0x73,
		0x74, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x1e, 0x17,
		0x0d, 0x31, 0x32, 0x30, 0x34, 0x31, 0x31, 0x31,
		0x37, 0x34, 0x30, 0x33, 0x35, 0x5a, 0x17, 0x0d,
		0x31, 0x33, 0x30, 0x34, 0x31, 0x31, 0x31, 0x37,
		0x34, 0x35, 0x33, 0x35, 0x5a, 0x30, 0x28, 0x31,
		0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x0a,
		0x13, 0x07, 0x41, 0x63, 0x6d, 0x65, 0x20, 0x43,
		0x6f, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55,
		0x04, 0x03, 0x13, 0x0b, 0x73, 0x6e, 0x69, 0x74,
		0x65, 0x73, 0x74, 0x2e, 0x63, 0x6f, 0x6d, 0x30,
		0x81, 0x9d, 0x30, 0x0b, 0x06, 0x09, 0x2a, 0x86,
		0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x03,
		0x81, 0x8d, 0x00, 0x30, 0x81, 0x89, 0x02, 0x81,
		0x81, 0x00, 0xbb, 0x79, 0xd6, 0xf5, 0x17, 0xb5,
		0xe5, 0xbf, 0x46, 0x10, 0xd0, 0xdc, 0x69, 0xbe,
		0xe6, 0x2b, 0x07, 0x43, 0x5a, 0xd0, 0x03, 0x2d,
		0x8a, 0x7a, 0x43, 0x85, 0xb7, 0x14, 0x52, 0xe7,
		0xa5, 0x65, 0x4c, 0x2c, 0x78, 0xb8, 0x23, 0x8c,
		0xb5, 0xb4, 0x82, 0xe5, 0xde, 0x1f, 0x95, 0x3b,
		0x7e, 0x62, 0xa5, 0x2c, 0xa5, 0x33, 0xd6, 0xfe,
		0x12, 0x5c, 0x7a, 0x56, 0xfc, 0xf5, 0x06, 0xbf,
		0xfa, 0x58, 0x7b, 0x26, 0x3f, 0xb5, 0xcd, 0x04,
		0xd3, 0xd0, 0xc9, 0x21, 0x96, 0x4a, 0xc7, 0xf4,
		0x54, 0x9f, 0x5a, 0xbf, 0xef, 0x42, 0x71, 0x00,
		0xfe, 0x18, 0x99, 0x07, 0x7f, 0x7e, 0x88, 0x7d,
		0x7d, 0xf1, 0x04, 0x39, 0xc4, 0xa2, 0x2e, 0xdb,
		0x51, 0xc9, 0x7c, 0xe3, 0xc0, 0x4c, 0x3b, 0x32,
		0x66, 0x01, 0xcf, 0xaf, 0xb1, 0x1d, 0xb8, 0x71,
		0x9a, 0x1d, 0xdb, 0xdb, 0x89, 0x6b, 0xae, 0xda,
		0x2d, 0x79, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3,
		0x32, 0x30, 0x30, 0x30, 0x0e, 0x06, 0x03, 0x55,
		0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03,
		0x02, 0x00, 0xa0, 0x30, 0x0d, 0x06, 0x03, 0x55,
		0x1d, 0x0e, 0x04, 0x06, 0x04, 0x04, 0x01, 0x02,
		0x03, 0x04, 0x30, 0x0f, 0x06, 0x03, 0x55, 0x1d,
		0x23, 0x04, 0x08, 0x30, 0x06, 0x80, 0x04, 0x01,
		0x02, 0x03, 0x04, 0x30, 0x0b, 0x06, 0x09, 0x2a,
		0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05,
		0x03, 0x81, 0x81, 0x00, 0x89, 0xc6, 0x45, 0x5f,
		0x1c, 0x1f, 0x5e, 0xf8, 0xeb, 0x1a, 0xb1, 0x74,
		0xee, 0x24, 0x39, 0x05, 0x9f, 0x5c, 0x42, 0x59,
		0xbb, 0x1a, 0x8d, 0x86, 0xcd, 0xb1, 0xd0, 0x56,
		0xf5, 0x6a, 0x71, 0x7d, 0xa4, 0x0e, 0x95, 0xab,
		0x90, 0xf5, 0x9e, 0x8d, 0xea, 0xf6, 0x27, 0xc1,
		0x57, 0x99, 0x50, 0x94, 0xdb, 0x08, 0x02, 0x26,
		0x6e, 0xb3, 0x4f, 0xc6, 0x84, 0x2d, 0xea, 0x8a,
		0x4b, 0x68, 0xd9, 0xc1, 0x38, 0x91, 0x03, 0xab,
		0x84, 0xfb, 0x9e, 0x1f, 0x85, 0xd9, 0xb5, 0xd2,
		0x3f, 0xf2, 0x31, 0x2c, 0x86, 0x70, 0xfb, 0xb5,
		0x40, 0x14, 0x82, 0x45, 0xa4, 0xeb, 0xaf, 0xe2,
		0x64, 0xd9, 0x0c, 0x8a, 0x4c, 0xf4, 0xf8, 0x5b,
		0x0f, 0xac, 0x12, 0xac, 0x2f, 0xc4, 0xa3, 0x15,
		0x4b, 0xad, 0x52, 0x46, 0x28, 0x68, 0xaf, 0x96,
		0xc6, 0x2c, 0x65, 0x25, 0xd6, 0x52, 0xb6, 0xe3,
		0x18, 0x45, 0xbd, 0xcc, 0x16, 0x03, 0x01, 0x00,
		0x04, 0x0e, 0x00, 0x00, 0x00,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x86, 0x10, 0x00, 0x00,
		0x82, 0x00, 0x80, 0x69, 0xc3, 0xd4, 0x0e, 0xcc,
		0xdc, 0xbc, 0x5e, 0xc2, 0x64, 0xa6, 0xde, 0x3c,
		0x0c, 0x7e, 0x0c, 0x6b, 0x80, 0x0f, 0xd4, 0x8f,
		0x02, 0x4b, 0xb2, 0xba, 0x8d, 0x01, 0xeb, 0x6b,
		0xa1, 0x2e, 0x79, 0x37, 0xba, 0xae, 0x24, 0xc2,
		0x26, 0x72, 0x51, 0xe1, 0x82, 0x8e, 0x51, 0x41,
		0x1c, 0x54, 0xa4, 0x26, 0xbe, 0x13, 0xcd, 0x1b,
		0xc6, 0xed, 0x3d, 0x1f, 0xfd, 0x72, 0x80, 0x90,
		0xdb, 0xbf, 0xd6, 0x39, 0x94, 0x5f, 0x48, 0xfb,
		0x25, 0x5a, 0xc9, 0x60, 0x9b, 0xd7, 0xc6, 0x20,
		0xa8, 0x66, 0x64, 0x13, 0xf3, 0x65, 0xc8, 0xb1,
		0xd5, 0x33, 0x21, 0x0e, 0x73, 0x41, 0xc0, 0x18,
		0x1a, 0x37, 0xfe, 0xcf, 0x28, 0x2a, 0xcd, 0xe4,
		0x0b, 0xac, 0xdd, 0x25, 0x5e, 0xcb, 0x17, 0x51,
		0x69, 0xd5, 0x8c, 0xf4, 0xb6, 0x21, 0x98, 0xef,
		0x20, 0xdb, 0x14, 0x67, 0xf3, 0x7c, 0x95, 0x6a,
		0x48, 0x2a, 0x6a, 0x14, 0x03, 0x01, 0x00, 0x01,
		0x01, 0x16, 0x03, 0x01, 0x00, 0x24, 0x36, 0x1b,
		0x09, 0xe5, 0xb9, 0xb9, 0x4d, 0x7d, 0xae, 0x87,
		0xb6, 0x0f, 0xaf, 0xec, 0x22, 0xba, 0x0d, 0xa5,
		0x96, 0x5e, 0x64, 0x65, 0xe7, 0xfb, 0xe3, 0xf3,
		0x6b, 0x72, 0xa8, 0xdb, 0xed, 0xd8, 0x69, 0x9c,
		0x08, 0xd8,
	},
	{
		0x14, 0x03, 0x01, 0x00, 0x01, 0x01, 0x16, 0x03,
		0x01, 0x00, 0x24, 0x60, 0xf7, 0x09, 0x5f, 0xd1,
		0xcb, 0xc9, 0xe1, 0x22, 0xb5, 0x2a, 0xcc, 0xde,
		0x7c, 0xa7, 0xb8, 0x85, 0x00, 0xbc, 0xfd, 0x85,
		0xe1, 0x91, 0x36, 0xbb, 0x07, 0x42, 0xad, 0x3d,
		0x29, 0x62, 0x69, 0xc1, 0x45, 0x92, 0x6f, 0x17,
		0x03, 0x01, 0x00, 0x21, 0x0d, 0xf9, 0xd5, 0x87,
		0xb9, 0x57, 0x3c, 0x50, 0x19, 0xe4, 0x3a, 0x50,
		0x45, 0xcc, 0x86, 0x89, 0xd4, 0x32, 0x79, 0x45,
		0x7c, 0x9f, 0x96, 0xd4, 0x54, 0x56, 0x0c, 0x63,
		0x72, 0x81, 0xc3, 0xd3, 0xe3, 0x15, 0x03, 0x01,
		0x00, 0x16, 0x84, 0xec, 0x2e, 0xf6, 0xaf, 0x4f,
		0xee, 0x48, 0x0f, 0xbe, 0xcd, 0x82, 0x5c, 0x56,
		0x16, 0xe4, 0xfb, 0x89, 0xc5, 0x57, 0x3e, 0x91,
	},
}

var issueSessionTicketTest = [][]byte{
	{
		0x16, 0x03, 0x01, 0x00, 0x5a, 0x01, 0x00, 0x00,
		0x56, 0x03, 0x01, 0x50, 0x77, 0x3e, 0x49, 0x7a,
		0xb7, 0x86, 0x5c, 0x27, 0xd2, 0x97, 0x61, 0xe3,
		0x49, 0x41, 0x48, 0xe7, 0x0e, 0xaa, 0x7e, 0x4d,
		0xb8, 0xdc, 0x01, 0x97, 0xfb, 0xab, 0x53, 0xb2,
		0x5e, 0x36, 0xf6, 0x00, 0x00, 0x28, 0x00, 0x39,
		0x00, 0x38, 0x00, 0x35, 0x00, 0x16, 0x00, 0x13,
		0x00, 0x0a, 0x00, 0x33, 0x00, 0x32, 0x00, 0x2f,
		0x00, 0x05, 0x00, 0x04, 0x00, 0x15, 0x00, 0x12,
		0x00, 0x09, 0x00, 0x14, 0x00, 0x11, 0x00, 0x08,
		0x00, 0x06, 0x00, 0x03, 0x00, 0xff, 0x02, 0x01,
		0x00, 0x00, 0x04, 0x00, 0x23, 0x00, 0x00,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x30, 0x02, 0x00, 0x00,
		0x2c, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
		0x04, 0x00, 0x23, 0x00, 0x00, 0x16, 0x03, 0x01,
		0x02, 0xbe, 0x0b, 0x00, 0x02, 0xba, 0x00, 0x02,
		0xb7, 0x00, 0x02, 0xb4, 0x30, 0x82, 0x02, 0xb0,
		0x30, 0x82, 0x02, 0x19, 0xa0, 0x03, 0x02, 0x01,
		0x02, 0x02, 0x09, 0x00, 0x85, 0xb0, 0xbb, 0xa4,
		0x8a, 0x7f, 0xb8, 0xca, 0x30, 0x0d, 0x06, 0x09,
		0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
		0x05, 0x05, 0x00, 0x30, 0x45, 0x31, 0x0b, 0x30,
		0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
		0x41, 0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03,
		0x55, 0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d,
		0x65, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31,
		0x21, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a,
		0x13, 0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e,
		0x65, 0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69,
		0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c,
		0x74, 0x64, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x30,
		0x30, 0x34, 0x32, 0x34, 0x30, 0x39, 0x30, 0x39,
		0x33, 0x38, 0x5a, 0x17, 0x0d, 0x31, 0x31, 0x30,
		0x34, 0x32, 0x34, 0x30, 0x39, 0x30, 0x39, 0x33,
		0x38, 0x5a, 0x30, 0x45, 0x31, 0x0b, 0x30, 0x09,
		0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x41,
		0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55,
		0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d, 0x65,
		0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x21,
		0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13,
		0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65,
		0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69, 0x74,
		0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c, 0x74,
		0x64, 0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09,
		0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
		0x01, 0x05, 0x00, 0x03, 0x81, 0x8d, 0x00, 0x30,
		0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xbb, 0x79,
		0xd6, 0xf5, 0x17, 0xb5, 0xe5, 0xbf, 0x46, 0x10,
		0xd0, 0xdc, 0x69, 0xbe, 0xe6, 0x2b, 0x07, 0x43,
		0x5a, 0xd0, 0x03, 0x2d, 0x8a, 0x7a, 0x43, 0x85,
		0xb7, 0x14, 0x52, 0xe7, 0xa5, 0x65, 0x4c, 0x2c,
		0x78, 0xb8, 0x23, 0x8c, 0xb5, 0xb4, 0x82, 0xe5,
		0xde, 0x1f, 0x95, 0x3b, 0x7e, 0x62, 0xa5, 0x2c,
		0xa5, 0x33, 0xd6, 0xfe, 0x12, 0x5c, 0x7a, 0x56,
		0xfc, 0xf5, 0x06, 0xbf, 0xfa, 0x58, 0x7b, 0x26,
		0x3f, 0xb5, 0xcd, 0x04, 0xd3, 0xd0, 0xc9, 0x21,
		0x96, 0x4a, 0xc7, 0xf4, 0x54, 0x9f, 0x5a, 0xbf,
		0xef, 0x42, 0x71, 0x00, 0xfe, 0x18, 0x99, 0x07,
		0x7f, 0x7e, 0x88, 0x7d, 0x7d, 0xf1, 0x04, 0x39,
		0xc4, 0xa2, 0x2e, 0xdb, 0x51, 0xc9, 0x7c, 0xe3,
		0xc0, 0x4c, 0x3b, 0x32, 0x66, 0x01, 0xcf, 0xaf,
		0xb1, 0x1d, 0xb8, 0x71, 0x9a, 0x1d, 0xdb, 0xdb,
		0x89, 0x6b, 0xae, 0xda, 0x2d, 0x79, 0x02, 0x03,
		0x01, 0x00, 0x01, 0xa3, 0x81, 0xa7, 0x30, 0x81,
		0xa4, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e,
		0x04, 0x16, 0x04, 0x14, 0xb1, 0xad, 0xe2, 0x85,
		0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69, 0xce, 0x23,
		0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18, 0x88, 0x39,
		0x30, 0x75, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04,
		0x6e, 0x30, 0x6c, 0x80, 0x14, 0xb1, 0xad, 0xe2,
		0x85, 0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69, 0xce,
		0x23, 0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18, 0x88,
		0x39, 0xa1, 0x49, 0xa4, 0x47, 0x30, 0x45, 0x31,
		0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
		0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11,
		0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53,
		0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74,
		0x65, 0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55,
		0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65,
		0x72, 0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64,
		0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79,
		0x20, 0x4c, 0x74, 0x64, 0x82, 0x09, 0x00, 0x85,
		0xb0, 0xbb, 0xa4, 0x8a, 0x7f, 0xb8, 0xca, 0x30,
		0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x05,
		0x30, 0x03, 0x01, 0x01, 0xff, 0x30, 0x0d, 0x06,
		0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
		0x01, 0x05, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00,
		0x08, 0x6c, 0x45, 0x24, 0xc7, 0x6b, 0xb1, 0x59,
		0xab, 0x0c, 0x52, 0xcc, 0xf2, 0xb0, 0x14, 0xd7,
		0x87, 0x9d, 0x7a, 0x64, 0x75, 0xb5, 0x5a, 0x95,
		0x66, 0xe4, 0xc5, 0x2b, 0x8e, 0xae, 0x12, 0x66,
		0x1f, 0xeb, 0x4f, 0x38, 0xb3, 0x6e, 0x60, 0xd3,
		0x92, 0xfd, 0xf7, 0x41, 0x08, 0xb5, 0x25, 0x13,
		0xb1, 0x18, 0x7a, 0x24, 0xfb, 0x30, 0x1d, 0xba,
		0xed, 0x98, 0xb9, 0x17, 0xec, 0xe7, 0xd7, 0x31,
		0x59, 0xdb, 0x95, 0xd3, 0x1d, 0x78, 0xea, 0x50,
		0x56, 0x5c, 0xd5, 0x82, 0x5a, 0x2d, 0x5a, 0x5f,
		0x33, 0xc4, 0xb6, 0xd8, 0xc9, 0x75, 0x90, 0x96,
		0x8c, 0x0f, 0x52, 0x98, 0xb5, 0xcd, 0x98, 0x1f,
		0x89, 0x20, 0x5f, 0xf2, 0xa0, 0x1c, 0xa3, 0x1b,
		0x96, 0x94, 0xdd, 0xa9, 0xfd, 0x57, 0xe9, 0x70,
		0xe8, 0x26, 0x6d, 0x71, 0x99, 0x9b, 0x26, 0x6e,
		0x38, 0x50, 0x29, 0x6c, 0x90, 0xa7, 0xbd, 0xd9,
		0x16, 0x03, 0x01, 0x00, 0x04, 0x0e, 0x00, 0x00,
		0x00,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x86, 0x10, 0x00, 0x00,
		0x82, 0x00, 0x80, 0x68, 0x10, 0xdc, 0x80, 0xbc,
		0xb3, 0x5a, 0x10, 0x75, 0x89, 0xcc, 0xe5, 0x9f,
		0xbf, 0xe2, 0xce, 0xa4, 0x9f, 0x7f, 0x60, 0xc4,
		0xfe, 0x5c, 0xb5, 0x02, 0x2d, 0xa5, 0xa9, 0x1e,
		0x2c, 0x10, 0x79, 0x15, 0x0f, 0xed, 0x96, 0xb3,
		0xa8, 0x5e, 0x21, 0xbc, 0x5b, 0xdc, 0x58, 0x04,
		0x7d, 0x37, 0xdb, 0xa0, 0x31, 0xe8, 0x4f, 0x04,
		0xbc, 0x46, 0x7c, 0xdb, 0x2e, 0x93, 0x07, 0xaf,
		0xa6, 0x36, 0xd3, 0x39, 0x8d, 0x1d, 0x95, 0xa8,
		0x50, 0x4b, 0xc4, 0x2b, 0xde, 0xd7, 0x04, 0x6d,
		0x77, 0x6c, 0x4d, 0x70, 0x51, 0x88, 0x16, 0x31,
		0x40, 0xb5, 0xba, 0x90, 0x47, 0x64, 0x0c, 0x87,
		0xa5, 0x19, 0xf9, 0x89, 0x24, 0x3c, 0x5e, 0x4b,
		0xaa, 0xe0, 0x60, 0x47, 0x0f, 0x2e, 0xcc, 0xc2,
		0xd5, 0x21, 0xed, 0x72, 0xd0, 0xa9, 0xdd, 0x2a,
		0x2b, 0xef, 0x08, 0x3a, 0x65, 0xea, 0x8b, 0x52,
		0x77, 0x2d, 0xcc, 0x14, 0x03, 0x01, 0x00, 0x01,
		0x01, 0x16, 0x03, 0x01, 0x00, 0x24, 0xe2, 0x95,
		0x62, 0x3c, 0x18, 0xe5, 0xc7, 0x2c, 0xda, 0x16,
		0x9b, 0x28, 0x0d, 0xf7, 0x88, 0x7b, 0x5d, 0x33,
		0x55, 0x3b, 0x01, 0x73, 0xf2, 0xc6, 0x4e, 0x96,
		0x01, 0x01, 0x83, 0x65, 0xd4, 0xef, 0x12, 0x13,
		0x1d, 0x42,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x72, 0x04, 0x00, 0x00,
		0x6e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x65,
		0xe8, 0x4b, 0xd1, 0xef, 0xba, 0xfb, 0x41, 0x92,
		0x6d, 0x37, 0x5f, 0xf8, 0x7d, 0x90, 0x0f, 0x01,
		0xf8, 0x8c, 0xee, 0xbc, 0xd9, 0x0c, 0x97, 0x7e,
		0x23, 0x46, 0xe2, 0x6b, 0x52, 0xc6, 0xc6, 0x97,
		0x1d, 0xab, 0xde, 0xa0, 0x86, 0x94, 0xc8, 0x2e,
		0x8b, 0x2e, 0x42, 0x5f, 0xc2, 0x70, 0x35, 0xc9,
		0xee, 0x37, 0xeb, 0x70, 0xaa, 0x59, 0x23, 0x6c,
		0xc8, 0xc1, 0x84, 0x89, 0x39, 0x87, 0x73, 0x0a,
		0x7e, 0xba, 0xca, 0xed, 0x63, 0xba, 0x4e, 0x4f,
		0xf3, 0x31, 0x4b, 0xf0, 0xee, 0x91, 0xa5, 0xb4,
		0x62, 0x01, 0x9e, 0xbd, 0xbc, 0xb3, 0x35, 0x14,
		0x03, 0x01, 0x00, 0x01, 0x01, 0x16, 0x03, 0x01,
		0x00, 0x24, 0x3f, 0x66, 0xe4, 0x98, 0xc1, 0x3f,
		0xc6, 0x2c, 0x81, 0xfb, 0xa9, 0x9f, 0x27, 0xe9,
		0x63, 0x20, 0x1e, 0x0e, 0x4f, 0xfc, 0x5d, 0x12,
		0xee, 0x77, 0x73, 0xc6, 0x96, 0x51, 0xf2, 0x26,
		0x35, 0x3f, 0xce, 0x6a, 0xa9, 0xfd, 0x17, 0x03,
		0x01, 0x00, 0x21, 0x8d, 0xd5, 0x67, 0x60, 0x5d,
		0xa7, 0x93, 0xcc, 0x39, 0x78, 0x59, 0xab, 0xdb,
		0x10, 0x96, 0xf2, 0xad, 0xa2, 0x85, 0xe2, 0x93,
		0x43, 0x43, 0xcf, 0x82, 0xbd, 0x1f, 0xdc, 0x7a,
		0x72, 0xd6, 0x83, 0x3b, 0x15, 0x03, 0x01, 0x00,
		0x16, 0x89, 0x55, 0xf6, 0x42, 0x71, 0xa9, 0xe9,
		0x05, 0x68, 0xe8, 0xce, 0x0d, 0x21, 0xe9, 0xec,
		0xf2, 0x27, 0x67, 0xa7, 0x94, 0xf8, 0x34,
	},
}
var serverResumeTest = [][]byte{
	{
		0x16, 0x03, 0x01, 0x00, 0xc2, 0x01, 0x00, 0x00,
		0xbe, 0x03, 0x01, 0x50, 0x77, 0x3e, 0x4f, 0x1f,
		0x6f, 0xa5, 0x81, 0xeb, 0xb8, 0x80, 0x55, 0xa4,
		0x76, 0xc2, 0x7f, 0x27, 0xf2, 0xe7, 0xc9, 0x7a,
		0x01, 0x3c, 0xd8, 0xc1, 0xde, 0x99, 0x1f, 0x7c,
		0xab, 0x35, 0x98, 0x00, 0x00, 0x28, 0x00, 0x39,
		0x00, 0x38, 0x00, 0x35, 0x00, 0x16, 0x00, 0x13,
		0x00, 0x0a, 0x00, 0x33, 0x00, 0x32, 0x00, 0x2f,
		0x00, 0x05, 0x00, 0x04, 0x00, 0x15, 0x00, 0x12,
		0x00, 0x09, 0x00, 0x14, 0x00, 0x11, 0x00, 0x08,
		0x00, 0x06, 0x00, 0x03, 0x00, 0xff, 0x02, 0x01,
		0x00, 0x00, 0x6c, 0x00, 0x23, 0x00, 0x68, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x65,
		0xe8, 0x4b, 0xd1, 0xef, 0xba, 0xfb, 0x41, 0x92,
		0x6d, 0x37, 0x5f, 0xf8, 0x7d, 0x90, 0x0f, 0x01,
		0xf8, 0x8c, 0xee, 0xbc, 0xd9, 0x0c, 0x97, 0x7e,
		0x23, 0x46, 0xe2, 0x6b, 0x52, 0xc6, 0xc6, 0x97,
		0x1d, 0xab, 0xde, 0xa0, 0x86, 0x94, 0xc8, 0x2e,
		0x8b, 0x2e, 0x42, 0x5f, 0xc2, 0x70, 0x35, 0xc9,
		0xee, 0x37, 0xeb, 0x70, 0xaa, 0x59, 0x23, 0x6c,
		0xc8, 0xc1, 0x84, 0x89, 0x39, 0x87, 0x73, 0x0a,
		0x7e, 0xba, 0xca, 0xed, 0x63, 0xba, 0x4e, 0x4f,
		0xf3, 0x31, 0x4b, 0xf0, 0xee, 0x91, 0xa5, 0xb4,
		0x62, 0x01, 0x9e, 0xbd, 0xbc, 0xb3, 0x35,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x2a, 0x02, 0x00, 0x00,
		0x26, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x14,
		0x03, 0x01, 0x00, 0x01, 0x01, 0x16, 0x03, 0x01,
		0x00, 0x24, 0xc5, 0x35, 0x74, 0x19, 0x05, 0xc5,
		0x85, 0x68, 0x48, 0xe8, 0xb5, 0xe9, 0xaf, 0x78,
		0xbd, 0x35, 0x6f, 0xe9, 0x79, 0x34, 0x1b, 0xf0,
		0x35, 0xd4, 0x4e, 0x55, 0x2e, 0x3c, 0xd5, 0xaf,
		0xfc, 0xba, 0xf5, 0x1e, 0x83, 0x32,
	},
	{
		0x14, 0x03, 0x01, 0x00, 0x01, 0x01, 0x16, 0x03,
		0x01, 0x00, 0x24, 0x27, 0x28, 0x88, 0xe1, 0x7e,
		0x0d, 0x9c, 0x12, 0x50, 0xf6, 0x7a, 0xa7, 0x32,
		0x21, 0x68, 0xba, 0xd8, 0x0a, 0xdc, 0x39, 0xef,
		0x68, 0x95, 0x82, 0xae, 0xbd, 0x12, 0x79, 0xa1,
		0x99, 0xfd, 0xd0, 0x10, 0x8e, 0x4b, 0xd8,
	},
	{
		0x17, 0x03, 0x01, 0x00, 0x21, 0xc5, 0x7e, 0x0a,
		0x52, 0x6a, 0xb9, 0xaa, 0x1d, 0xae, 0x9e, 0x24,
		0x9c, 0x34, 0x1e, 0xdb, 0x50, 0x95, 0xee, 0x76,
		0xd7, 0x28, 0x88, 0x08, 0xe3, 0x2e, 0x58, 0xf7,
		0xdb, 0x34, 0x75, 0xa5, 0x7f, 0x9d, 0x15, 0x03,
		0x01, 0x00, 0x16, 0x2c, 0xc1, 0x29, 0x5f, 0x12,
		0x1d, 0x19, 0xab, 0xb3, 0xf4, 0x35, 0x1c, 0x62,
		0x6a, 0x80, 0x29, 0x0d, 0x0e, 0xef, 0x7d, 0x6e,
		0x50,
	},
}

var clientauthRSATests = []clientauthTest{
	// Server asks for cert with empty CA list, client doesn't give it.
	// go test -run "TestRunServer" -serve -clientauth 1
	{"RequestClientCert, none given", RequestClientCert, nil, [][]byte{
		{
			0x16, 0x03, 0x01, 0x01, 0x1e, 0x01, 0x00, 0x01,
			0x1a, 0x03, 0x03, 0x51, 0xe5, 0x6c, 0xb5, 0x5a,
			0xc2, 0xf5, 0xf0, 0x92, 0x94, 0x8a, 0x64, 0x18,
			0xa4, 0x2b, 0x82, 0x07, 0xbc, 0xd9, 0xd9, 0xf9,
			0x7b, 0xd2, 0xd0, 0xee, 0xa2, 0x70, 0x4e, 0x23,
			0x88, 0x7c, 0x95, 0x00, 0x00, 0x82, 0xc0, 0x30,
			0xc0, 0x2c, 0xc0, 0x28, 0xc0, 0x24, 0xc0, 0x14,
			0xc0, 0x0a, 0x00, 0xa3, 0x00, 0x9f, 0x00, 0x6b,
			0x00, 0x6a, 0x00, 0x39, 0x00, 0x38, 0xc0, 0x32,
			0xc0, 0x2e, 0xc0, 0x2a, 0xc0, 0x26, 0xc0, 0x0f,
			0xc0, 0x05, 0x00, 0x9d, 0x00, 0x3d, 0x00, 0x35,
			0xc0, 0x12, 0xc0, 0x08, 0x00, 0x16, 0x00, 0x13,
			0xc0, 0x0d, 0xc0, 0x03, 0x00, 0x0a, 0xc0, 0x2f,
			0xc0, 0x2b, 0xc0, 0x27, 0xc0, 0x23, 0xc0, 0x13,
			0xc0, 0x09, 0x00, 0xa2, 0x00, 0x9e, 0x00, 0x67,
			0x00, 0x40, 0x00, 0x33, 0x00, 0x32, 0xc0, 0x31,
			0xc0, 0x2d, 0xc0, 0x29, 0xc0, 0x25, 0xc0, 0x0e,
			0xc0, 0x04, 0x00, 0x9c, 0x00, 0x3c, 0x00, 0x2f,
			0x00, 0x07, 0xc0, 0x11, 0xc0, 0x07, 0xc0, 0x0c,
			0xc0, 0x02, 0x00, 0x05, 0x00, 0x04, 0x00, 0x15,
			0x00, 0x12, 0x00, 0x09, 0x00, 0x14, 0x00, 0x11,
			0x00, 0x08, 0x00, 0x06, 0x00, 0x03, 0x00, 0xff,
			0x01, 0x00, 0x00, 0x6f, 0x00, 0x0b, 0x00, 0x04,
			0x03, 0x00, 0x01, 0x02, 0x00, 0x0a, 0x00, 0x34,
			0x00, 0x32, 0x00, 0x0e, 0x00, 0x0d, 0x00, 0x19,
			0x00, 0x0b, 0x00, 0x0c, 0x00, 0x18, 0x00, 0x09,
			0x00, 0x0a, 0x00, 0x16, 0x00, 0x17, 0x00, 0x08,
			0x00, 0x06, 0x00, 0x07, 0x00, 0x14, 0x00, 0x15,
			0x00, 0x04, 0x00, 0x05, 0x00, 0x12, 0x00, 0x13,
			0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x0f,
			0x00, 0x10, 0x00, 0x11, 0x00, 0x23, 0x00, 0x00,
			0x00, 0x0d, 0x00, 0x22, 0x00, 0x20, 0x06, 0x01,
			0x06, 0x02, 0x06, 0x03, 0x05, 0x01, 0x05, 0x02,
			0x05, 0x03, 0x04, 0x01, 0x04, 0x02, 0x04, 0x03,
			0x03, 0x01, 0x03, 0x02, 0x03, 0x03, 0x02, 0x01,
			0x02, 0x02, 0x02, 0x03, 0x01, 0x01, 0x00, 0x0f,
			0x00, 0x01, 0x01,
		},
		{
			0x16, 0x03, 0x01, 0x00, 0x30, 0x02, 0x00, 0x00,
			0x2c, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
			0x04, 0x00, 0x23, 0x00, 0x00, 0x16, 0x03, 0x01,
			0x02, 0xbe, 0x0b, 0x00, 0x02, 0xba, 0x00, 0x02,
			0xb7, 0x00, 0x02, 0xb4, 0x30, 0x82, 0x02, 0xb0,
			0x30, 0x82, 0x02, 0x19, 0xa0, 0x03, 0x02, 0x01,
			0x02, 0x02, 0x09, 0x00, 0x85, 0xb0, 0xbb, 0xa4,
			0x8a, 0x7f, 0xb8, 0xca, 0x30, 0x0d, 0x06, 0x09,
			0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
			0x05, 0x05, 0x00, 0x30, 0x45, 0x31, 0x0b, 0x30,
			0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
			0x41, 0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03,
			0x55, 0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d,
			0x65, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31,
			0x21, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a,
			0x13, 0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e,
			0x65, 0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69,
			0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c,
			0x74, 0x64, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x30,
			0x30, 0x34, 0x32, 0x34, 0x30, 0x39, 0x30, 0x39,
			0x33, 0x38, 0x5a, 0x17, 0x0d, 0x31, 0x31, 0x30,
			0x34, 0x32, 0x34, 0x30, 0x39, 0x30, 0x39, 0x33,
			0x38, 0x5a, 0x30, 0x45, 0x31, 0x0b, 0x30, 0x09,
			0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x41,
			0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55,
			0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d, 0x65,
			0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x21,
			0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13,
			0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65,
			0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69, 0x74,
			0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c, 0x74,
			0x64, 0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09,
			0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
			0x01, 0x05, 0x00, 0x03, 0x81, 0x8d, 0x00, 0x30,
			0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xbb, 0x79,
			0xd6, 0xf5, 0x17, 0xb5, 0xe5, 0xbf, 0x46, 0x10,
			0xd0, 0xdc, 0x69, 0xbe, 0xe6, 0x2b, 0x07, 0x43,
			0x5a, 0xd0, 0x03, 0x2d, 0x8a, 0x7a, 0x43, 0x85,
			0xb7, 0x14, 0x52, 0xe7, 0xa5, 0x65, 0x4c, 0x2c,
			0x78, 0xb8, 0x23, 0x8c, 0xb5, 0xb4, 0x82, 0xe5,
			0xde, 0x1f, 0x95, 0x3b, 0x7e, 0x62, 0xa5, 0x2c,
			0xa5, 0x33, 0xd6, 0xfe, 0x12, 0x5c, 0x7a, 0x56,
			0xfc, 0xf5, 0x06, 0xbf, 0xfa, 0x58, 0x7b, 0x26,
			0x3f, 0xb5, 0xcd, 0x04, 0xd3, 0xd0, 0xc9, 0x21,
			0x96, 0x4a, 0xc7, 0xf4, 0x54, 0x9f, 0x5a, 0xbf,
			0xef, 0x42, 0x71, 0x00, 0xfe, 0x18, 0x99, 0x07,
			0x7f, 0x7e, 0x88, 0x7d, 0x7d, 0xf1, 0x04, 0x39,
			0xc4, 0xa2, 0x2e, 0xdb, 0x51, 0xc9, 0x7c, 0xe3,
			0xc0, 0x4c, 0x3b, 0x32, 0x66, 0x01, 0xcf, 0xaf,
			0xb1, 0x1d, 0xb8, 0x71, 0x9a, 0x1d, 0xdb, 0xdb,
			0x89, 0x6b, 0xae, 0xda, 0x2d, 0x79, 0x02, 0x03,
			0x01, 0x00, 0x01, 0xa3, 0x81, 0xa7, 0x30, 0x81,
			0xa4, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e,
			0x04, 0x16, 0x04, 0x14, 0xb1, 0xad, 0xe2, 0x85,
			0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69, 0xce, 0x23,
			0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18, 0x88, 0x39,
			0x30, 0x75, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04,
			0x6e, 0x30, 0x6c, 0x80, 0x14, 0xb1, 0xad, 0xe2,
			0x85, 0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69, 0xce,
			0x23, 0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18, 0x88,
			0x39, 0xa1, 0x49, 0xa4, 0x47, 0x30, 0x45, 0x31,
			0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
			0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11,
			0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53,
			0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74,
			0x65, 0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55,
			0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65,
			0x72, 0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64,
			0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79,
			0x20, 0x4c, 0x74, 0x64, 0x82, 0x09, 0x00, 0x85,
			0xb0, 0xbb, 0xa4, 0x8a, 0x7f, 0xb8, 0xca, 0x30,
			0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x05,
			0x30, 0x03, 0x01, 0x01, 0xff, 0x30, 0x0d, 0x06,
			0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
			0x01, 0x05, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00,
			0x08, 0x6c, 0x45, 0x24, 0xc7, 0x6b, 0xb1, 0x59,
			0xab, 0x0c, 0x52, 0xcc, 0xf2, 0xb0, 0x14, 0xd7,
			0x87, 0x9d, 0x7a, 0x64, 0x75, 0xb5, 0x5a, 0x95,
			0x66, 0xe4, 0xc5, 0x2b, 0x8e, 0xae, 0x12, 0x66,
			0x1f, 0xeb, 0x4f, 0x38, 0xb3, 0x6e, 0x60, 0xd3,
			0x92, 0xfd, 0xf7, 0x41, 0x08, 0xb5, 0x25, 0x13,
			0xb1, 0x18, 0x7a, 0x24, 0xfb, 0x30, 0x1d, 0xba,
			0xed, 0x98, 0xb9, 0x17, 0xec, 0xe7, 0xd7, 0x31,
			0x59, 0xdb, 0x95, 0xd3, 0x1d, 0x78, 0xea, 0x50,
			0x56, 0x5c, 0xd5, 0x82, 0x5a, 0x2d, 0x5a, 0x5f,
			0x33, 0xc4, 0xb6, 0xd8, 0xc9, 0x75, 0x90, 0x96,
			0x8c, 0x0f, 0x52, 0x98, 0xb5, 0xcd, 0x98, 0x1f,
			0x89, 0x20, 0x5f, 0xf2, 0xa0, 0x1c, 0xa3, 0x1b,
			0x96, 0x94, 0xdd, 0xa9, 0xfd, 0x57, 0xe9, 0x70,
			0xe8, 0x26, 0x6d, 0x71, 0x99, 0x9b, 0x26, 0x6e,
			0x38, 0x50, 0x29, 0x6c, 0x90, 0xa7, 0xbd, 0xd9,
			0x16, 0x03, 0x01, 0x00, 0x09, 0x0d, 0x00, 0x00,
			0x05, 0x02, 0x01, 0x40, 0x00, 0x00, 0x16, 0x03,
			0x01, 0x00, 0x04, 0x0e, 0x00, 0x00, 0x00,
		},
		{
			0x16, 0x03, 0x01, 0x00, 0x07, 0x0b, 0x00, 0x00,
			0x03, 0x00, 0x00, 0x00, 0x16, 0x03, 0x01, 0x00,
			0x86, 0x10, 0x00, 0x00, 0x82, 0x00, 0x80, 0x36,
			0xfc, 0xd8, 0xc8, 0xa2, 0x67, 0xc8, 0xc6, 0xf4,
			0x28, 0x70, 0xe1, 0x5a, 0x02, 0x8f, 0xef, 0x42,
			0xe0, 0xd3, 0xb8, 0xd6, 0x6b, 0xe4, 0xee, 0x5c,
			0xcf, 0x42, 0xc4, 0xfa, 0xcd, 0x0f, 0xfe, 0xf4,
			0x76, 0x76, 0x47, 0x73, 0xa8, 0x72, 0x8f, 0xa2,
			0x56, 0x81, 0x83, 0xb8, 0x84, 0x72, 0x67, 0xdd,
			0xbe, 0x05, 0x4b, 0x84, 0xd9, 0xd2, 0xb6, 0xc2,
			0xe7, 0x20, 0xac, 0x1f, 0x46, 0x9d, 0x05, 0x47,
			0x8e, 0x89, 0xc0, 0x42, 0x57, 0x4a, 0xa2, 0x98,
			0xe5, 0x39, 0x4f, 0xc4, 0x27, 0x6d, 0x43, 0xa8,
			0x83, 0x76, 0xe6, 0xad, 0xe3, 0x17, 0x68, 0x31,
			0xcb, 0x7e, 0xfc, 0xe7, 0x4b, 0x76, 0x3d, 0x3c,
			0xfa, 0x77, 0x65, 0xc9, 0x4c, 0x5b, 0xce, 0x5e,
			0xf7, 0x8b, 0xa8, 0xa6, 0xdd, 0xb2, 0xef, 0x0b,
			0x46, 0x83, 0xdf, 0x0a, 0x8c, 0x22, 0x12, 0x6e,
			0xe1, 0x45, 0x54, 0x88, 0xd1, 0xe8, 0xd2, 0x14,
			0x03, 0x01, 0x00, 0x01, 0x01, 0x16, 0x03, 0x01,
			0x00, 0x24, 0x30, 0x8c, 0x7d, 0x40, 0xfc, 0x5e,
			0x80, 0x9c, 0xc4, 0x7c, 0x62, 0x01, 0xa1, 0x37,
			0xcf, 0x1a, 0x75, 0x28, 0x8d, 0xeb, 0x63, 0xcc,
			0x02, 0xa6, 0x66, 0xdf, 0x36, 0x01, 0xb3, 0x9d,
			0x38, 0x42, 0x16, 0x91, 0xf0, 0x02,
		},
		{
			0x16, 0x03, 0x01, 0x00, 0x72, 0x04, 0x00, 0x00,
			0x6e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x65,
			0xe8, 0x4b, 0xd1, 0xef, 0xba, 0x96, 0x9a, 0x2a,
			0x6c, 0x8c, 0x7e, 0x38, 0x10, 0x46, 0x86, 0x1d,
			0x19, 0x1d, 0x62, 0x29, 0x3f, 0x58, 0xfb, 0x6d,
			0x89, 0xd2, 0x81, 0x9a, 0x1c, 0xb3, 0x58, 0xb3,
			0x19, 0x39, 0x17, 0x47, 0x49, 0xc9, 0xfe, 0x4a,
			0x7a, 0x32, 0xac, 0x2c, 0x43, 0xf9, 0xa9, 0xea,
			0xec, 0x51, 0x46, 0xf1, 0xb8, 0x59, 0x23, 0x70,
			0xce, 0x7c, 0xb9, 0x47, 0x70, 0xa3, 0xc9, 0xae,
			0x47, 0x7b, 0x7e, 0xc7, 0xcf, 0x76, 0x12, 0x76,
			0x18, 0x90, 0x12, 0xcd, 0xf3, 0xd4, 0x27, 0x81,
			0xfc, 0x46, 0x03, 0x3e, 0x05, 0x87, 0x6f, 0x14,
			0x03, 0x01, 0x00, 0x01, 0x01, 0x16, 0x03, 0x01,
			0x00, 0x24, 0xc3, 0xa0, 0x29, 0xb1, 0x52, 0x82,
			0xef, 0x85, 0xa1, 0x64, 0x0f, 0xe4, 0xa3, 0xfb,
			0xa7, 0x1d, 0x22, 0x4c, 0xcb, 0xd6, 0x5b, 0x18,
			0x61, 0xc7, 0x7c, 0xf2, 0x67, 0x4a, 0xc7, 0x11,
			0x9d, 0x8e, 0x0e, 0x15, 0x22, 0xcf, 0x17, 0x03,
			0x01, 0x00, 0x21, 0xfd, 0xbb, 0xf1, 0xa9, 0x7c,
			0xbf, 0x92, 0xb3, 0xfa, 0x2c, 0x08, 0x6f, 0x22,
			0x78, 0x80, 0xf2, 0x2e, 0x86, 0x26, 0x21, 0x36,
			0x3f, 0x32, 0xdf, 0xb6, 0x47, 0xa5, 0xf8, 0x27,
			0xc1, 0xe9, 0x53, 0x90, 0x15, 0x03, 0x01, 0x00,
			0x16, 0xfe, 0xef, 0x2e, 0xa0, 0x5d, 0xe0, 0xce,
			0x94, 0x20, 0x56, 0x61, 0x6e, 0xe5, 0x62, 0xce,
			0x27, 0x57, 0x3e, 0x30, 0x32, 0x77, 0x53,
		},
	}},

	// Server asks for cert with empty CA list, client gives one
	// go test -run "TestRunServer" -serve -clientauth 1
	{"RequestClientCert, client gives it", RequestClientCert, []*x509.Certificate{clientCertificate}, [][]byte{
		{
			0x16, 0x03, 0x01, 0x01, 0x1e, 0x01, 0x00, 0x01,
			0x1a, 0x03, 0x03, 0x51, 0xe5, 0x74, 0x0e, 0x95,
			0x6f, 0x4f, 0x4a, 0xbf, 0xb7, 0xc0, 0x6c, 0xac,
			0xd9, 0xfe, 0x7d, 0xd0, 0x51, 0x19, 0x62, 0x62,
			0x1c, 0x6e, 0x57, 0x77, 0xd2, 0x31, 0xaf, 0x88,
			0xb9, 0xc0, 0x1d, 0x00, 0x00, 0x82, 0xc0, 0x30,
			0xc0, 0x2c, 0xc0, 0x28, 0xc0, 0x24, 0xc0, 0x14,
			0xc0, 0x0a, 0x00, 0xa3, 0x00, 0x9f, 0x00, 0x6b,
			0x00, 0x6a, 0x00, 0x39, 0x00, 0x38, 0xc0, 0x32,
			0xc0, 0x2e, 0xc0, 0x2a, 0xc0, 0x26, 0xc0, 0x0f,
			0xc0, 0x05, 0x00, 0x9d, 0x00, 0x3d, 0x00, 0x35,
			0xc0, 0x12, 0xc0, 0x08, 0x00, 0x16, 0x00, 0x13,
			0xc0, 0x0d, 0xc0, 0x03, 0x00, 0x0a, 0xc0, 0x2f,
			0xc0, 0x2b, 0xc0, 0x27, 0xc0, 0x23, 0xc0, 0x13,
			0xc0, 0x09, 0x00, 0xa2, 0x00, 0x9e, 0x00, 0x67,
			0x00, 0x40, 0x00, 0x33, 0x00, 0x32, 0xc0, 0x31,
			0xc0, 0x2d, 0xc0, 0x29, 0xc0, 0x25, 0xc0, 0x0e,
			0xc0, 0x04, 0x00, 0x9c, 0x00, 0x3c, 0x00, 0x2f,
			0x00, 0x07, 0xc0, 0x11, 0xc0, 0x07, 0xc0, 0x0c,
			0xc0, 0x02, 0x00, 0x05, 0x00, 0x04, 0x00, 0x15,
			0x00, 0x12, 0x00, 0x09, 0x00, 0x14, 0x00, 0x11,
			0x00, 0x08, 0x00, 0x06, 0x00, 0x03, 0x00, 0xff,
			0x01, 0x00, 0x00, 0x6f, 0x00, 0x0b, 0x00, 0x04,
			0x03, 0x00, 0x01, 0x02, 0x00, 0x0a, 0x00, 0x34,
			0x00, 0x32, 0x00, 0x0e, 0x00, 0x0d, 0x00, 0x19,
			0x00, 0x0b, 0x00, 0x0c, 0x00, 0x18, 0x00, 0x09,
			0x00, 0x0a, 0x00, 0x16, 0x00, 0x17, 0x00, 0x08,
			0x00, 0x06, 0x00, 0x07, 0x00, 0x14, 0x00, 0x15,
			0x00, 0x04, 0x00, 0x05, 0x00, 0x12, 0x00, 0x13,
			0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x0f,
			0x00, 0x10, 0x00, 0x11, 0x00, 0x23, 0x00, 0x00,
			0x00, 0x0d, 0x00, 0x22, 0x00, 0x20, 0x06, 0x01,
			0x06, 0x02, 0x06, 0x03, 0x05, 0x01, 0x05, 0x02,
			0x05, 0x03, 0x04, 0x01, 0x04, 0x02, 0x04, 0x03,
			0x03, 0x01, 0x03, 0x02, 0x03, 0x03, 0x02, 0x01,
			0x02, 0x02, 0x02, 0x03, 0x01, 0x01, 0x00, 0x0f,
			0x00, 0x01, 0x01,
		},
		{
			0x16, 0x03, 0x01, 0x00, 0x30, 0x02, 0x00, 0x00,
			0x2c, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
			0x04, 0x00, 0x23, 0x00, 0x00, 0x16, 0x03, 0x01,
			0x02, 0xbe, 0x0b, 0x00, 0x02, 0xba, 0x00, 0x02,
			0xb7, 0x00, 0x02, 0xb4, 0x30, 0x82, 0x02, 0xb0,
			0x30, 0x82, 0x02, 0x19, 0xa0, 0x03, 0x02, 0x01,
			0x02, 0x02, 0x09, 0x00, 0x85, 0xb0, 0xbb, 0xa4,
			0x8a, 0x7f, 0xb8, 0xca, 0x30, 0x0d, 0x06, 0x09,
			0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
			0x05, 0x05, 0x00, 0x30, 0x45, 0x31, 0x0b, 0x30,
			0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
			0x41, 0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03,
			0x55, 0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d,
			0x65, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31,
			0x21, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a,
			0x13, 0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e,
			0x65, 0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69,
			0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c,
			0x74, 0x64, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x30,
			0x30, 0x34, 0x32, 0x34, 0x30, 0x39, 0x30, 0x39,
			0x33, 0x38, 0x5a, 0x17, 0x0d, 0x31, 0x31, 0x30,
			0x34, 0x32, 0x34, 0x30, 0x39, 0x30, 0x39, 0x33,
			0x38, 0x5a, 0x30, 0x45, 0x31, 0x0b, 0x30, 0x09,
			0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x41,
			0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55,
			0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d, 0x65,
			0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x21,
			0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13,
			0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65,
			0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69, 0x74,
			0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c, 0x74,
			0x64, 0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09,
			0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
			0x01, 0x05, 0x00, 0x03, 0x81, 0x8d, 0x00, 0x30,
			0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xbb, 0x79,
			0xd6, 0xf5, 0x17, 0xb5, 0xe5, 0xbf, 0x46, 0x10,
			0xd0, 0xdc, 0x69, 0xbe, 0xe6, 0x2b, 0x07, 0x43,
			0x5a, 0xd0, 0x03, 0x2d, 0x8a, 0x7a, 0x43, 0x85,
			0xb7, 0x14, 0x52, 0xe7, 0xa5, 0x65, 0x4c, 0x2c,
			0x78, 0xb8, 0x23, 0x8c, 0xb5, 0xb4, 0x82, 0xe5,
			0xde, 0x1f, 0x95, 0x3b, 0x7e, 0x62, 0xa5, 0x2c,
			0xa5, 0x33, 0xd6, 0xfe, 0x12, 0x5c, 0x7a, 0x56,
			0xfc, 0xf5, 0x06, 0xbf, 0xfa, 0x58, 0x7b, 0x26,
			0x3f, 0xb5, 0xcd, 0x04, 0xd3, 0xd0, 0xc9, 0x21,
			0x96, 0x4a, 0xc7, 0xf4, 0x54, 0x9f, 0x5a, 0xbf,
			0xef, 0x42, 0x71, 0x00, 0xfe, 0x18, 0x99, 0x07,
			0x7f, 0x7e, 0x88, 0x7d, 0x7d, 0xf1, 0x04, 0x39,
			0xc4, 0xa2, 0x2e, 0xdb, 0x51, 0xc9, 0x7c, 0xe3,
			0xc0, 0x4c, 0x3b, 0x32, 0x66, 0x01, 0xcf, 0xaf,
			0xb1, 0x1d, 0xb8, 0x71, 0x9a, 0x1d, 0xdb, 0xdb,
			0x89, 0x6b, 0xae, 0xda, 0x2d, 0x79, 0x02, 0x03,
			0x01, 0x00, 0x01, 0xa3, 0x81, 0xa7, 0x30, 0x81,
			0xa4, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e,
			0x04, 0x16, 0x04, 0x14, 0xb1, 0xad, 0xe2, 0x85,
			0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69, 0xce, 0x23,
			0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18, 0x88, 0x39,
			0x30, 0x75, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04,
			0x6e, 0x30, 0x6c, 0x80, 0x14, 0xb1, 0xad, 0xe2,
			0x85, 0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69, 0xce,
			0x23, 0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18, 0x88,
			0x39, 0xa1, 0x49, 0xa4, 0x47, 0x30, 0x45, 0x31,
			0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
			0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11,
			0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53,
			0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74,
			0x65, 0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55,
			0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65,
			0x72, 0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64,
			0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79,
			0x20, 0x4c, 0x74, 0x64, 0x82, 0x09, 0x00, 0x85,
			0xb0, 0xbb, 0xa4, 0x8a, 0x7f, 0xb8, 0xca, 0x30,
			0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x05,
			0x30, 0x03, 0x01, 0x01, 0xff, 0x30, 0x0d, 0x06,
			0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
			0x01, 0x05, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00,
			0x08, 0x6c, 0x45, 0x24, 0xc7, 0x6b, 0xb1, 0x59,
			0xab, 0x0c, 0x52, 0xcc, 0xf2, 0xb0, 0x14, 0xd7,
			0x87, 0x9d, 0x7a, 0x64, 0x75, 0xb5, 0x5a, 0x95,
			0x66, 0xe4, 0xc5, 0x2b, 0x8e, 0xae, 0x12, 0x66,
			0x1f, 0xeb, 0x4f, 0x38, 0xb3, 0x6e, 0x60, 0xd3,
			0x92, 0xfd, 0xf7, 0x41, 0x08, 0xb5, 0x25, 0x13,
			0xb1, 0x18, 0x7a, 0x24, 0xfb, 0x30, 0x1d, 0xba,
			0xed, 0x98, 0xb9, 0x17, 0xec, 0xe7, 0xd7, 0x31,
			0x59, 0xdb, 0x95, 0xd3, 0x1d, 0x78, 0xea, 0x50,
			0x56, 0x5c, 0xd5, 0x82, 0x5a, 0x2d, 0x5a, 0x5f,
			0x33, 0xc4, 0xb6, 0xd8, 0xc9, 0x75, 0x90, 0x96,
			0x8c, 0x0f, 0x52, 0x98, 0xb5, 0xcd, 0x98, 0x1f,
			0x89, 0x20, 0x5f, 0xf2, 0xa0, 0x1c, 0xa3, 0x1b,
			0x96, 0x94, 0xdd, 0xa9, 0xfd, 0x57, 0xe9, 0x70,
			0xe8, 0x26, 0x6d, 0x71, 0x99, 0x9b, 0x26, 0x6e,
			0x38, 0x50, 0x29, 0x6c, 0x90, 0xa7, 0xbd, 0xd9,
			0x16, 0x03, 0x01, 0x00, 0x09, 0x0d, 0x00, 0x00,
			0x05, 0x02, 0x01, 0x40, 0x00, 0x00, 0x16, 0x03,
			0x01, 0x00, 0x04, 0x0e, 0x00, 0x00, 0x00,
		},
		{
			0x16, 0x03, 0x01, 0x01, 0xfb, 0x0b, 0x00, 0x01,
			0xf7, 0x00, 0x01, 0xf4, 0x00, 0x01, 0xf1, 0x30,
			0x82, 0x01, 0xed, 0x30, 0x82, 0x01, 0x58, 0xa0,
			0x03, 0x02, 0x01, 0x02, 0x02, 0x01, 0x00, 0x30,
			0x0b, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
			0x0d, 0x01, 0x01, 0x05, 0x30, 0x26, 0x31, 0x10,
			0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13,
			0x07, 0x41, 0x63, 0x6d, 0x65, 0x20, 0x43, 0x6f,
			0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04,
			0x03, 0x13, 0x09, 0x31, 0x32, 0x37, 0x2e, 0x30,
			0x2e, 0x30, 0x2e, 0x31, 0x30, 0x1e, 0x17, 0x0d,
			0x31, 0x31, 0x31, 0x32, 0x30, 0x38, 0x30, 0x37,
			0x35, 0x35, 0x31, 0x32, 0x5a, 0x17, 0x0d, 0x31,
			0x32, 0x31, 0x32, 0x30, 0x37, 0x30, 0x38, 0x30,
			0x30, 0x31, 0x32, 0x5a, 0x30, 0x26, 0x31, 0x10,
			0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13,
			0x07, 0x41, 0x63, 0x6d, 0x65, 0x20, 0x43, 0x6f,
			0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04,
			0x03, 0x13, 0x09, 0x31, 0x32, 0x37, 0x2e, 0x30,
			0x2e, 0x30, 0x2e, 0x31, 0x30, 0x81, 0x9c, 0x30,
			0x0b, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
			0x0d, 0x01, 0x01, 0x01, 0x03, 0x81, 0x8c, 0x00,
			0x30, 0x81, 0x88, 0x02, 0x81, 0x80, 0x4e, 0xd0,
			0x7b, 0x31, 0xe3, 0x82, 0x64, 0xd9, 0x59, 0xc0,
			0xc2, 0x87, 0xa4, 0x5e, 0x1e, 0x8b, 0x73, 0x33,
			0xc7, 0x63, 0x53, 0xdf, 0x66, 0x92, 0x06, 0x84,
			0xf6, 0x64, 0xd5, 0x8f, 0xe4, 0x36, 0xa7, 0x1d,
			0x2b, 0xe8, 0xb3, 0x20, 0x36, 0x45, 0x23, 0xb5,
			0xe3, 0x95, 0xae, 0xed, 0xe0, 0xf5, 0x20, 0x9c,
			0x8d, 0x95, 0xdf, 0x7f, 0x5a, 0x12, 0xef, 0x87,
			0xe4, 0x5b, 0x68, 0xe4, 0xe9, 0x0e, 0x74, 0xec,
			0x04, 0x8a, 0x7f, 0xde, 0x93, 0x27, 0xc4, 0x01,
			0x19, 0x7a, 0xbd, 0xf2, 0xdc, 0x3d, 0x14, 0xab,
			0xd0, 0x54, 0xca, 0x21, 0x0c, 0xd0, 0x4d, 0x6e,
			0x87, 0x2e, 0x5c, 0xc5, 0xd2, 0xbb, 0x4d, 0x4b,
			0x4f, 0xce, 0xb6, 0x2c, 0xf7, 0x7e, 0x88, 0xec,
			0x7c, 0xd7, 0x02, 0x91, 0x74, 0xa6, 0x1e, 0x0c,
			0x1a, 0xda, 0xe3, 0x4a, 0x5a, 0x2e, 0xde, 0x13,
			0x9c, 0x4c, 0x40, 0x88, 0x59, 0x93, 0x02, 0x03,
			0x01, 0x00, 0x01, 0xa3, 0x32, 0x30, 0x30, 0x30,
			0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
			0xff, 0x04, 0x04, 0x03, 0x02, 0x00, 0xa0, 0x30,
			0x0d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x06,
			0x04, 0x04, 0x01, 0x02, 0x03, 0x04, 0x30, 0x0f,
			0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x08, 0x30,
			0x06, 0x80, 0x04, 0x01, 0x02, 0x03, 0x04, 0x30,
			0x0b, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
			0x0d, 0x01, 0x01, 0x05, 0x03, 0x81, 0x81, 0x00,
			0x36, 0x1f, 0xb3, 0x7a, 0x0c, 0x75, 0xc9, 0x6e,
			0x37, 0x46, 0x61, 0x2b, 0xd5, 0xbd, 0xc0, 0xa7,
			0x4b, 0xcc, 0x46, 0x9a, 0x81, 0x58, 0x7c, 0x85,
			0x79, 0x29, 0xc8, 0xc8, 0xc6, 0x67, 0xdd, 0x32,
			0x56, 0x45, 0x2b, 0x75, 0xb6, 0xe9, 0x24, 0xa9,
			0x50, 0x9a, 0xbe, 0x1f, 0x5a, 0xfa, 0x1a, 0x15,
			0xd9, 0xcc, 0x55, 0x95, 0x72, 0x16, 0x83, 0xb9,
			0xc2, 0xb6, 0x8f, 0xfd, 0x88, 0x8c, 0x38, 0x84,
			0x1d, 0xab, 0x5d, 0x92, 0x31, 0x13, 0x4f, 0xfd,
			0x83, 0x3b, 0xc6, 0x9d, 0xf1, 0x11, 0x62, 0xb6,
			0x8b, 0xec, 0xab, 0x67, 0xbe, 0xc8, 0x64, 0xb0,
			0x11, 0x50, 0x46, 0x58, 0x17, 0x6b, 0x99, 0x1c,
			0xd3, 0x1d, 0xfc, 0x06, 0xf1, 0x0e, 0xe5, 0x96,
			0xa8, 0x0c, 0xf9, 0x78, 0x20, 0xb7, 0x44, 0x18,
			0x51, 0x8d, 0x10, 0x7e, 0x4f, 0x94, 0x67, 0xdf,
			0xa3, 0x4e, 0x70, 0x73, 0x8e, 0x90, 0x91, 0x85,
			0x16, 0x03, 0x01, 0x00, 0x86, 0x10, 0x00, 0x00,
			0x82, 0x00, 0x80, 0x0a, 0x4e, 0x89, 0xdf, 0x3a,
			0x3f, 0xf0, 0x4f, 0xef, 0x1a, 0x90, 0xd4, 0x3c,
			0xaf, 0x10, 0x57, 0xb0, 0xa1, 0x5f, 0xcd, 0x62,
			0x01, 0xe9, 0x0c, 0x36, 0x42, 0xfd, 0xaf, 0x23,
			0xf9, 0x14, 0xa6, 0x72, 0x26, 0x4e, 0x01, 0xdb,
			0xac, 0xb7, 0x4c, 0xe6, 0xa9, 0x52, 0xe2, 0xec,
			0x26, 0x8c, 0x7a, 0x64, 0xf8, 0x0b, 0x4c, 0x2f,
			0xa9, 0xcb, 0x75, 0xaf, 0x60, 0xd4, 0xb4, 0xe6,
			0xe8, 0xdb, 0x78, 0x78, 0x85, 0xf6, 0x0c, 0x95,
			0xcc, 0xb6, 0x55, 0xb9, 0xba, 0x9e, 0x91, 0xbc,
			0x66, 0xdb, 0x1e, 0x28, 0xab, 0x73, 0xce, 0x8b,
			0xd0, 0xd3, 0xe8, 0xbc, 0xd0, 0x21, 0x28, 0xbd,
			0xfb, 0x74, 0x64, 0xde, 0x3b, 0x3b, 0xd3, 0x4c,
			0x32, 0x40, 0x82, 0xba, 0x91, 0x1e, 0xe8, 0x47,
			0xc2, 0x09, 0xb7, 0x16, 0xaa, 0x25, 0xa9, 0x3c,
			0x6c, 0xa7, 0xf8, 0xc9, 0x54, 0x84, 0xc6, 0xf7,
			0x56, 0x05, 0xa4, 0x16, 0x03, 0x01, 0x00, 0x86,
			0x0f, 0x00, 0x00, 0x82, 0x00, 0x80, 0x4b, 0xab,
			0xda, 0xac, 0x2a, 0xb3, 0xe6, 0x34, 0x55, 0xcd,
			0xf2, 0x4b, 0x67, 0xe3, 0xd3, 0xff, 0xa3, 0xf4,
			0x79, 0x82, 0x01, 0x47, 0x8a, 0xe3, 0x9f, 0x89,
			0x70, 0xbe, 0x24, 0x24, 0xb7, 0x69, 0x60, 0xed,
			0x55, 0xa0, 0xca, 0x72, 0xb6, 0x4a, 0xbc, 0x1d,
			0xe2, 0x3f, 0xb5, 0x31, 0xda, 0x02, 0xf6, 0x37,
			0x51, 0xf8, 0x4c, 0x88, 0x2e, 0xb3, 0x8a, 0xe8,
			0x7b, 0x4a, 0x90, 0x36, 0xe4, 0xa6, 0x31, 0x95,
			0x8b, 0xa0, 0xc6, 0x91, 0x12, 0xb9, 0x35, 0x4e,
			0x72, 0xeb, 0x5c, 0xa2, 0xe8, 0x4c, 0x68, 0xf9,
			0x69, 0xfa, 0x70, 0x60, 0x6c, 0x7f, 0x32, 0x99,
			0xf1, 0xc3, 0x2d, 0xb4, 0x59, 0x58, 0x87, 0xaf,
			0x67, 0x62, 0x90, 0xe7, 0x8d, 0xd0, 0xa3, 0x77,
			0x33, 0xc2, 0x9b, 0xd5, 0x9c, 0xc7, 0xea, 0x25,
			0x98, 0x76, 0x9c, 0xe0, 0x6a, 0x03, 0x3a, 0x10,
			0xfd, 0x10, 0x3d, 0x55, 0x53, 0xa0, 0x14, 0x03,
			0x01, 0x00, 0x01, 0x01, 0x16, 0x03, 0x01, 0x00,
			0x24, 0xd5, 0x12, 0xfc, 0xb9, 0x5a, 0xe3, 0x27,
			0x01, 0xbe, 0xc3, 0x77, 0x17, 0x1a, 0xbb, 0x4f,
			0xae, 0xd5, 0xa7, 0xee, 0x56, 0x61, 0x0d, 0x40,
			0xf4, 0xa4, 0xb5, 0xcc, 0x76, 0xfd, 0xbd, 0x13,
			0x04, 0xe1, 0xb8, 0xc7, 0x36,
		},
		{
			0x16, 0x03, 0x01, 0x02, 0x67, 0x04, 0x00, 0x02,
			0x63, 0x00, 0x00, 0x00, 0x00, 0x02, 0x5d, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x65,
			0xe8, 0x4b, 0xd1, 0xef, 0xba, 0x1f, 0xe2, 0x69,
			0x07, 0x7f, 0x85, 0x2d, 0x4e, 0x2a, 0x2e, 0xbd,
			0x05, 0xe9, 0xc1, 0x6c, 0x9e, 0xbf, 0x47, 0x18,
			0x91, 0x77, 0xf7, 0xe8, 0xb6, 0x27, 0x37, 0xa6,
			0x6b, 0x87, 0x29, 0xbb, 0x3b, 0xe5, 0x68, 0x62,
			0x04, 0x3e, 0xad, 0x4d, 0xff, 0xad, 0xf1, 0x22,
			0x87, 0x8d, 0xf6, 0x04, 0x3b, 0x59, 0x22, 0xf7,
			0xfd, 0x88, 0x0e, 0xa4, 0x09, 0xc0, 0x0d, 0x10,
			0x80, 0x10, 0x79, 0xee, 0x70, 0x96, 0xdb, 0x22,
			0x8b, 0xb7, 0xac, 0xe0, 0x98, 0xad, 0xe9, 0xe3,
			0xcb, 0xea, 0x9f, 0xe6, 0x83, 0x28, 0x7c, 0x7e,
			0x4e, 0x9a, 0x8d, 0xd9, 0xf3, 0x86, 0xf4, 0x89,
			0x8b, 0x79, 0x8f, 0xbb, 0xe9, 0x74, 0x02, 0x02,
			0x14, 0x04, 0xea, 0xba, 0x16, 0x10, 0xa1, 0x85,
			0xbe, 0x4e, 0x4e, 0x92, 0xc5, 0x83, 0xf6, 0x1e,
			0x1f, 0xd4, 0x25, 0xc2, 0xc2, 0xb9, 0xce, 0x33,
			0x63, 0x66, 0x79, 0x1f, 0x54, 0x35, 0xc1, 0xe8,
			0x89, 0x34, 0x78, 0x94, 0x36, 0x14, 0xef, 0x01,
			0x1f, 0xf1, 0xbd, 0x77, 0x2c, 0x4d, 0xac, 0x5c,
			0x5c, 0x4a, 0xc6, 0xed, 0xd8, 0x0e, 0x72, 0x84,
			0x83, 0xdc, 0x56, 0x84, 0xc8, 0xf3, 0x89, 0x56,
			0xfd, 0x89, 0xc1, 0xc9, 0x9a, 0x29, 0x91, 0x7e,
			0x19, 0xe9, 0x8b, 0x5b, 0x11, 0x15, 0x4e, 0x6c,
			0xf4, 0x89, 0xe7, 0x6d, 0x68, 0x1e, 0xf9, 0x6c,
			0x23, 0x72, 0x05, 0x68, 0x82, 0x60, 0x84, 0x1f,
			0x83, 0x20, 0x09, 0x86, 0x10, 0x81, 0xec, 0xec,
			0xdc, 0x25, 0x53, 0x20, 0xfa, 0xa9, 0x41, 0x64,
			0xd6, 0x20, 0xf3, 0xf4, 0x52, 0xf2, 0x80, 0x62,
			0x83, 0xc9, 0x23, 0x66, 0x44, 0x95, 0x5a, 0x99,
			0x8a, 0xe1, 0x26, 0x63, 0xc1, 0x8b, 0x31, 0xf9,
			0x21, 0x06, 0x77, 0x04, 0x27, 0xf2, 0x0c, 0x63,
			0x83, 0x45, 0xa0, 0xa9, 0x7b, 0xcf, 0xdf, 0xd7,
			0x56, 0x75, 0xbc, 0xdd, 0x95, 0x36, 0xb1, 0x75,
			0x39, 0x05, 0x00, 0x3c, 0x8a, 0x79, 0xd6, 0xe9,
			0xf0, 0x4b, 0xdc, 0x51, 0x6b, 0x01, 0x94, 0x16,
			0x87, 0x12, 0x92, 0x6c, 0x07, 0xc1, 0xf5, 0x58,
			0xb7, 0x2a, 0x81, 0xf5, 0xa0, 0x37, 0x8b, 0xa6,
			0x22, 0xfe, 0x28, 0x0a, 0x7e, 0x68, 0xe2, 0xda,
			0x6c, 0x53, 0xee, 0x0e, 0x8d, 0x2d, 0x8b, 0x0b,
			0xda, 0xf8, 0x99, 0x3e, 0x0e, 0xed, 0x9f, 0xc1,
			0x2b, 0xf6, 0xfe, 0xe9, 0x52, 0x38, 0x7b, 0x83,
			0x9a, 0x50, 0xa6, 0xd7, 0x49, 0x83, 0x43, 0x7e,
			0x82, 0xec, 0xc7, 0x09, 0x3d, 0x3d, 0xb1, 0xee,
			0xe8, 0xc5, 0x6a, 0xc3, 0x3d, 0x4b, 0x4c, 0x6a,
			0xbb, 0x0b, 0x2c, 0x24, 0x2e, 0xdb, 0x7d, 0x57,
			0x87, 0xb4, 0x80, 0xa5, 0xae, 0xff, 0x54, 0xa8,
			0xa5, 0x27, 0x69, 0x95, 0xc8, 0xe7, 0x79, 0xc7,
			0x89, 0x2a, 0x73, 0x49, 0xcb, 0xf5, 0xc5, 0xbc,
			0x4a, 0xe0, 0x73, 0xa9, 0xbc, 0x88, 0x64, 0x96,
			0x98, 0xa5, 0x1e, 0xe3, 0x43, 0xc1, 0x7d, 0x78,
			0xc7, 0x94, 0x72, 0xd4, 0x2c, 0x6e, 0x85, 0x39,
			0x9a, 0xaf, 0xdb, 0xa1, 0xe9, 0xe2, 0xcb, 0x37,
			0x04, 0xc6, 0x8c, 0x81, 0xd3, 0x2a, 0xb7, 0xbe,
			0x6c, 0x07, 0x1f, 0x5e, 0xd9, 0x00, 0xd2, 0xf7,
			0xe1, 0xa7, 0xbc, 0x0c, 0xb6, 0x6d, 0xfb, 0x3f,
			0x3d, 0x24, 0xaa, 0xfb, 0x7e, 0xe1, 0xb5, 0x1b,
			0xff, 0x38, 0xaa, 0x69, 0x59, 0x38, 0x52, 0x9a,
			0x0e, 0x6d, 0xbc, 0xde, 0x4f, 0x13, 0x09, 0x17,
			0xc4, 0xa9, 0x05, 0x84, 0xbc, 0x50, 0xef, 0x40,
			0xb0, 0x4c, 0x24, 0x32, 0xed, 0x94, 0x2c, 0xdd,
			0xda, 0x20, 0x24, 0x67, 0xe2, 0xea, 0x71, 0x3d,
			0x4a, 0x04, 0x0d, 0x98, 0x29, 0x20, 0x4c, 0xeb,
			0x70, 0xce, 0x45, 0x9e, 0x5a, 0xaf, 0xb6, 0xa3,
			0x92, 0xc8, 0x28, 0xf2, 0xe3, 0xe8, 0x8a, 0x5d,
			0x0a, 0x33, 0x79, 0x9b, 0x6a, 0xf3, 0x30, 0x01,
			0x1d, 0x47, 0xbd, 0x01, 0xcc, 0x4d, 0x71, 0xc0,
			0x56, 0xfa, 0xfd, 0x37, 0xed, 0x0f, 0x27, 0xc0,
			0xbb, 0xa0, 0xee, 0xc3, 0x79, 0x8b, 0xe7, 0x41,
			0x8f, 0xfa, 0x3a, 0xcb, 0x45, 0x3b, 0x85, 0x9f,
			0x06, 0x90, 0xb2, 0x51, 0x7a, 0xc3, 0x11, 0x41,
			0x4b, 0xe3, 0x26, 0x94, 0x3e, 0xa2, 0xfd, 0x0a,
			0xda, 0x50, 0xf6, 0x50, 0x78, 0x19, 0x6c, 0x52,
			0xd1, 0x12, 0x76, 0xc2, 0x50, 0x2f, 0x0b, 0xca,
			0x33, 0xe5, 0x79, 0x93, 0x14, 0x03, 0x01, 0x00,
			0x01, 0x01, 0x16, 0x03, 0x01, 0x00, 0x24, 0x2b,
			0x51, 0x42, 0x95, 0x6b, 0xca, 0x9f, 0x42, 0x5d,
			0xd2, 0xd9, 0x67, 0xf9, 0x49, 0x30, 0xfd, 0x2a,
			0x46, 0xd3, 0x04, 0xf4, 0x86, 0xf9, 0x11, 0x34,
			0x82, 0xac, 0xe2, 0xc2, 0x2d, 0xc4, 0xd0, 0xfe,
			0xa9, 0xc9, 0x4b, 0x17, 0x03, 0x01, 0x00, 0x21,
			0x65, 0x1c, 0xe9, 0x5c, 0xb6, 0xe2, 0x7c, 0x8e,
			0x49, 0x12, 0x1b, 0xe6, 0x40, 0xd3, 0x97, 0x21,
			0x76, 0x01, 0xe5, 0x80, 0x5e, 0xf3, 0x11, 0x47,
			0x25, 0x02, 0x78, 0x8e, 0x6b, 0xae, 0xb3, 0xf3,
			0x59, 0x15, 0x03, 0x01, 0x00, 0x16, 0x38, 0xc1,
			0x99, 0x2e, 0xf8, 0x6f, 0x45, 0xa4, 0x10, 0x79,
			0x5b, 0xc1, 0x47, 0x9a, 0xf6, 0x5c, 0x90, 0xeb,
			0xa6, 0xe3, 0x1a, 0x24,
		},
	}},
}

var tls11ECDHEAESServerScript = [][]byte{
	{
		0x16, 0x03, 0x01, 0x01, 0x46, 0x01, 0x00, 0x01,
		0x42, 0x03, 0x03, 0x51, 0x9f, 0xa3, 0xb0, 0xb7,
		0x1d, 0x26, 0x93, 0x36, 0xc0, 0x8d, 0x7e, 0xf8,
		0x4f, 0x6f, 0xc9, 0x3c, 0x31, 0x1e, 0x7f, 0xb1,
		0xf0, 0xc1, 0x0f, 0xf9, 0x0c, 0xa2, 0xd5, 0xca,
		0x48, 0xe5, 0x35, 0x00, 0x00, 0xd0, 0xc0, 0x30,
		0xc0, 0x2c, 0xc0, 0x28, 0xc0, 0x24, 0xc0, 0x14,
		0xc0, 0x0a, 0xc0, 0x22, 0xc0, 0x21, 0x00, 0xa5,
		0x00, 0xa3, 0x00, 0xa1, 0x00, 0x9f, 0x00, 0x6b,
		0x00, 0x6a, 0x00, 0x69, 0x00, 0x68, 0x00, 0x39,
		0x00, 0x38, 0x00, 0x37, 0x00, 0x36, 0x00, 0x88,
		0x00, 0x87, 0x00, 0x86, 0x00, 0x85, 0xc0, 0x32,
		0xc0, 0x2e, 0xc0, 0x2a, 0xc0, 0x26, 0xc0, 0x0f,
		0xc0, 0x05, 0x00, 0x9d, 0x00, 0x3d, 0x00, 0x35,
		0x00, 0x84, 0xc0, 0x12, 0xc0, 0x08, 0xc0, 0x1c,
		0xc0, 0x1b, 0x00, 0x16, 0x00, 0x13, 0x00, 0x10,
		0x00, 0x0d, 0xc0, 0x0d, 0xc0, 0x03, 0x00, 0x0a,
		0xc0, 0x2f, 0xc0, 0x2b, 0xc0, 0x27, 0xc0, 0x23,
		0xc0, 0x13, 0xc0, 0x09, 0xc0, 0x1f, 0xc0, 0x1e,
		0x00, 0xa4, 0x00, 0xa2, 0x00, 0xa0, 0x00, 0x9e,
		0x00, 0x67, 0x00, 0x40, 0x00, 0x3f, 0x00, 0x3e,
		0x00, 0x33, 0x00, 0x32, 0x00, 0x31, 0x00, 0x30,
		0x00, 0x9a, 0x00, 0x99, 0x00, 0x98, 0x00, 0x97,
		0x00, 0x45, 0x00, 0x44, 0x00, 0x43, 0x00, 0x42,
		0xc0, 0x31, 0xc0, 0x2d, 0xc0, 0x29, 0xc0, 0x25,
		0xc0, 0x0e, 0xc0, 0x04, 0x00, 0x9c, 0x00, 0x3c,
		0x00, 0x2f, 0x00, 0x96, 0x00, 0x41, 0x00, 0x07,
		0xc0, 0x11, 0xc0, 0x07, 0xc0, 0x0c, 0xc0, 0x02,
		0x00, 0x05, 0x00, 0x04, 0x00, 0x15, 0x00, 0x12,
		0x00, 0x0f, 0x00, 0x0c, 0x00, 0x09, 0x00, 0x14,
		0x00, 0x11, 0x00, 0x0e, 0x00, 0x0b, 0x00, 0x08,
		0x00, 0x06, 0x00, 0x03, 0x00, 0xff, 0x01, 0x00,
		0x00, 0x49, 0x00, 0x0b, 0x00, 0x04, 0x03, 0x00,
		0x01, 0x02, 0x00, 0x0a, 0x00, 0x34, 0x00, 0x32,
		0x00, 0x0e, 0x00, 0x0d, 0x00, 0x19, 0x00, 0x0b,
		0x00, 0x0c, 0x00, 0x18, 0x00, 0x09, 0x00, 0x0a,
		0x00, 0x16, 0x00, 0x17, 0x00, 0x08, 0x00, 0x06,
		0x00, 0x07, 0x00, 0x14, 0x00, 0x15, 0x00, 0x04,
		0x00, 0x05, 0x00, 0x12, 0x00, 0x13, 0x00, 0x01,
		0x00, 0x02, 0x00, 0x03, 0x00, 0x0f, 0x00, 0x10,
		0x00, 0x11, 0x00, 0x23, 0x00, 0x00, 0x00, 0x0f,
		0x00, 0x01, 0x01,
	},
	{
		0x16, 0x03, 0x02, 0x00, 0x30, 0x02, 0x00, 0x00,
		0x2c, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0xc0, 0x13, 0x00, 0x00,
		0x04, 0x00, 0x23, 0x00, 0x00, 0x16, 0x03, 0x02,
		0x02, 0xbe, 0x0b, 0x00, 0x02, 0xba, 0x00, 0x02,
		0xb7, 0x00, 0x02, 0xb4, 0x30, 0x82, 0x02, 0xb0,
		0x30, 0x82, 0x02, 0x19, 0xa0, 0x03, 0x02, 0x01,
		0x02, 0x02, 0x09, 0x00, 0x85, 0xb0, 0xbb, 0xa4,
		0x8a, 0x7f, 0xb8, 0xca, 0x30, 0x0d, 0x06, 0x09,
		0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
		0x05, 0x05, 0x00, 0x30, 0x45, 0x31, 0x0b, 0x30,
		0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
		0x41, 0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03,
		0x55, 0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d,
		0x65, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31,
		0x21, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a,
		0x13, 0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e,
		0x65, 0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69,
		0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c,
		0x74, 0x64, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x30,
		0x30, 0x34, 0x32, 0x34, 0x30, 0x39, 0x30, 0x39,
		0x33, 0x38, 0x5a, 0x17, 0x0d, 0x31, 0x31, 0x30,
		0x34, 0x32, 0x34, 0x30, 0x39, 0x30, 0x39, 0x33,
		0x38, 0x5a, 0x30, 0x45, 0x31, 0x0b, 0x30, 0x09,
		0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x41,
		0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55,
		0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d, 0x65,
		0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x21,
		0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13,
		0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65,
		0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69, 0x74,
		0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c, 0x74,
		0x64, 0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09,
		0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
		0x01, 0x05, 0x00, 0x03, 0x81, 0x8d, 0x00, 0x30,
		0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xbb, 0x79,
		0xd6, 0xf5, 0x17, 0xb5, 0xe5, 0xbf, 0x46, 0x10,
		0xd0, 0xdc, 0x69, 0xbe, 0xe6, 0x2b, 0x07, 0x43,
		0x5a, 0xd0, 0x03, 0x2d, 0x8a, 0x7a, 0x43, 0x85,
		0xb7, 0x14, 0x52, 0xe7, 0xa5, 0x65, 0x4c, 0x2c,
		0x78, 0xb8, 0x23, 0x8c, 0xb5, 0xb4, 0x82, 0xe5,
		0xde, 0x1f, 0x95, 0x3b, 0x7e, 0x62, 0xa5, 0x2c,
		0xa5, 0x33, 0xd6, 0xfe, 0x12, 0x5c, 0x7a, 0x56,
		0xfc, 0xf5, 0x06, 0xbf, 0xfa, 0x58, 0x7b, 0x26,
		0x3f, 0xb5, 0xcd, 0x04, 0xd3, 0xd0, 0xc9, 0x21,
		0x96, 0x4a, 0xc7, 0xf4, 0x54, 0x9f, 0x5a, 0xbf,
		0xef, 0x42, 0x71, 0x00, 0xfe, 0x18, 0x99, 0x07,
		0x7f, 0x7e, 0x88, 0x7d, 0x7d, 0xf1, 0x04, 0x39,
		0xc4, 0xa2, 0x2e, 0xdb, 0x51, 0xc9, 0x7c, 0xe3,
		0xc0, 0x4c, 0x3b, 0x32, 0x66, 0x01, 0xcf, 0xaf,
		0xb1, 0x1d, 0xb8, 0x71, 0x9a, 0x1d, 0xdb, 0xdb,
		0x89, 0x6b, 0xae, 0xda, 0x2d, 0x79, 0x02, 0x03,
		0x01, 0x00, 0x01, 0xa3, 0x81, 0xa7, 0x30, 0x81,
		0xa4, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e,
		0x04, 0x16, 0x04, 0x14, 0xb1, 0xad, 0xe2, 0x85,
		0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69, 0xce, 0x23,
		0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18, 0x88, 0x39,
		0x30, 0x75, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04,
		0x6e, 0x30, 0x6c, 0x80, 0x14, 0xb1, 0xad, 0xe2,
		0x85, 0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69, 0xce,
		0x23, 0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18, 0x88,
		0x39, 0xa1, 0x49, 0xa4, 0x47, 0x30, 0x45, 0x31,
		0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
		0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11,
		0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53,
		0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74,
		0x65, 0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55,
		0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65,
		0x72, 0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64,
		0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79,
		0x20, 0x4c, 0x74, 0x64, 0x82, 0x09, 0x00, 0x85,
		0xb0, 0xbb, 0xa4, 0x8a, 0x7f, 0xb8, 0xca, 0x30,
		0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x05,
		0x30, 0x03, 0x01, 0x01, 0xff, 0x30, 0x0d, 0x06,
		0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
		0x01, 0x05, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00,
		0x08, 0x6c, 0x45, 0x24, 0xc7, 0x6b, 0xb1, 0x59,
		0xab, 0x0c, 0x52, 0xcc, 0xf2, 0xb0, 0x14, 0xd7,
		0x87, 0x9d, 0x7a, 0x64, 0x75, 0xb5, 0x5a, 0x95,
		0x66, 0xe4, 0xc5, 0x2b, 0x8e, 0xae, 0x12, 0x66,
		0x1f, 0xeb, 0x4f, 0x38, 0xb3, 0x6e, 0x60, 0xd3,
		0x92, 0xfd, 0xf7, 0x41, 0x08, 0xb5, 0x25, 0x13,
		0xb1, 0x18, 0x7a, 0x24, 0xfb, 0x30, 0x1d, 0xba,
		0xed, 0x98, 0xb9, 0x17, 0xec, 0xe7, 0xd7, 0x31,
		0x59, 0xdb, 0x95, 0xd3, 0x1d, 0x78, 0xea, 0x50,
		0x56, 0x5c, 0xd5, 0x82, 0x5a, 0x2d, 0x5a, 0x5f,
		0x33, 0xc4, 0xb6, 0xd8, 0xc9, 0x75, 0x90, 0x96,
		0x8c, 0x0f, 0x52, 0x98, 0xb5, 0xcd, 0x98, 0x1f,
		0x89, 0x20, 0x5f, 0xf2, 0xa0, 0x1c, 0xa3, 0x1b,
		0x96, 0x94, 0xdd, 0xa9, 0xfd, 0x57, 0xe9, 0x70,
		0xe8, 0x26, 0x6d, 0x71, 0x99, 0x9b, 0x26, 0x6e,
		0x38, 0x50, 0x29, 0x6c, 0x90, 0xa7, 0xbd, 0xd9,
		0x16, 0x03, 0x02, 0x01, 0x0f, 0x0c, 0x00, 0x01,
		0x0b, 0x03, 0x00, 0x19, 0x85, 0x04, 0x01, 0x39,
		0xdc, 0xee, 0x44, 0x17, 0x5e, 0xdb, 0xd7, 0x27,
		0xaf, 0xb6, 0x56, 0xd9, 0xb4, 0x43, 0x5a, 0x99,
		0xcf, 0xaa, 0x31, 0x37, 0x0c, 0x6f, 0x3a, 0xa0,
		0xf8, 0x53, 0xc4, 0x74, 0xd1, 0x91, 0x0a, 0x46,
		0xf5, 0x38, 0x3b, 0x5c, 0x09, 0xd8, 0x97, 0xdc,
		0x4b, 0xaa, 0x70, 0x26, 0x48, 0xf2, 0xd6, 0x0b,
		0x31, 0xc9, 0xf8, 0xd4, 0x98, 0x43, 0xe1, 0x6c,
		0xd5, 0xc7, 0xb2, 0x8e, 0x0b, 0x01, 0xe6, 0xb6,
		0x00, 0x28, 0x80, 0x7b, 0xfc, 0x96, 0x8f, 0x0d,
		0xa2, 0x4f, 0xb0, 0x79, 0xaf, 0xdc, 0x61, 0x28,
		0x63, 0x33, 0x78, 0xf6, 0x31, 0x39, 0xfd, 0x8a,
		0xf4, 0x15, 0x18, 0x11, 0xfe, 0xdb, 0xd5, 0x07,
		0xda, 0x2c, 0xed, 0x49, 0xa0, 0x23, 0xbf, 0xd0,
		0x3a, 0x38, 0x1d, 0x54, 0xae, 0x1c, 0x7b, 0xea,
		0x29, 0xee, 0xd0, 0x38, 0xc1, 0x76, 0xa7, 0x7f,
		0x2a, 0xf4, 0xce, 0x1e, 0xac, 0xcc, 0x94, 0x79,
		0x90, 0x33, 0x00, 0x80, 0x16, 0x83, 0x9b, 0xf9,
		0x72, 0xdb, 0x9f, 0x55, 0x02, 0xe1, 0x04, 0xf7,
		0xb5, 0x3f, 0x4c, 0x71, 0x13, 0x5a, 0x91, 0xe9,
		0x1d, 0xeb, 0x9d, 0x9c, 0xfb, 0x88, 0xef, 0xca,
		0xec, 0x7d, 0x9b, 0xdd, 0xd9, 0xee, 0x2b, 0x8e,
		0xef, 0xf8, 0xb6, 0xc7, 0x7d, 0xfe, 0xda, 0x7f,
		0x90, 0x2e, 0x53, 0xf1, 0x64, 0x95, 0xfc, 0x66,
		0xfc, 0x87, 0x27, 0xb6, 0x9f, 0xc8, 0x3a, 0x95,
		0x68, 0x17, 0xe1, 0x7d, 0xf1, 0x88, 0xe8, 0x17,
		0x5f, 0x99, 0x90, 0x3f, 0x47, 0x47, 0x81, 0x06,
		0xe2, 0x8e, 0x22, 0x56, 0x8f, 0xc2, 0x14, 0xe5,
		0x62, 0xa7, 0x0d, 0x41, 0x3c, 0xc7, 0x4a, 0x0a,
		0x74, 0x4b, 0xda, 0x00, 0x8e, 0x4f, 0x90, 0xe6,
		0xd7, 0x68, 0xe5, 0x8b, 0xf2, 0x3f, 0x53, 0x1d,
		0x7a, 0xe6, 0xb3, 0xe9, 0x8a, 0xc9, 0x4d, 0x19,
		0xa6, 0xcf, 0xf9, 0xed, 0x5e, 0x26, 0xdc, 0x90,
		0x1c, 0x41, 0xad, 0x7c, 0x16, 0x03, 0x02, 0x00,
		0x04, 0x0e, 0x00, 0x00, 0x00,
	},
	{
		0x16, 0x03, 0x02, 0x00, 0x8a, 0x10, 0x00, 0x00,
		0x86, 0x85, 0x04, 0x01, 0x11, 0xf2, 0xa4, 0x2d,
		0x1a, 0x75, 0x6c, 0xbc, 0x2d, 0x91, 0x95, 0x07,
		0xbe, 0xd6, 0x41, 0x7a, 0xbb, 0xc2, 0x7b, 0xa6,
		0x9b, 0xe3, 0xdc, 0x41, 0x7f, 0x1e, 0x2e, 0xcc,
		0x6d, 0xa3, 0x85, 0x53, 0x98, 0x9f, 0x2d, 0xe6,
		0x3c, 0xb9, 0x82, 0xa6, 0x80, 0x53, 0x9b, 0x71,
		0xfd, 0x27, 0xe5, 0xe5, 0xdf, 0x13, 0xba, 0x56,
		0x62, 0x30, 0x4a, 0x57, 0x27, 0xa7, 0xcc, 0x26,
		0x54, 0xe8, 0x65, 0x6e, 0x4d, 0x00, 0xbf, 0x8a,
		0xcc, 0x89, 0x6a, 0x6c, 0x88, 0xda, 0x79, 0x4f,
		0xc5, 0xad, 0x6d, 0x1d, 0x7c, 0x53, 0x7b, 0x1a,
		0x96, 0xf2, 0xf8, 0x30, 0x01, 0x0b, 0xc2, 0xf0,
		0x78, 0x41, 0xf4, 0x0d, 0xe0, 0xbe, 0xb9, 0x36,
		0xe0, 0xb7, 0xee, 0x16, 0xeb, 0x25, 0x67, 0x04,
		0xc0, 0x2e, 0xd8, 0x34, 0x4a, 0x65, 0xa5, 0xf1,
		0x95, 0x75, 0xc7, 0x39, 0xa9, 0x68, 0xa9, 0x53,
		0x93, 0x5b, 0xca, 0x7b, 0x7f, 0xc0, 0x63, 0x14,
		0x03, 0x02, 0x00, 0x01, 0x01, 0x16, 0x03, 0x02,
		0x00, 0x40, 0x01, 0xb1, 0xae, 0x1b, 0x8a, 0x65,
		0xf8, 0x37, 0x50, 0x39, 0x76, 0xef, 0xaa, 0xda,
		0x84, 0xc9, 0x5f, 0x80, 0xdc, 0xfa, 0xe0, 0x46,
		0x5a, 0xc7, 0x77, 0x9d, 0x76, 0x03, 0xa6, 0xd5,
		0x0e, 0xbf, 0x25, 0x30, 0x5c, 0x99, 0x7d, 0xcd,
		0x2b, 0xaa, 0x2e, 0x8c, 0xdd, 0xda, 0xaa, 0xd7,
		0xf1, 0xf6, 0x33, 0x47, 0x51, 0x1e, 0x83, 0xa1,
		0x83, 0x04, 0xd2, 0xb2, 0xc8, 0xbc, 0x11, 0xc5,
		0x1a, 0x87,
	},
	{
		0x16, 0x03, 0x02, 0x00, 0x72, 0x04, 0x00, 0x00,
		0x6e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x65,
		0xeb, 0x8b, 0xc7, 0xef, 0xba, 0xe8, 0x0f, 0x69,
		0xfe, 0xfb, 0xc3, 0x3d, 0x90, 0x5d, 0xd7, 0xb2,
		0x51, 0x64, 0xac, 0xc3, 0xae, 0x33, 0x03, 0x42,
		0x45, 0x2d, 0xa7, 0x57, 0xbd, 0xa3, 0x85, 0x64,
		0xa6, 0xfe, 0x5c, 0x33, 0x04, 0x93, 0xf2, 0x7c,
		0x06, 0x6d, 0xd7, 0xd7, 0xcf, 0x4a, 0xaf, 0xb2,
		0xdd, 0x06, 0xdc, 0x28, 0x14, 0x59, 0x23, 0x02,
		0xef, 0x97, 0x6a, 0xe8, 0xec, 0xca, 0x10, 0x44,
		0xcd, 0xb8, 0x50, 0x16, 0x46, 0x5a, 0x05, 0xda,
		0x04, 0xb3, 0x0e, 0xe9, 0xf0, 0x74, 0xc5, 0x23,
		0xc2, 0x0e, 0xa1, 0x54, 0x66, 0x7b, 0xe8, 0x14,
		0x03, 0x02, 0x00, 0x01, 0x01, 0x16, 0x03, 0x02,
		0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x6b, 0x43, 0x1c, 0x58, 0xbc, 0x85,
		0xf7, 0xc1, 0x76, 0xbc, 0x72, 0x33, 0x41, 0x6b,
		0xb8, 0xf8, 0xfd, 0x53, 0x21, 0xc2, 0x41, 0x1b,
		0x72, 0x4f, 0xce, 0x97, 0xca, 0x14, 0x23, 0x4d,
		0xbc, 0x44, 0xd6, 0xd7, 0xfc, 0xbc, 0xfd, 0xfd,
		0x5d, 0x33, 0x42, 0x1b, 0x52, 0x40, 0x0a, 0x2b,
		0x6c, 0x98, 0x17, 0x03, 0x02, 0x00, 0x40, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1d,
		0x31, 0xef, 0x03, 0x7d, 0xa5, 0x74, 0x92, 0x24,
		0x34, 0xae, 0x4e, 0xc9, 0xfc, 0x59, 0xcb, 0x64,
		0xf4, 0x45, 0xb1, 0xac, 0x02, 0xf2, 0x87, 0xe7,
		0x2f, 0xfd, 0x01, 0xca, 0x78, 0x02, 0x2e, 0x3a,
		0x38, 0xcd, 0xb1, 0xe0, 0xf2, 0x2e, 0xf6, 0x27,
		0xa0, 0xac, 0x1f, 0x91, 0x43, 0xc2, 0x3d, 0x15,
		0x03, 0x02, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x9f, 0x30, 0x24, 0x56,
		0x2c, 0xde, 0xa0, 0xe6, 0x44, 0x35, 0x30, 0x51,
		0xec, 0xd4, 0x69, 0x2d, 0x46, 0x64, 0x04, 0x21,
		0xfe, 0x7c, 0x4d, 0xc5, 0xd0, 0x8c, 0xf9, 0xd2,
		0x3f, 0x88, 0x69, 0xd5,
	},
}

// $ go test -run TestRunServer -serve -clientauth 1 \
//     -ciphersuites=0xc011 -minversion=0x0303 -maxversion=0x0303
var tls12ServerScript = [][]byte{
	{
		0x16, 0x03, 0x01, 0x01, 0x1e, 0x01, 0x00, 0x01,
		0x1a, 0x03, 0x03, 0x51, 0xe5, 0x76, 0x84, 0x0e,
		0xb9, 0x17, 0xca, 0x08, 0x47, 0xd9, 0xbd, 0xd0,
		0x94, 0xd1, 0x97, 0xca, 0x5b, 0xe7, 0x20, 0xac,
		0x8e, 0xbb, 0xc7, 0x29, 0xe9, 0x26, 0xcf, 0x7d,
		0xb3, 0xdc, 0x99, 0x00, 0x00, 0x82, 0xc0, 0x30,
		0xc0, 0x2c, 0xc0, 0x28, 0xc0, 0x24, 0xc0, 0x14,
		0xc0, 0x0a, 0x00, 0xa3, 0x00, 0x9f, 0x00, 0x6b,
		0x00, 0x6a, 0x00, 0x39, 0x00, 0x38, 0xc0, 0x32,
		0xc0, 0x2e, 0xc0, 0x2a, 0xc0, 0x26, 0xc0, 0x0f,
		0xc0, 0x05, 0x00, 0x9d, 0x00, 0x3d, 0x00, 0x35,
		0xc0, 0x12, 0xc0, 0x08, 0x00, 0x16, 0x00, 0x13,
		0xc0, 0x0d, 0xc0, 0x03, 0x00, 0x0a, 0xc0, 0x2f,
		0xc0, 0x2b, 0xc0, 0x27, 0xc0, 0x23, 0xc0, 0x13,
		0xc0, 0x09, 0x00, 0xa2, 0x00, 0x9e, 0x00, 0x67,
		0x00, 0x40, 0x00, 0x33, 0x00, 0x32, 0xc0, 0x31,
		0xc0, 0x2d, 0xc0, 0x29, 0xc0, 0x25, 0xc0, 0x0e,
		0xc0, 0x04, 0x00, 0x9c, 0x00, 0x3c, 0x00, 0x2f,
		0x00, 0x07, 0xc0, 0x11, 0xc0, 0x07, 0xc0, 0x0c,
		0xc0, 0x02, 0x00, 0x05, 0x00, 0x04, 0x00, 0x15,
		0x00, 0x12, 0x00, 0x09, 0x00, 0x14, 0x00, 0x11,
		0x00, 0x08, 0x00, 0x06, 0x00, 0x03, 0x00, 0xff,
		0x01, 0x00, 0x00, 0x6f, 0x00, 0x0b, 0x00, 0x04,
		0x03, 0x00, 0x01, 0x02, 0x00, 0x0a, 0x00, 0x34,
		0x00, 0x32, 0x00, 0x0e, 0x00, 0x0d, 0x00, 0x19,
		0x00, 0x0b, 0x00, 0x0c, 0x00, 0x18, 0x00, 0x09,
		0x00, 0x0a, 0x00, 0x16, 0x00, 0x17, 0x00, 0x08,
		0x00, 0x06, 0x00, 0x07, 0x00, 0x14, 0x00, 0x15,
		0x00, 0x04, 0x00, 0x05, 0x00, 0x12, 0x00, 0x13,
		0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x0f,
		0x00, 0x10, 0x00, 0x11, 0x00, 0x23, 0x00, 0x00,
		0x00, 0x0d, 0x00, 0x22, 0x00, 0x20, 0x06, 0x01,
		0x06, 0x02, 0x06, 0x03, 0x05, 0x01, 0x05, 0x02,
		0x05, 0x03, 0x04, 0x01, 0x04, 0x02, 0x04, 0x03,
		0x03, 0x01, 0x03, 0x02, 0x03, 0x03, 0x02, 0x01,
		0x02, 0x02, 0x02, 0x03, 0x01, 0x01, 0x00, 0x0f,
		0x00, 0x01, 0x01,
	},
	{
		0x16, 0x03, 0x03, 0x00, 0x30, 0x02, 0x00, 0x00,
		0x2c, 0x03, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0xc0, 0x11, 0x00, 0x00,
		0x04, 0x00, 0x23, 0x00, 0x00, 0x16, 0x03, 0x03,
		0x02, 0xbe, 0x0b, 0x00, 0x02, 0xba, 0x00, 0x02,
		0xb7, 0x00, 0x02, 0xb4, 0x30, 0x82, 0x02, 0xb0,
		0x30, 0x82, 0x02, 0x19, 0xa0, 0x03, 0x02, 0x01,
		0x02, 0x02, 0x09, 0x00, 0x85, 0xb0, 0xbb, 0xa4,
		0x8a, 0x7f, 0xb8, 0xca, 0x30, 0x0d, 0x06, 0x09,
		0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
		0x05, 0x05, 0x00, 0x30, 0x45, 0x31, 0x0b, 0x30,
		0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
		0x41, 0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03,
		0x55, 0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d,
		0x65, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31,
		0x21, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a,
		0x13, 0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e,
		0x65, 0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69,
		0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c,
		0x74, 0x64, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x30,
		0x30, 0x34, 0x32, 0x34, 0x30, 0x39, 0x30, 0x39,
		0x33, 0x38, 0x5a, 0x17, 0x0d, 0x31, 0x31, 0x30,
		0x34, 0x32, 0x34, 0x30, 0x39, 0x30, 0x39, 0x33,
		0x38, 0x5a, 0x30, 0x45, 0x31, 0x0b, 0x30, 0x09,
		0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x41,
		0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55,
		0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d, 0x65,
		0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x21,
		0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13,
		0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65,
		0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69, 0x74,
		0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c, 0x74,
		0x64, 0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09,
		0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
		0x01, 0x05, 0x00, 0x03, 0x81, 0x8d, 0x00, 0x30,
		0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xbb, 0x79,
		0xd6, 0xf5, 0x17, 0xb5, 0xe5, 0xbf, 0x46, 0x10,
		0xd0, 0xdc, 0x69, 0xbe, 0xe6, 0x2b, 0x07, 0x43,
		0x5a, 0xd0, 0x03, 0x2d, 0x8a, 0x7a, 0x43, 0x85,
		0xb7, 0x14, 0x52, 0xe7, 0xa5, 0x65, 0x4c, 0x2c,
		0x78, 0xb8, 0x23, 0x8c, 0xb5, 0xb4, 0x82, 0xe5,
		0xde, 0x1f, 0x95, 0x3b, 0x7e, 0x62, 0xa5, 0x2c,
		0xa5, 0x33, 0xd6, 0xfe, 0x12, 0x5c, 0x7a, 0x56,
		0xfc, 0xf5, 0x06, 0xbf, 0xfa, 0x58, 0x7b, 0x26,
		0x3f, 0xb5, 0xcd, 0x04, 0xd3, 0xd0, 0xc9, 0x21,
		0x96, 0x4a, 0xc7, 0xf4, 0x54, 0x9f, 0x5a, 0xbf,
		0xef, 0x42, 0x71, 0x00, 0xfe, 0x18, 0x99, 0x07,
		0x7f, 0x7e, 0x88, 0x7d, 0x7d, 0xf1, 0x04, 0x39,
		0xc4, 0xa2, 0x2e, 0xdb, 0x51, 0xc9, 0x7c, 0xe3,
		0xc0, 0x4c, 0x3b, 0x32, 0x66, 0x01, 0xcf, 0xaf,
		0xb1, 0x1d, 0xb8, 0x71, 0x9a, 0x1d, 0xdb, 0xdb,
		0x89, 0x6b, 0xae, 0xda, 0x2d, 0x79, 0x02, 0x03,
		0x01, 0x00, 0x01, 0xa3, 0x81, 0xa7, 0x30, 0x81,
		0xa4, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e,
		0x04, 0x16, 0x04, 0x14, 0xb1, 0xad, 0xe2, 0x85,
		0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69, 0xce, 0x23,
		0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18, 0x88, 0x39,
		0x30, 0x75, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04,
		0x6e, 0x30, 0x6c, 0x80, 0x14, 0xb1, 0xad, 0xe2,
		0x85, 0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69, 0xce,
		0x23, 0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18, 0x88,
		0x39, 0xa1, 0x49, 0xa4, 0x47, 0x30, 0x45, 0x31,
		0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
		0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11,
		0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53,
		0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74,
		0x65, 0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55,
		0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65,
		0x72, 0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64,
		0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79,
		0x20, 0x4c, 0x74, 0x64, 0x82, 0x09, 0x00, 0x85,
		0xb0, 0xbb, 0xa4, 0x8a, 0x7f, 0xb8, 0xca, 0x30,
		0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x05,
		0x30, 0x03, 0x01, 0x01, 0xff, 0x30, 0x0d, 0x06,
		0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
		0x01, 0x05, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00,
		0x08, 0x6c, 0x45, 0x24, 0xc7, 0x6b, 0xb1, 0x59,
		0xab, 0x0c, 0x52, 0xcc, 0xf2, 0xb0, 0x14, 0xd7,
		0x87, 0x9d, 0x7a, 0x64, 0x75, 0xb5, 0x5a, 0x95,
		0x66, 0xe4, 0xc5, 0x2b, 0x8e, 0xae, 0x12, 0x66,
		0x1f, 0xeb, 0x4f, 0x38, 0xb3, 0x6e, 0x60, 0xd3,
		0x92, 0xfd, 0xf7, 0x41, 0x08, 0xb5, 0x25, 0x13,
		0xb1, 0x18, 0x7a, 0x24, 0xfb, 0x30, 0x1d, 0xba,
		0xed, 0x98, 0xb9, 0x17, 0xec, 0xe7, 0xd7, 0x31,
		0x59, 0xdb, 0x95, 0xd3, 0x1d, 0x78, 0xea, 0x50,
		0x56, 0x5c, 0xd5, 0x82, 0x5a, 0x2d, 0x5a, 0x5f,
		0x33, 0xc4, 0xb6, 0xd8, 0xc9, 0x75, 0x90, 0x96,
		0x8c, 0x0f, 0x52, 0x98, 0xb5, 0xcd, 0x98, 0x1f,
		0x89, 0x20, 0x5f, 0xf2, 0xa0, 0x1c, 0xa3, 0x1b,
		0x96, 0x94, 0xdd, 0xa9, 0xfd, 0x57, 0xe9, 0x70,
		0xe8, 0x26, 0x6d, 0x71, 0x99, 0x9b, 0x26, 0x6e,
		0x38, 0x50, 0x29, 0x6c, 0x90, 0xa7, 0xbd, 0xd9,
		0x16, 0x03, 0x03, 0x01, 0x11, 0x0c, 0x00, 0x01,
		0x0d, 0x03, 0x00, 0x19, 0x85, 0x04, 0x01, 0x39,
		0xdc, 0xee, 0x44, 0x17, 0x5e, 0xdb, 0xd7, 0x27,
		0xaf, 0xb6, 0x56, 0xd9, 0xb4, 0x43, 0x5a, 0x99,
		0xcf, 0xaa, 0x31, 0x37, 0x0c, 0x6f, 0x3a, 0xa0,
		0xf8, 0x53, 0xc4, 0x74, 0xd1, 0x91, 0x0a, 0x46,
		0xf5, 0x38, 0x3b, 0x5c, 0x09, 0xd8, 0x97, 0xdc,
		0x4b, 0xaa, 0x70, 0x26, 0x48, 0xf2, 0xd6, 0x0b,
		0x31, 0xc9, 0xf8, 0xd4, 0x98, 0x43, 0xe1, 0x6c,
		0xd5, 0xc7, 0xb2, 0x8e, 0x0b, 0x01, 0xe6, 0xb6,
		0x00, 0x28, 0x80, 0x7b, 0xfc, 0x96, 0x8f, 0x0d,
		0xa2, 0x4f, 0xb0, 0x79, 0xaf, 0xdc, 0x61, 0x28,
		0x63, 0x33, 0x78, 0xf6, 0x31, 0x39, 0xfd, 0x8a,
		0xf4, 0x15, 0x18, 0x11, 0xfe, 0xdb, 0xd5, 0x07,
		0xda, 0x2c, 0xed, 0x49, 0xa0, 0x23, 0xbf, 0xd0,
		0x3a, 0x38, 0x1d, 0x54, 0xae, 0x1c, 0x7b, 0xea,
		0x29, 0xee, 0xd0, 0x38, 0xc1, 0x76, 0xa7, 0x7f,
		0x2a, 0xf4, 0xce, 0x1e, 0xac, 0xcc, 0x94, 0x79,
		0x90, 0x33, 0x04, 0x01, 0x00, 0x80, 0x4a, 0xf9,
		0xf5, 0x0a, 0x61, 0x37, 0x7e, 0x4e, 0x92, 0xb5,
		0x1c, 0x91, 0x21, 0xb2, 0xb5, 0x17, 0x00, 0xbf,
		0x01, 0x5f, 0x30, 0xec, 0x62, 0x08, 0xd6, 0x9d,
		0x1a, 0x08, 0x05, 0x72, 0x8b, 0xf4, 0x49, 0x85,
		0xa7, 0xbf, 0x3f, 0x75, 0x58, 0x3e, 0x26, 0x82,
		0xc3, 0x28, 0x07, 0xf9, 0x41, 0x7d, 0x03, 0x14,
		0x3b, 0xc3, 0x05, 0x64, 0xff, 0x52, 0xf4, 0x75,
		0x6a, 0x87, 0xcd, 0xdf, 0x93, 0x31, 0x0a, 0x71,
		0x60, 0x17, 0xc6, 0x33, 0xf0, 0x79, 0xb6, 0x7b,
		0xd0, 0x9c, 0xa0, 0x5f, 0x74, 0x14, 0x2c, 0x5a,
		0xb4, 0x3f, 0x39, 0xf5, 0xe4, 0x9f, 0xbe, 0x6d,
		0x21, 0xd2, 0xa9, 0x42, 0xf7, 0xdc, 0xa6, 0x65,
		0xb7, 0x6a, 0x7e, 0x2e, 0x14, 0xd3, 0xf6, 0xf3,
		0x4b, 0x4c, 0x5b, 0x1a, 0x70, 0x7a, 0xbc, 0xb0,
		0x12, 0xf3, 0x6e, 0x0c, 0xcf, 0x43, 0x22, 0xae,
		0x5b, 0xba, 0x00, 0xf8, 0xfd, 0xaf, 0x16, 0x03,
		0x03, 0x00, 0x0f, 0x0d, 0x00, 0x00, 0x0b, 0x02,
		0x01, 0x40, 0x00, 0x04, 0x04, 0x01, 0x04, 0x03,
		0x00, 0x00, 0x16, 0x03, 0x03, 0x00, 0x04, 0x0e,
		0x00, 0x00, 0x00,
	},
	{
		0x16, 0x03, 0x03, 0x01, 0xfb, 0x0b, 0x00, 0x01,
		0xf7, 0x00, 0x01, 0xf4, 0x00, 0x01, 0xf1, 0x30,
		0x82, 0x01, 0xed, 0x30, 0x82, 0x01, 0x58, 0xa0,
		0x03, 0x02, 0x01, 0x02, 0x02, 0x01, 0x00, 0x30,
		0x0b, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
		0x0d, 0x01, 0x01, 0x05, 0x30, 0x26, 0x31, 0x10,
		0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13,
		0x07, 0x41, 0x63, 0x6d, 0x65, 0x20, 0x43, 0x6f,
		0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04,
		0x03, 0x13, 0x09, 0x31, 0x32, 0x37, 0x2e, 0x30,
		0x2e, 0x30, 0x2e, 0x31, 0x30, 0x1e, 0x17, 0x0d,
		0x31, 0x31, 0x31, 0x32, 0x30, 0x38, 0x30, 0x37,
		0x35, 0x35, 0x31, 0x32, 0x5a, 0x17, 0x0d, 0x31,
		0x32, 0x31, 0x32, 0x30, 0x37, 0x30, 0x38, 0x30,
		0x30, 0x31, 0x32, 0x5a, 0x30, 0x26, 0x31, 0x10,
		0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13,
		0x07, 0x41, 0x63, 0x6d, 0x65, 0x20, 0x43, 0x6f,
		0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04,
		0x03, 0x13, 0x09, 0x31, 0x32, 0x37, 0x2e, 0x30,
		0x2e, 0x30, 0x2e, 0x31, 0x30, 0x81, 0x9c, 0x30,
		0x0b, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
		0x0d, 0x01, 0x01, 0x01, 0x03, 0x81, 0x8c, 0x00,
		0x30, 0x81, 0x88, 0x02, 0x81, 0x80, 0x4e, 0xd0,
		0x7b, 0x31, 0xe3, 0x82, 0x64, 0xd9, 0x59, 0xc0,
		0xc2, 0x87, 0xa4, 0x5e, 0x1e, 0x8b, 0x73, 0x33,
		0xc7, 0x63, 0x53, 0xdf, 0x66, 0x92, 0x06, 0x84,
		0xf6, 0x64, 0xd5, 0x8f, 0xe4, 0x36, 0xa7, 0x1d,
		0x2b, 0xe8, 0xb3, 0x20, 0x36, 0x45, 0x23, 0xb5,
		0xe3, 0x95, 0xae, 0xed, 0xe0, 0xf5, 0x20, 0x9c,
		0x8d, 0x95, 0xdf, 0x7f, 0x5a, 0x12, 0xef, 0x87,
		0xe4, 0x5b, 0x68, 0xe4, 0xe9, 0x0e, 0x74, 0xec,
		0x04, 0x8a, 0x7f, 0xde, 0x93, 0x27, 0xc4, 0x01,
		0x19, 0x7a, 0xbd, 0xf2, 0xdc, 0x3d, 0x14, 0xab,
		0xd0, 0x54, 0xca, 0x21, 0x0c, 0xd0, 0x4d, 0x6e,
		0x87, 0x2e, 0x5c, 0xc5, 0xd2, 0xbb, 0x4d, 0x4b,
		0x4f, 0xce, 0xb6, 0x2c, 0xf7, 0x7e, 0x88, 0xec,
		0x7c, 0xd7, 0x02, 0x91, 0x74, 0xa6, 0x1e, 0x0c,
		0x1a, 0xda, 0xe3, 0x4a, 0x5a, 0x2e, 0xde, 0x13,
		0x9c, 0x4c, 0x40, 0x88, 0x59, 0x93, 0x02, 0x03,
		0x01, 0x00, 0x01, 0xa3, 0x32, 0x30, 0x30, 0x30,
		0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
		0xff, 0x04, 0x04, 0x03, 0x02, 0x00, 0xa0, 0x30,
		0x0d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x06,
		0x04, 0x04, 0x01, 0x02, 0x03, 0x04, 0x30, 0x0f,
		0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x08, 0x30,
		0x06, 0x80, 0x04, 0x01, 0x02, 0x03, 0x04, 0x30,
		0x0b, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
		0x0d, 0x01, 0x01, 0x05, 0x03, 0x81, 0x81, 0x00,
		0x36, 0x1f, 0xb3, 0x7a, 0x0c, 0x75, 0xc9, 0x6e,
		0x37, 0x46, 0x61, 0x2b, 0xd5, 0xbd, 0xc0, 0xa7,
		0x4b, 0xcc, 0x46, 0x9a, 0x81, 0x58, 0x7c, 0x85,
		0x79, 0x29, 0xc8, 0xc8, 0xc6, 0x67, 0xdd, 0x32,
		0x56, 0x45, 0x2b, 0x75, 0xb6, 0xe9, 0x24, 0xa9,
		0x50, 0x9a, 0xbe, 0x1f, 0x5a, 0xfa, 0x1a, 0x15,
		0xd9, 0xcc, 0x55, 0x95, 0x72, 0x16, 0x83, 0xb9,
		0xc2, 0xb6, 0x8f, 0xfd, 0x88, 0x8c, 0x38, 0x84,
		0x1d, 0xab, 0x5d, 0x92, 0x31, 0x13, 0x4f, 0xfd,
		0x83, 0x3b, 0xc6, 0x9d, 0xf1, 0x11, 0x62, 0xb6,
		0x8b, 0xec, 0xab, 0x67, 0xbe, 0xc8, 0x64, 0xb0,
		0x11, 0x50, 0x46, 0x58, 0x17, 0x6b, 0x99, 0x1c,
		0xd3, 0x1d, 0xfc, 0x06, 0xf1, 0x0e, 0xe5, 0x96,
		0xa8, 0x0c, 0xf9, 0x78, 0x20, 0xb7, 0x44, 0x18,
		0x51, 0x8d, 0x10, 0x7e, 0x4f, 0x94, 0x67, 0xdf,
		0xa3, 0x4e, 0x70, 0x73, 0x8e, 0x90, 0x91, 0x85,
		0x16, 0x03, 0x03, 0x00, 0x8a, 0x10, 0x00, 0x00,
		0x86, 0x85, 0x04, 0x01, 0x5d, 0x3a, 0x92, 0x59,
		0x7f, 0x9a, 0x22, 0x36, 0x0e, 0x1b, 0x1d, 0x2a,
		0x05, 0xb7, 0xa4, 0xb6, 0x5d, 0xfc, 0x51, 0x6e,
		0x15, 0xe5, 0x89, 0x7c, 0xe2, 0xfa, 0x87, 0x38,
		0x05, 0x79, 0x15, 0x92, 0xb4, 0x8f, 0x88, 0x8f,
		0x9d, 0x5d, 0xa0, 0xaf, 0xf8, 0xce, 0xf9, 0x6f,
		0x83, 0xf4, 0x08, 0x69, 0xe4, 0x91, 0xc5, 0xed,
		0xb9, 0xc5, 0xa8, 0x1f, 0x4b, 0xec, 0xef, 0x91,
		0xc1, 0xa3, 0x34, 0x24, 0x18, 0x00, 0x2d, 0xcd,
		0xe6, 0x44, 0xef, 0x5a, 0x3e, 0x52, 0x63, 0x5b,
		0x36, 0x1f, 0x7e, 0xce, 0x9e, 0xaa, 0xda, 0x8d,
		0xb5, 0xc9, 0xea, 0xd8, 0x1b, 0xd1, 0x1c, 0x7c,
		0x07, 0xfc, 0x3c, 0x2d, 0x70, 0x1f, 0xf9, 0x4d,
		0xcb, 0xaa, 0xad, 0x07, 0xd5, 0x6d, 0xbd, 0xa6,
		0x61, 0xf3, 0x2f, 0xa3, 0x9c, 0x45, 0x02, 0x4a,
		0xac, 0x6c, 0xb6, 0x37, 0x95, 0xb1, 0x4a, 0xb5,
		0x0a, 0x4e, 0x60, 0x67, 0xd7, 0xe0, 0x04, 0x16,
		0x03, 0x03, 0x00, 0x88, 0x0f, 0x00, 0x00, 0x84,
		0x04, 0x01, 0x00, 0x80, 0x08, 0x83, 0x53, 0xf0,
		0xf8, 0x14, 0xf5, 0xc2, 0xd1, 0x8b, 0xf0, 0xa5,
		0xc1, 0xd8, 0x1a, 0x36, 0x4b, 0x75, 0x77, 0x02,
		0x19, 0xd8, 0x11, 0x3f, 0x5a, 0x36, 0xfc, 0xe9,
		0x2b, 0x4b, 0xf9, 0xfe, 0xda, 0x8a, 0x0f, 0x6e,
		0x3d, 0xd3, 0x52, 0x87, 0xf7, 0x9c, 0x78, 0x39,
		0xa8, 0xf1, 0xd7, 0xf7, 0x4e, 0x35, 0x33, 0xf9,
		0xc5, 0x76, 0xa8, 0x12, 0xc4, 0x91, 0x33, 0x1d,
		0x93, 0x8c, 0xbf, 0xb1, 0x83, 0x00, 0x90, 0xc5,
		0x52, 0x3e, 0xe0, 0x0a, 0xe8, 0x92, 0x75, 0xdf,
		0x54, 0x5f, 0x9f, 0x95, 0x76, 0x62, 0xb5, 0x85,
		0x69, 0xa4, 0x86, 0x85, 0x6c, 0xf3, 0x6b, 0x2a,
		0x72, 0x7b, 0x4d, 0x42, 0x33, 0x67, 0x4a, 0xce,
		0xb5, 0xdb, 0x9b, 0xae, 0xc0, 0xb0, 0x10, 0xeb,
		0x3b, 0xf4, 0xc2, 0x9a, 0x64, 0x47, 0x4c, 0x1e,
		0xa5, 0x91, 0x7f, 0x6d, 0xd1, 0x03, 0xf5, 0x4a,
		0x90, 0x69, 0x18, 0xb1, 0x14, 0x03, 0x03, 0x00,
		0x01, 0x01, 0x16, 0x03, 0x03, 0x00, 0x24, 0x59,
		0xfc, 0x7e, 0xae, 0xb3, 0xbf, 0xab, 0x4d, 0xdb,
		0x4e, 0xab, 0xa9, 0x6d, 0x6b, 0x4c, 0x60, 0xb6,
		0x16, 0xe0, 0xab, 0x7f, 0x52, 0x2d, 0xa1, 0xfc,
		0xe1, 0x80, 0xd2, 0x8a, 0xa1, 0xe5, 0x8f, 0xa1,
		0x70, 0x93, 0x23,
	},
	{
		0x16, 0x03, 0x03, 0x02, 0x67, 0x04, 0x00, 0x02,
		0x63, 0x00, 0x00, 0x00, 0x00, 0x02, 0x5d, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x65,
		0xea, 0x8b, 0xc5, 0xef, 0xba, 0x64, 0xb7, 0x23,
		0x08, 0x86, 0x4f, 0x37, 0xe0, 0x8f, 0xbd, 0x75,
		0x71, 0x2b, 0xcb, 0x20, 0x75, 0x11, 0x3b, 0xa2,
		0x9e, 0x39, 0x3c, 0x03, 0xef, 0x6e, 0x41, 0xd7,
		0xcf, 0x1a, 0x2c, 0xf2, 0xfe, 0xc2, 0xd3, 0x65,
		0x59, 0x00, 0x9d, 0x03, 0xb4, 0xf2, 0x20, 0xe4,
		0x33, 0x80, 0xcd, 0xf6, 0xe4, 0x59, 0x22, 0xf7,
		0xfd, 0x88, 0x0e, 0xa4, 0x09, 0xc0, 0x0d, 0x10,
		0x80, 0x10, 0x79, 0xee, 0x70, 0x96, 0xdb, 0x22,
		0x8b, 0xb7, 0xac, 0xe0, 0x98, 0xad, 0xe9, 0xe3,
		0xcb, 0xea, 0x9f, 0xe6, 0x83, 0x28, 0x7c, 0x7e,
		0x4e, 0x9a, 0x8d, 0xd9, 0xf3, 0x86, 0xf4, 0x89,
		0x8b, 0x79, 0x8f, 0xbb, 0xe9, 0x74, 0x02, 0x02,
		0x14, 0x04, 0xea, 0xba, 0x16, 0x10, 0xa1, 0x85,
		0xbe, 0x4e, 0x4e, 0x92, 0xc5, 0x83, 0xf6, 0x1e,
		0x1f, 0xd4, 0x25, 0xc2, 0xc2, 0xb9, 0xce, 0x33,
		0x63, 0x66, 0x79, 0x1f, 0x54, 0x35, 0xc1, 0xe8,
		0x89, 0x34, 0x78, 0x94, 0x36, 0x14, 0xef, 0x01,
		0x1f, 0xf1, 0xbd, 0x77, 0x2c, 0x4d, 0xac, 0x5c,
		0x5c, 0x4a, 0xc6, 0xed, 0xd8, 0x0e, 0x72, 0x84,
		0x83, 0xdc, 0x56, 0x84, 0xc8, 0xf3, 0x89, 0x56,
		0xfd, 0x89, 0xc1, 0xc9, 0x9a, 0x29, 0x91, 0x7e,
		0x19, 0xe9, 0x8b, 0x5b, 0x11, 0x15, 0x4e, 0x6c,
		0xf4, 0x89, 0xe7, 0x6d, 0x68, 0x1e, 0xf9, 0x6c,
		0x23, 0x72, 0x05, 0x68, 0x82, 0x60, 0x84, 0x1f,
		0x83, 0x20, 0x09, 0x86, 0x10, 0x81, 0xec, 0xec,
		0xdc, 0x25, 0x53, 0x20, 0xfa, 0xa9, 0x41, 0x64,
		0xd6, 0x20, 0xf3, 0xf4, 0x52, 0xf2, 0x80, 0x62,
		0x83, 0xc9, 0x23, 0x66, 0x44, 0x95, 0x5a, 0x99,
		0x8a, 0xe1, 0x26, 0x63, 0xc1, 0x8b, 0x31, 0xf9,
		0x21, 0x06, 0x77, 0x04, 0x27, 0xf2, 0x0c, 0x63,
		0x83, 0x45, 0xa0, 0xa9, 0x7b, 0xcf, 0xdf, 0xd7,
		0x56, 0x75, 0xbc, 0xdd, 0x95, 0x36, 0xb1, 0x75,
		0x39, 0x05, 0x00, 0x3c, 0x8a, 0x79, 0xd6, 0xe9,
		0xf0, 0x4b, 0xdc, 0x51, 0x6b, 0x01, 0x94, 0x16,
		0x87, 0x12, 0x92, 0x6c, 0x07, 0xc1, 0xf5, 0x58,
		0xb7, 0x2a, 0x81, 0xf5, 0xa0, 0x37, 0x8b, 0xa6,
		0x22, 0xfe, 0x28, 0x0a, 0x7e, 0x68, 0xe2, 0xda,
		0x6c, 0x53, 0xee, 0x0e, 0x8d, 0x2d, 0x8b, 0x0b,
		0xda, 0xf8, 0x99, 0x3e, 0x0e, 0xed, 0x9f, 0xc1,
		0x2b, 0xf6, 0xfe, 0xe9, 0x52, 0x38, 0x7b, 0x83,
		0x9a, 0x50, 0xa6, 0xd7, 0x49, 0x83, 0x43, 0x7e,
		0x82, 0xec, 0xc7, 0x09, 0x3d, 0x3d, 0xb1, 0xee,
		0xe8, 0xc5, 0x6a, 0xc3, 0x3d, 0x4b, 0x4c, 0x6a,
		0xbb, 0x0b, 0x2c, 0x24, 0x2e, 0xdb, 0x7d, 0x57,
		0x87, 0xb4, 0x80, 0xa5, 0xae, 0xff, 0x54, 0xa8,
		0xa5, 0x27, 0x69, 0x95, 0xc8, 0xe7, 0x79, 0xc7,
		0x89, 0x2a, 0x73, 0x49, 0xcb, 0xf5, 0xc5, 0xbc,
		0x4a, 0xe0, 0x73, 0xa9, 0xbc, 0x88, 0x64, 0x96,
		0x98, 0xa5, 0x1e, 0xe3, 0x43, 0xc1, 0x7d, 0x78,
		0xc7, 0x94, 0x72, 0xd4, 0x2c, 0x6e, 0x85, 0x39,
		0x9a, 0xaf, 0xdb, 0xa1, 0xe9, 0xe2, 0xcb, 0x37,
		0x04, 0xc6, 0x8c, 0x81, 0xd3, 0x2a, 0xb7, 0xbe,
		0x6c, 0x07, 0x1f, 0x5e, 0xd9, 0x00, 0xd2, 0xf7,
		0xe1, 0xa7, 0xbc, 0x0c, 0xb6, 0x6d, 0xfb, 0x3f,
		0x3d, 0x24, 0xaa, 0xfb, 0x7e, 0xe1, 0xb5, 0x1b,
		0xff, 0x38, 0xaa, 0x69, 0x59, 0x38, 0x52, 0x9a,
		0x0e, 0x6d, 0xbc, 0xde, 0x4f, 0x13, 0x09, 0x17,
		0xc4, 0xa9, 0x05, 0x84, 0xbc, 0x50, 0xef, 0x40,
		0xb0, 0x4c, 0x24, 0x32, 0xed, 0x94, 0x2c, 0xdd,
		0xda, 0x20, 0x24, 0x67, 0xe2, 0xea, 0x71, 0x3d,
		0x4a, 0x04, 0x0d, 0x98, 0x29, 0x20, 0x4c, 0xeb,
		0x70, 0xce, 0x45, 0x9e, 0x5a, 0xaf, 0xb6, 0xa3,
		0x92, 0xc8, 0x28, 0xf2, 0xe3, 0xe8, 0x8a, 0x5d,
		0x0a, 0x33, 0x79, 0x9b, 0x6a, 0xf3, 0x30, 0x01,
		0x1d, 0x47, 0xbd, 0x01, 0xcc, 0x4d, 0x71, 0xc0,
		0x56, 0xfa, 0xfd, 0x37, 0xed, 0x0f, 0x27, 0xc0,
		0xbb, 0xa0, 0xee, 0xc3, 0x79, 0x8b, 0xe7, 0x41,
		0x8f, 0xfa, 0x3a, 0xcb, 0x45, 0x3b, 0x85, 0x9f,
		0x06, 0x90, 0xb2, 0x51, 0xc0, 0x48, 0x10, 0xac,
		0x2a, 0xec, 0xec, 0x48, 0x7a, 0x19, 0x47, 0xc4,
		0x2a, 0xeb, 0xb3, 0xa2, 0x07, 0x22, 0x32, 0x78,
		0xf4, 0x73, 0x5e, 0x92, 0x42, 0x15, 0xa1, 0x90,
		0x91, 0xd0, 0xeb, 0x12, 0x14, 0x03, 0x03, 0x00,
		0x01, 0x01, 0x16, 0x03, 0x03, 0x00, 0x24, 0x45,
		0x4b, 0x80, 0x42, 0x46, 0xde, 0xbb, 0xe7, 0x76,
		0xd1, 0x33, 0x92, 0xfc, 0x46, 0x17, 0x6d, 0x21,
		0xf6, 0x0e, 0x16, 0xca, 0x9b, 0x9b, 0x04, 0x65,
		0x16, 0x40, 0x44, 0x64, 0xbc, 0x58, 0xfa, 0x2a,
		0x49, 0xe9, 0xed, 0x17, 0x03, 0x03, 0x00, 0x21,
		0x89, 0x71, 0xcd, 0x56, 0x54, 0xbf, 0x73, 0xde,
		0xfb, 0x4b, 0x4e, 0xf1, 0x7f, 0xc6, 0x75, 0xa6,
		0xbd, 0x6b, 0x6c, 0xd9, 0xdc, 0x0c, 0x71, 0xb4,
		0xb9, 0xbb, 0x6e, 0xfa, 0x9e, 0xc7, 0xc7, 0x4c,
		0x24, 0x15, 0x03, 0x03, 0x00, 0x16, 0x62, 0xea,
		0x65, 0x69, 0x68, 0x4a, 0xce, 0xa7, 0x9e, 0xce,
		0xc0, 0xf1, 0x5c, 0x96, 0xd9, 0x1f, 0x49, 0xac,
		0x2d, 0x05, 0x89, 0x94,
	},
}

// cert.pem and key.pem were generated with generate_cert.go
// Thus, they have no ExtKeyUsage fields and trigger an error
// when verification is turned on.

var clientCertificate = loadPEMCert(`
-----BEGIN CERTIFICATE-----
MIIB7TCCAVigAwIBAgIBADALBgkqhkiG9w0BAQUwJjEQMA4GA1UEChMHQWNtZSBD
bzESMBAGA1UEAxMJMTI3LjAuMC4xMB4XDTExMTIwODA3NTUxMloXDTEyMTIwNzA4
MDAxMlowJjEQMA4GA1UEChMHQWNtZSBDbzESMBAGA1UEAxMJMTI3LjAuMC4xMIGc
MAsGCSqGSIb3DQEBAQOBjAAwgYgCgYBO0Hsx44Jk2VnAwoekXh6LczPHY1PfZpIG
hPZk1Y/kNqcdK+izIDZFI7Xjla7t4PUgnI2V339aEu+H5Fto5OkOdOwEin/ekyfE
ARl6vfLcPRSr0FTKIQzQTW6HLlzF0rtNS0/Otiz3fojsfNcCkXSmHgwa2uNKWi7e
E5xMQIhZkwIDAQABozIwMDAOBgNVHQ8BAf8EBAMCAKAwDQYDVR0OBAYEBAECAwQw
DwYDVR0jBAgwBoAEAQIDBDALBgkqhkiG9w0BAQUDgYEANh+zegx1yW43RmEr1b3A
p0vMRpqBWHyFeSnIyMZn3TJWRSt1tukkqVCavh9a+hoV2cxVlXIWg7nCto/9iIw4
hB2rXZIxE0/9gzvGnfERYraL7KtnvshksBFQRlgXa5kc0x38BvEO5ZaoDPl4ILdE
GFGNEH5PlGffo05wc46QkYU=
-----END CERTIFICATE-----
`)

/* corresponding key.pem for cert.pem is:
-----BEGIN RSA PRIVATE KEY-----
MIICWgIBAAKBgE7QezHjgmTZWcDCh6ReHotzM8djU99mkgaE9mTVj+Q2px0r6LMg
NkUjteOVru3g9SCcjZXff1oS74fkW2jk6Q507ASKf96TJ8QBGXq98tw9FKvQVMoh
DNBNbocuXMXSu01LT862LPd+iOx81wKRdKYeDBra40paLt4TnExAiFmTAgMBAAEC
gYBxvXd8yNteFTns8A/2yomEMC4yeosJJSpp1CsN3BJ7g8/qTnrVPxBy+RU+qr63
t2WquaOu/cr5P8iEsa6lk20tf8pjKLNXeX0b1RTzK8rJLbS7nGzP3tvOhL096VtQ
dAo4ROEaro0TzYpHmpciSvxVIeEIAAdFDObDJPKqcJAxyQJBAJizfYgK8Gzx9fsx
hxp+VteCbVPg2euASH5Yv3K5LukRdKoSzHE2grUVQgN/LafC0eZibRanxHegYSr7
7qaswKUCQQCEIWor/X4XTMdVj3Oj+vpiw75y/S9gh682+myZL+d/02IEkwnB098P
RkKVpenBHyrGg0oeN5La7URILWKj7CPXAkBKo6F+d+phNjwIFoN1Xb/RA32w/D1I
saG9sF+UEhRt9AxUfW/U/tIQ9V0ZHHcSg1XaCM5Nvp934brdKdvTOKnJAkBD5h/3
Rybatlvg/fzBEaJFyq09zhngkxlZOUtBVTqzl17RVvY2orgH02U4HbCHy4phxOn7
qTdQRYlHRftgnWK1AkANibn9PRYJ7mJyJ9Dyj2QeNcSkSTzrt0tPvUMf4+meJymN
1Ntu5+S1DLLzfxlaljWG6ylW6DNxujCyuXIV2rvA
-----END RSA PRIVATE KEY-----
*/

var clientECDSACertificate = loadPEMCert(`
-----BEGIN CERTIFICATE-----
MIIB/DCCAV4CCQCaMIRsJjXZFzAJBgcqhkjOPQQBMEUxCzAJBgNVBAYTAkFVMRMw
EQYDVQQIEwpTb21lLVN0YXRlMSEwHwYDVQQKExhJbnRlcm5ldCBXaWRnaXRzIFB0
eSBMdGQwHhcNMTIxMTE0MTMyNTUzWhcNMjIxMTEyMTMyNTUzWjBBMQswCQYDVQQG
EwJBVTEMMAoGA1UECBMDTlNXMRAwDgYDVQQHEwdQeXJtb250MRIwEAYDVQQDEwlK
b2VsIFNpbmcwgZswEAYHKoZIzj0CAQYFK4EEACMDgYYABACVjJF1FMBexFe01MNv
ja5oHt1vzobhfm6ySD6B5U7ixohLZNz1MLvT/2XMW/TdtWo+PtAd3kfDdq0Z9kUs
jLzYHQFMH3CQRnZIi4+DzEpcj0B22uCJ7B0rxE4wdihBsmKo+1vx+U56jb0JuK7q
ixgnTy5w/hOWusPTQBbNZU6sER7m8TAJBgcqhkjOPQQBA4GMADCBiAJCAOAUxGBg
C3JosDJdYUoCdFzCgbkWqD8pyDbHgf9stlvZcPE4O1BIKJTLCRpS8V3ujfK58PDa
2RU6+b0DeoeiIzXsAkIBo9SKeDUcSpoj0gq+KxAxnZxfvuiRs9oa9V2jI/Umi0Vw
jWVim34BmT0Y9hCaOGGbLlfk+syxis7iI6CH8OFnUes=
-----END CERTIFICATE-----
`)

/* corresponding key for cert is:
-----BEGIN EC PARAMETERS-----
BgUrgQQAIw==
-----END EC PARAMETERS-----
-----BEGIN EC PRIVATE KEY-----
MIHcAgEBBEIBkJN9X4IqZIguiEVKMqeBUP5xtRsEv4HJEtOpOGLELwO53SD78Ew8
k+wLWoqizS3NpQyMtrU8JFdWfj+C57UNkOugBwYFK4EEACOhgYkDgYYABACVjJF1
FMBexFe01MNvja5oHt1vzobhfm6ySD6B5U7ixohLZNz1MLvT/2XMW/TdtWo+PtAd
3kfDdq0Z9kUsjLzYHQFMH3CQRnZIi4+DzEpcj0B22uCJ7B0rxE4wdihBsmKo+1vx
+U56jb0JuK7qixgnTy5w/hOWusPTQBbNZU6sER7m8Q==
-----END EC PRIVATE KEY-----
*/
var clientauthECDSATests = []clientauthTest{
	// Server asks for cert with empty CA list, client gives one
	//   go test -run "TestRunServer" -serve \
	//     -clientauth 1 -ciphersuites=0xc00a
	//   openssl s_client -host 127.0.0.1 -port 10443 \
	//     -cipher ECDHE-ECDSA-AES256-SHA -key client.key -cert client.crt
	{"RequestClientCert, client gives it", RequestClientCert, []*x509.Certificate{clientECDSACertificate}, [][]byte{
		{
			0x16, 0x03, 0x01, 0x00, 0xa0, 0x01, 0x00, 0x00,
			0x9c, 0x03, 0x03, 0x51, 0xe5, 0x73, 0xc5, 0xae,
			0x51, 0x94, 0xb4, 0xf2, 0xe8, 0xf6, 0x03, 0x0e,
			0x3b, 0x34, 0xaf, 0xf0, 0xdc, 0x1b, 0xcc, 0xd8,
			0x0c, 0x45, 0x82, 0xd4, 0xd6, 0x76, 0x04, 0x6e,
			0x4f, 0x7a, 0x24, 0x00, 0x00, 0x04, 0xc0, 0x0a,
			0x00, 0xff, 0x01, 0x00, 0x00, 0x6f, 0x00, 0x0b,
			0x00, 0x04, 0x03, 0x00, 0x01, 0x02, 0x00, 0x0a,
			0x00, 0x34, 0x00, 0x32, 0x00, 0x0e, 0x00, 0x0d,
			0x00, 0x19, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x18,
			0x00, 0x09, 0x00, 0x0a, 0x00, 0x16, 0x00, 0x17,
			0x00, 0x08, 0x00, 0x06, 0x00, 0x07, 0x00, 0x14,
			0x00, 0x15, 0x00, 0x04, 0x00, 0x05, 0x00, 0x12,
			0x00, 0x13, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03,
			0x00, 0x0f, 0x00, 0x10, 0x00, 0x11, 0x00, 0x23,
			0x00, 0x00, 0x00, 0x0d, 0x00, 0x22, 0x00, 0x20,
			0x06, 0x01, 0x06, 0x02, 0x06, 0x03, 0x05, 0x01,
			0x05, 0x02, 0x05, 0x03, 0x04, 0x01, 0x04, 0x02,
			0x04, 0x03, 0x03, 0x01, 0x03, 0x02, 0x03, 0x03,
			0x02, 0x01, 0x02, 0x02, 0x02, 0x03, 0x01, 0x01,
			0x00, 0x0f, 0x00, 0x01, 0x01,
		},
		{
			0x16, 0x03, 0x01, 0x00, 0x30, 0x02, 0x00, 0x00,
			0x2c, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0xc0, 0x0a, 0x00, 0x00,
			0x04, 0x00, 0x23, 0x00, 0x00, 0x16, 0x03, 0x01,
			0x02, 0x0e, 0x0b, 0x00, 0x02, 0x0a, 0x00, 0x02,
			0x07, 0x00, 0x02, 0x04, 0x30, 0x82, 0x02, 0x00,
			0x30, 0x82, 0x01, 0x62, 0x02, 0x09, 0x00, 0xb8,
			0xbf, 0x2d, 0x47, 0xa0, 0xd2, 0xeb, 0xf4, 0x30,
			0x09, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d,
			0x04, 0x01, 0x30, 0x45, 0x31, 0x0b, 0x30, 0x09,
			0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x41,
			0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55,
			0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d, 0x65,
			0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x21,
			0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13,
			0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65,
			0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69, 0x74,
			0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c, 0x74,
			0x64, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x32, 0x31,
			0x31, 0x32, 0x32, 0x31, 0x35, 0x30, 0x36, 0x33,
			0x32, 0x5a, 0x17, 0x0d, 0x32, 0x32, 0x31, 0x31,
			0x32, 0x30, 0x31, 0x35, 0x30, 0x36, 0x33, 0x32,
			0x5a, 0x30, 0x45, 0x31, 0x0b, 0x30, 0x09, 0x06,
			0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x41, 0x55,
			0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04,
			0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d, 0x65, 0x2d,
			0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x21, 0x30,
			0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x18,
			0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65, 0x74,
			0x20, 0x57, 0x69, 0x64, 0x67, 0x69, 0x74, 0x73,
			0x20, 0x50, 0x74, 0x79, 0x20, 0x4c, 0x74, 0x64,
			0x30, 0x81, 0x9b, 0x30, 0x10, 0x06, 0x07, 0x2a,
			0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x05,
			0x2b, 0x81, 0x04, 0x00, 0x23, 0x03, 0x81, 0x86,
			0x00, 0x04, 0x00, 0xc4, 0xa1, 0xed, 0xbe, 0x98,
			0xf9, 0x0b, 0x48, 0x73, 0x36, 0x7e, 0xc3, 0x16,
			0x56, 0x11, 0x22, 0xf2, 0x3d, 0x53, 0xc3, 0x3b,
			0x4d, 0x21, 0x3d, 0xcd, 0x6b, 0x75, 0xe6, 0xf6,
			0xb0, 0xdc, 0x9a, 0xdf, 0x26, 0xc1, 0xbc, 0xb2,
			0x87, 0xf0, 0x72, 0x32, 0x7c, 0xb3, 0x64, 0x2f,
			0x1c, 0x90, 0xbc, 0xea, 0x68, 0x23, 0x10, 0x7e,
			0xfe, 0xe3, 0x25, 0xc0, 0x48, 0x3a, 0x69, 0xe0,
			0x28, 0x6d, 0xd3, 0x37, 0x00, 0xef, 0x04, 0x62,
			0xdd, 0x0d, 0xa0, 0x9c, 0x70, 0x62, 0x83, 0xd8,
			0x81, 0xd3, 0x64, 0x31, 0xaa, 0x9e, 0x97, 0x31,
			0xbd, 0x96, 0xb0, 0x68, 0xc0, 0x9b, 0x23, 0xde,
			0x76, 0x64, 0x3f, 0x1a, 0x5c, 0x7f, 0xe9, 0x12,
			0x0e, 0x58, 0x58, 0xb6, 0x5f, 0x70, 0xdd, 0x9b,
			0xd8, 0xea, 0xd5, 0xd7, 0xf5, 0xd5, 0xcc, 0xb9,
			0xb6, 0x9f, 0x30, 0x66, 0x5b, 0x66, 0x9a, 0x20,
			0xe2, 0x27, 0xe5, 0xbf, 0xfe, 0x3b, 0x30, 0x09,
			0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04,
			0x01, 0x03, 0x81, 0x8c, 0x00, 0x30, 0x81, 0x88,
			0x02, 0x42, 0x01, 0x88, 0xa2, 0x4f, 0xeb, 0xe2,
			0x45, 0xc5, 0x48, 0x7d, 0x1b, 0xac, 0xf5, 0xed,
			0x98, 0x9d, 0xae, 0x47, 0x70, 0xc0, 0x5e, 0x1b,
			0xb6, 0x2f, 0xbd, 0xf1, 0xb6, 0x4d, 0xb7, 0x61,
			0x40, 0xd3, 0x11, 0xa2, 0xce, 0xee, 0x0b, 0x7e,
			0x92, 0x7e, 0xff, 0x76, 0x9d, 0xc3, 0x3b, 0x7e,
			0xa5, 0x3f, 0xce, 0xfa, 0x10, 0xe2, 0x59, 0xec,
			0x47, 0x2d, 0x7c, 0xac, 0xda, 0x4e, 0x97, 0x0e,
			0x15, 0xa0, 0x6f, 0xd0, 0x02, 0x42, 0x01, 0x4d,
			0xfc, 0xbe, 0x67, 0x13, 0x9c, 0x2d, 0x05, 0x0e,
			0xbd, 0x3f, 0xa3, 0x8c, 0x25, 0xc1, 0x33, 0x13,
			0x83, 0x0d, 0x94, 0x06, 0xbb, 0xd4, 0x37, 0x7a,
			0xf6, 0xec, 0x7a, 0xc9, 0x86, 0x2e, 0xdd, 0xd7,
			0x11, 0x69, 0x7f, 0x85, 0x7c, 0x56, 0xde, 0xfb,
			0x31, 0x78, 0x2b, 0xe4, 0xc7, 0x78, 0x0d, 0xae,
			0xcb, 0xbe, 0x9e, 0x4e, 0x36, 0x24, 0x31, 0x7b,
			0x6a, 0x0f, 0x39, 0x95, 0x12, 0x07, 0x8f, 0x2a,
			0x16, 0x03, 0x01, 0x01, 0x1a, 0x0c, 0x00, 0x01,
			0x16, 0x03, 0x00, 0x19, 0x85, 0x04, 0x01, 0x39,
			0xdc, 0xee, 0x44, 0x17, 0x5e, 0xdb, 0xd7, 0x27,
			0xaf, 0xb6, 0x56, 0xd9, 0xb4, 0x43, 0x5a, 0x99,
			0xcf, 0xaa, 0x31, 0x37, 0x0c, 0x6f, 0x3a, 0xa0,
			0xf8, 0x53, 0xc4, 0x74, 0xd1, 0x91, 0x0a, 0x46,
			0xf5, 0x38, 0x3b, 0x5c, 0x09, 0xd8, 0x97, 0xdc,
			0x4b, 0xaa, 0x70, 0x26, 0x48, 0xf2, 0xd6, 0x0b,
			0x31, 0xc9, 0xf8, 0xd4, 0x98, 0x43, 0xe1, 0x6c,
			0xd5, 0xc7, 0xb2, 0x8e, 0x0b, 0x01, 0xe6, 0xb6,
			0x00, 0x28, 0x80, 0x7b, 0xfc, 0x96, 0x8f, 0x0d,
			0xa2, 0x4f, 0xb0, 0x79, 0xaf, 0xdc, 0x61, 0x28,
			0x63, 0x33, 0x78, 0xf6, 0x31, 0x39, 0xfd, 0x8a,
			0xf4, 0x15, 0x18, 0x11, 0xfe, 0xdb, 0xd5, 0x07,
			0xda, 0x2c, 0xed, 0x49, 0xa0, 0x23, 0xbf, 0xd0,
			0x3a, 0x38, 0x1d, 0x54, 0xae, 0x1c, 0x7b, 0xea,
			0x29, 0xee, 0xd0, 0x38, 0xc1, 0x76, 0xa7, 0x7f,
			0x2a, 0xf4, 0xce, 0x1e, 0xac, 0xcc, 0x94, 0x79,
			0x90, 0x33, 0x00, 0x8b, 0x30, 0x81, 0x88, 0x02,
			0x42, 0x00, 0xc6, 0x85, 0x8e, 0x06, 0xb7, 0x04,
			0x04, 0xe9, 0xcd, 0x9e, 0x3e, 0xcb, 0x66, 0x23,
			0x95, 0xb4, 0x42, 0x9c, 0x64, 0x81, 0x39, 0x05,
			0x3f, 0xb5, 0x21, 0xf8, 0x28, 0xaf, 0x60, 0x6b,
			0x4d, 0x3d, 0xba, 0xa1, 0x4b, 0x5e, 0x77, 0xef,
			0xe7, 0x59, 0x28, 0xfe, 0x1d, 0xc1, 0x27, 0xa2,
			0xff, 0xa8, 0xde, 0x33, 0x48, 0xb3, 0xc1, 0x85,
			0x6a, 0x42, 0x9b, 0xf9, 0x7e, 0x7e, 0x31, 0xc2,
			0xe5, 0xbd, 0x66, 0x02, 0x42, 0x00, 0xad, 0x7d,
			0x06, 0x35, 0xab, 0xec, 0x8d, 0xac, 0xd4, 0xba,
			0x1b, 0x49, 0x5e, 0x05, 0x5f, 0xf0, 0x97, 0x93,
			0x82, 0xb8, 0x2b, 0x8d, 0x91, 0x98, 0x63, 0x8e,
			0xb4, 0x14, 0x62, 0xdb, 0x1e, 0xc9, 0x2b, 0x30,
			0xf8, 0x41, 0x9b, 0xa6, 0xe6, 0xbc, 0xde, 0x0e,
			0x68, 0x30, 0x21, 0xf4, 0xa8, 0xa9, 0x1b, 0xec,
			0x44, 0x4f, 0x5d, 0x02, 0x2f, 0x60, 0x45, 0x60,
			0xba, 0xe0, 0x4e, 0xc0, 0xd4, 0x3b, 0x01, 0x16,
			0x03, 0x01, 0x00, 0x09, 0x0d, 0x00, 0x00, 0x05,
			0x02, 0x01, 0x40, 0x00, 0x00, 0x16, 0x03, 0x01,
			0x00, 0x04, 0x0e, 0x00, 0x00, 0x00,
		},
		{
			0x16, 0x03, 0x01, 0x02, 0x0a, 0x0b, 0x00, 0x02,
			0x06, 0x00, 0x02, 0x03, 0x00, 0x02, 0x00, 0x30,
			0x82, 0x01, 0xfc, 0x30, 0x82, 0x01, 0x5e, 0x02,
			0x09, 0x00, 0x9a, 0x30, 0x84, 0x6c, 0x26, 0x35,
			0xd9, 0x17, 0x30, 0x09, 0x06, 0x07, 0x2a, 0x86,
			0x48, 0xce, 0x3d, 0x04, 0x01, 0x30, 0x45, 0x31,
			0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
			0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11,
			0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53,
			0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74,
			0x65, 0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55,
			0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65,
			0x72, 0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64,
			0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79,
			0x20, 0x4c, 0x74, 0x64, 0x30, 0x1e, 0x17, 0x0d,
			0x31, 0x32, 0x31, 0x31, 0x31, 0x34, 0x31, 0x33,
			0x32, 0x35, 0x35, 0x33, 0x5a, 0x17, 0x0d, 0x32,
			0x32, 0x31, 0x31, 0x31, 0x32, 0x31, 0x33, 0x32,
			0x35, 0x35, 0x33, 0x5a, 0x30, 0x41, 0x31, 0x0b,
			0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
			0x02, 0x41, 0x55, 0x31, 0x0c, 0x30, 0x0a, 0x06,
			0x03, 0x55, 0x04, 0x08, 0x13, 0x03, 0x4e, 0x53,
			0x57, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55,
			0x04, 0x07, 0x13, 0x07, 0x50, 0x79, 0x72, 0x6d,
			0x6f, 0x6e, 0x74, 0x31, 0x12, 0x30, 0x10, 0x06,
			0x03, 0x55, 0x04, 0x03, 0x13, 0x09, 0x4a, 0x6f,
			0x65, 0x6c, 0x20, 0x53, 0x69, 0x6e, 0x67, 0x30,
			0x81, 0x9b, 0x30, 0x10, 0x06, 0x07, 0x2a, 0x86,
			0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x05, 0x2b,
			0x81, 0x04, 0x00, 0x23, 0x03, 0x81, 0x86, 0x00,
			0x04, 0x00, 0x95, 0x8c, 0x91, 0x75, 0x14, 0xc0,
			0x5e, 0xc4, 0x57, 0xb4, 0xd4, 0xc3, 0x6f, 0x8d,
			0xae, 0x68, 0x1e, 0xdd, 0x6f, 0xce, 0x86, 0xe1,
			0x7e, 0x6e, 0xb2, 0x48, 0x3e, 0x81, 0xe5, 0x4e,
			0xe2, 0xc6, 0x88, 0x4b, 0x64, 0xdc, 0xf5, 0x30,
			0xbb, 0xd3, 0xff, 0x65, 0xcc, 0x5b, 0xf4, 0xdd,
			0xb5, 0x6a, 0x3e, 0x3e, 0xd0, 0x1d, 0xde, 0x47,
			0xc3, 0x76, 0xad, 0x19, 0xf6, 0x45, 0x2c, 0x8c,
			0xbc, 0xd8, 0x1d, 0x01, 0x4c, 0x1f, 0x70, 0x90,
			0x46, 0x76, 0x48, 0x8b, 0x8f, 0x83, 0xcc, 0x4a,
			0x5c, 0x8f, 0x40, 0x76, 0xda, 0xe0, 0x89, 0xec,
			0x1d, 0x2b, 0xc4, 0x4e, 0x30, 0x76, 0x28, 0x41,
			0xb2, 0x62, 0xa8, 0xfb, 0x5b, 0xf1, 0xf9, 0x4e,
			0x7a, 0x8d, 0xbd, 0x09, 0xb8, 0xae, 0xea, 0x8b,
			0x18, 0x27, 0x4f, 0x2e, 0x70, 0xfe, 0x13, 0x96,
			0xba, 0xc3, 0xd3, 0x40, 0x16, 0xcd, 0x65, 0x4e,
			0xac, 0x11, 0x1e, 0xe6, 0xf1, 0x30, 0x09, 0x06,
			0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x01,
			0x03, 0x81, 0x8c, 0x00, 0x30, 0x81, 0x88, 0x02,
			0x42, 0x00, 0xe0, 0x14, 0xc4, 0x60, 0x60, 0x0b,
			0x72, 0x68, 0xb0, 0x32, 0x5d, 0x61, 0x4a, 0x02,
			0x74, 0x5c, 0xc2, 0x81, 0xb9, 0x16, 0xa8, 0x3f,
			0x29, 0xc8, 0x36, 0xc7, 0x81, 0xff, 0x6c, 0xb6,
			0x5b, 0xd9, 0x70, 0xf1, 0x38, 0x3b, 0x50, 0x48,
			0x28, 0x94, 0xcb, 0x09, 0x1a, 0x52, 0xf1, 0x5d,
			0xee, 0x8d, 0xf2, 0xb9, 0xf0, 0xf0, 0xda, 0xd9,
			0x15, 0x3a, 0xf9, 0xbd, 0x03, 0x7a, 0x87, 0xa2,
			0x23, 0x35, 0xec, 0x02, 0x42, 0x01, 0xa3, 0xd4,
			0x8a, 0x78, 0x35, 0x1c, 0x4a, 0x9a, 0x23, 0xd2,
			0x0a, 0xbe, 0x2b, 0x10, 0x31, 0x9d, 0x9c, 0x5f,
			0xbe, 0xe8, 0x91, 0xb3, 0xda, 0x1a, 0xf5, 0x5d,
			0xa3, 0x23, 0xf5, 0x26, 0x8b, 0x45, 0x70, 0x8d,
			0x65, 0x62, 0x9b, 0x7e, 0x01, 0x99, 0x3d, 0x18,
			0xf6, 0x10, 0x9a, 0x38, 0x61, 0x9b, 0x2e, 0x57,
			0xe4, 0xfa, 0xcc, 0xb1, 0x8a, 0xce, 0xe2, 0x23,
			0xa0, 0x87, 0xf0, 0xe1, 0x67, 0x51, 0xeb, 0x16,
			0x03, 0x01, 0x00, 0x8a, 0x10, 0x00, 0x00, 0x86,
			0x85, 0x04, 0x00, 0xcd, 0x1c, 0xe8, 0x66, 0x5b,
			0xa8, 0x9d, 0x83, 0x2f, 0x7e, 0x1d, 0x0b, 0x59,
			0x23, 0xbc, 0x30, 0xcf, 0xa3, 0xaf, 0x21, 0xdc,
			0xf2, 0x57, 0x49, 0x56, 0x30, 0x25, 0x7c, 0x84,
			0x5d, 0xad, 0xaa, 0x9c, 0x7b, 0x2a, 0x95, 0x58,
			0x3d, 0x30, 0x87, 0x01, 0x3b, 0xb7, 0xea, 0xcb,
			0xc4, 0xa3, 0xeb, 0x22, 0xbf, 0x2d, 0x61, 0x17,
			0x8c, 0x9b, 0xe8, 0x1b, 0xb2, 0x87, 0x16, 0x78,
			0xd5, 0xfd, 0x8b, 0xdd, 0x00, 0x0f, 0xda, 0x8e,
			0xfd, 0x28, 0x36, 0xeb, 0xe4, 0xc5, 0x42, 0x14,
			0xc7, 0xbd, 0x29, 0x5e, 0x9a, 0xed, 0x5e, 0xc1,
			0xf7, 0xf4, 0xbd, 0xbd, 0x15, 0x9c, 0xe8, 0x44,
			0x71, 0xa7, 0xb6, 0xe9, 0xfa, 0x7e, 0x97, 0xcb,
			0x96, 0x3e, 0x53, 0x76, 0xfb, 0x11, 0x1f, 0x36,
			0x8f, 0x30, 0xfb, 0x71, 0x3a, 0x75, 0x3a, 0x25,
			0x7b, 0xa2, 0xc1, 0xf9, 0x3e, 0x58, 0x5f, 0x07,
			0x16, 0xed, 0xe1, 0xf7, 0xc1, 0xb1, 0x16, 0x03,
			0x01, 0x00, 0x90, 0x0f, 0x00, 0x00, 0x8c, 0x00,
			0x8a, 0x30, 0x81, 0x87, 0x02, 0x42, 0x00, 0xb2,
			0xd3, 0x91, 0xe6, 0xd5, 0x9b, 0xb2, 0xb8, 0x03,
			0xf4, 0x85, 0x4d, 0x43, 0x79, 0x1f, 0xb6, 0x6f,
			0x0c, 0xcd, 0x67, 0x5f, 0x5e, 0xca, 0xee, 0xb3,
			0xe4, 0xab, 0x1e, 0x58, 0xc3, 0x04, 0xa9, 0x8a,
			0xa7, 0xcf, 0xaa, 0x33, 0x88, 0xd5, 0x35, 0xd2,
			0x80, 0x8f, 0xfa, 0x1b, 0x3c, 0x3d, 0xf7, 0x80,
			0x50, 0xde, 0x80, 0x30, 0x64, 0xee, 0xc0, 0xb3,
			0x91, 0x6e, 0x5d, 0x1e, 0xc0, 0xdc, 0x3a, 0x93,
			0x02, 0x41, 0x4e, 0xca, 0x98, 0x41, 0x8c, 0x36,
			0xf2, 0x12, 0xbf, 0x8e, 0x0f, 0x69, 0x8e, 0xf8,
			0x7b, 0x9d, 0xba, 0x9c, 0x5c, 0x48, 0x79, 0xf4,
			0xba, 0x3d, 0x06, 0xa5, 0xab, 0x47, 0xe0, 0x1a,
			0x45, 0x28, 0x3a, 0x8f, 0xbf, 0x14, 0x24, 0x36,
			0xd1, 0x1d, 0x29, 0xdc, 0xde, 0x72, 0x5b, 0x76,
			0x41, 0x67, 0xe8, 0xe5, 0x71, 0x4a, 0x77, 0xe9,
			0xed, 0x02, 0x19, 0xdd, 0xe4, 0xaa, 0xe9, 0x2d,
			0xe7, 0x47, 0x32, 0x14, 0x03, 0x01, 0x00, 0x01,
			0x01, 0x16, 0x03, 0x01, 0x00, 0x30, 0xfa, 0xc3,
			0xf2, 0x35, 0xd0, 0x6d, 0x32, 0x78, 0x6a, 0xd6,
			0xe6, 0x70, 0x5e, 0x00, 0x4c, 0x35, 0xf1, 0xe0,
			0x21, 0xcf, 0xc3, 0x78, 0xcd, 0xe0, 0x2b, 0x0b,
			0xf4, 0xeb, 0xf9, 0xc0, 0x38, 0xf2, 0x9a, 0x31,
			0x55, 0x07, 0x2b, 0x8d, 0x68, 0x40, 0x31, 0x08,
			0xaa, 0xe3, 0x16, 0xcf, 0x4b, 0xd4,
		},
		{
			0x16, 0x03, 0x01, 0x02, 0x76, 0x04, 0x00, 0x02,
			0x72, 0x00, 0x00, 0x00, 0x00, 0x02, 0x6c, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x65,
			0xe8, 0x8b, 0xde, 0xef, 0xba, 0xf9, 0xdb, 0x95,
			0x24, 0xa5, 0x49, 0xb3, 0x23, 0xd8, 0x73, 0x88,
			0x50, 0x42, 0xed, 0xeb, 0xa3, 0xd8, 0xab, 0x31,
			0x9c, 0xd0, 0x00, 0x01, 0xef, 0xc0, 0xbf, 0xab,
			0x59, 0x55, 0xb5, 0xb9, 0xef, 0xa5, 0xa6, 0xec,
			0x69, 0xed, 0x00, 0x2f, 0x47, 0xdb, 0x75, 0x52,
			0x0c, 0xe5, 0x86, 0xb7, 0x02, 0x59, 0x22, 0xf7,
			0xfd, 0x8b, 0xff, 0xa4, 0x09, 0xc0, 0x1c, 0x10,
			0x80, 0x10, 0x7f, 0x4c, 0x7a, 0x94, 0x40, 0x10,
			0x0d, 0xda, 0x8a, 0xe5, 0x4a, 0xbc, 0xd0, 0xc0,
			0x4b, 0xa5, 0x33, 0x97, 0xc6, 0xe7, 0x40, 0x7f,
			0x7f, 0x8c, 0xf9, 0xf8, 0xc8, 0xb8, 0xfb, 0x8c,
			0xdd, 0x28, 0x81, 0xae, 0xfd, 0x37, 0x20, 0x3a,
			0x40, 0x37, 0x99, 0xc4, 0x21, 0x01, 0xc4, 0x91,
			0xb0, 0x5e, 0x11, 0xc5, 0xa9, 0xfd, 0x9a, 0x02,
			0x7e, 0x97, 0x6a, 0x86, 0x89, 0xb8, 0xc1, 0x32,
			0x4c, 0x7e, 0x6d, 0x47, 0x61, 0x0e, 0xe3, 0xc2,
			0xf0, 0x62, 0x3c, 0xc6, 0x71, 0x4f, 0xbb, 0x47,
			0x65, 0xb1, 0xd9, 0x22, 0x79, 0x15, 0xea, 0x1f,
			0x4b, 0x2a, 0x8a, 0xa4, 0xc8, 0x73, 0x34, 0xba,
			0x83, 0xe4, 0x70, 0x99, 0xc9, 0xcf, 0xbe, 0x64,
			0x99, 0xb9, 0xfa, 0xe9, 0xaf, 0x5d, 0xc7, 0x20,
			0x26, 0xde, 0xc5, 0x06, 0x12, 0x36, 0x4f, 0x4d,
			0xc0, 0xbb, 0x81, 0x5b, 0x5e, 0x38, 0xc3, 0x07,
			0x21, 0x04, 0x1a, 0x53, 0x9c, 0x59, 0xac, 0x2d,
			0xe6, 0xa5, 0x93, 0xa5, 0x19, 0xc6, 0xb0, 0xf7,
			0x56, 0x5d, 0xdf, 0xd1, 0xf4, 0xfd, 0x44, 0x6d,
			0xc6, 0xa2, 0x31, 0xa7, 0x35, 0x42, 0x18, 0x50,
			0x0c, 0x4f, 0x6e, 0xe3, 0x3b, 0xa3, 0xaa, 0x1c,
			0xbe, 0x41, 0x0d, 0xce, 0x6c, 0x62, 0xe1, 0x96,
			0x2d, 0xbd, 0x14, 0x31, 0xe3, 0xc4, 0x5b, 0xbf,
			0xf6, 0xde, 0xec, 0x42, 0xe8, 0xc7, 0x2a, 0x0b,
			0xdb, 0x2d, 0x7c, 0xf0, 0x3f, 0x45, 0x32, 0x45,
			0x09, 0x47, 0x09, 0x0f, 0x21, 0x22, 0x45, 0x06,
			0x11, 0xb8, 0xf9, 0xe6, 0x67, 0x90, 0x4b, 0x4a,
			0xde, 0x81, 0xfb, 0xeb, 0xe7, 0x9a, 0x08, 0x30,
			0xcf, 0x51, 0xe1, 0xd9, 0xfa, 0x79, 0xa3, 0xcc,
			0x65, 0x1a, 0x83, 0x86, 0xc9, 0x7a, 0x41, 0xf5,
			0xdf, 0xa0, 0x7c, 0x44, 0x23, 0x17, 0xf3, 0x62,
			0xe8, 0xa9, 0x31, 0x1e, 0x6b, 0x05, 0x4b, 0x4f,
			0x9d, 0x91, 0x46, 0x92, 0xa6, 0x25, 0x32, 0xca,
			0xa1, 0x75, 0xda, 0xe6, 0x80, 0x3e, 0x7f, 0xd1,
			0x26, 0x57, 0x07, 0x42, 0xe4, 0x91, 0xff, 0xbd,
			0x44, 0xae, 0x98, 0x5c, 0x1d, 0xdf, 0x11, 0xe3,
			0xae, 0x87, 0x5e, 0xb7, 0x69, 0xad, 0x34, 0x7f,
			0x3a, 0x07, 0x7c, 0xdf, 0xfc, 0x76, 0x17, 0x8b,
			0x62, 0xc8, 0xe1, 0x78, 0x2a, 0xc8, 0xb9, 0x8a,
			0xbb, 0x5c, 0xfb, 0x38, 0x74, 0x91, 0x6e, 0x12,
			0x0c, 0x1f, 0x8e, 0xe1, 0xc2, 0x01, 0xd8, 0x9d,
			0x23, 0x0f, 0xc4, 0x67, 0x5d, 0xe5, 0x67, 0x4b,
			0x94, 0x6e, 0x69, 0x72, 0x90, 0x2d, 0x52, 0x78,
			0x8e, 0x61, 0xba, 0xdf, 0x4e, 0xf5, 0xdc, 0xfb,
			0x73, 0xbe, 0x03, 0x70, 0xd9, 0x01, 0x30, 0xf3,
			0xa1, 0xbb, 0x9a, 0x5f, 0xec, 0x9e, 0xed, 0x8d,
			0xdd, 0x53, 0xfd, 0x60, 0xc3, 0x2b, 0x7a, 0x00,
			0x2c, 0xf9, 0x0a, 0x57, 0x47, 0x45, 0x43, 0xb3,
			0x23, 0x01, 0x9c, 0xee, 0x54, 0x4d, 0x58, 0xd3,
			0x71, 0x1c, 0xc9, 0xd3, 0x30, 0x9e, 0x14, 0xa5,
			0xf3, 0xbf, 0x4d, 0x9b, 0xb7, 0x13, 0x21, 0xae,
			0xd2, 0x8d, 0x6e, 0x6f, 0x1c, 0xcc, 0xb2, 0x41,
			0xb2, 0x64, 0x56, 0x83, 0xce, 0xd1, 0x0c, 0x79,
			0x32, 0x78, 0xef, 0xc5, 0x21, 0xb1, 0xe8, 0xc4,
			0x42, 0xa7, 0x8d, 0xc1, 0xfa, 0xa1, 0x9c, 0x3c,
			0x21, 0xd8, 0xe9, 0x90, 0xe2, 0x7c, 0x14, 0x26,
			0xfe, 0x61, 0x3e, 0xf9, 0x71, 0x1d, 0x5d, 0x49,
			0x3b, 0xb1, 0xb8, 0x42, 0xa1, 0xb8, 0x1c, 0x75,
			0x7d, 0xee, 0xed, 0xfc, 0xe6, 0x20, 0x2b, 0x9e,
			0x10, 0x52, 0xda, 0x56, 0x4d, 0x64, 0x6c, 0x41,
			0xc1, 0xf7, 0x60, 0x0c, 0x10, 0x65, 0x6f, 0xd4,
			0xe9, 0x9b, 0x0d, 0x83, 0x13, 0xc8, 0x5a, 0xa3,
			0x56, 0x2a, 0x42, 0xc6, 0x1c, 0xfe, 0xdb, 0xba,
			0x3d, 0x04, 0x12, 0xfd, 0x28, 0xeb, 0x78, 0xdd,
			0xbc, 0xc8, 0x0d, 0xa1, 0xce, 0xd4, 0x54, 0xbf,
			0xaf, 0xe1, 0x60, 0x0c, 0xa3, 0xc3, 0xc3, 0x62,
			0x58, 0xc1, 0x79, 0xa7, 0x95, 0x41, 0x09, 0x24,
			0xc6, 0x9a, 0x50, 0x14, 0x03, 0x01, 0x00, 0x01,
			0x01, 0x16, 0x03, 0x01, 0x00, 0x30, 0x4d, 0x7b,
			0x5f, 0x28, 0x5e, 0x68, 0x6c, 0xa3, 0x65, 0xc7,
			0x7e, 0x49, 0x6c, 0xb3, 0x67, 0xbb, 0xd0, 0x75,
			0xa2, 0x9e, 0x8c, 0x92, 0x4f, 0x8c, 0x33, 0x14,
			0x7c, 0x6c, 0xf1, 0x74, 0x97, 0xc3, 0xe0, 0x10,
			0xe9, 0x0d, 0xc2, 0x30, 0x5c, 0x23, 0xee, 0x1d,
			0x16, 0x2e, 0xb9, 0x96, 0x2b, 0x2d, 0x17, 0x03,
			0x01, 0x00, 0x20, 0xf2, 0xc8, 0xa7, 0x1b, 0x60,
			0x46, 0xee, 0xe5, 0x7e, 0xc9, 0x35, 0xb3, 0xf1,
			0x7c, 0x32, 0x0c, 0x85, 0x94, 0x59, 0x57, 0x27,
			0xb0, 0xbd, 0x52, 0x86, 0x90, 0xf1, 0xb7, 0x4d,
			0x1e, 0xc1, 0x16, 0x17, 0x03, 0x01, 0x00, 0x30,
			0xff, 0x85, 0x50, 0xdf, 0x3f, 0xfc, 0xa2, 0x61,
			0x1a, 0x12, 0xc0, 0x1e, 0x10, 0x32, 0x88, 0x50,
			0xa0, 0x2c, 0x80, 0xda, 0x77, 0xea, 0x09, 0x47,
			0xe0, 0x85, 0x07, 0x29, 0x45, 0x65, 0x19, 0xa3,
			0x8d, 0x99, 0xb8, 0xbf, 0xb6, 0xbc, 0x76, 0xe2,
			0x50, 0x24, 0x82, 0x0a, 0xfd, 0xdd, 0x35, 0x09,
			0x15, 0x03, 0x01, 0x00, 0x20, 0xe7, 0x36, 0xf6,
			0x61, 0xd2, 0x95, 0x3c, 0xb6, 0x65, 0x7b, 0xb2,
			0xb8, 0xdf, 0x03, 0x53, 0xeb, 0xf7, 0x16, 0xe0,
			0xe0, 0x15, 0x22, 0x71, 0x70, 0x62, 0x73, 0xad,
			0xb5, 0x1a, 0x77, 0x44, 0x57,
		},
	}},
}

var aesGCMServerScript = [][]byte{
	{
		0x16, 0x03, 0x01, 0x01, 0x1c, 0x01, 0x00, 0x01,
		0x18, 0x03, 0x03, 0x52, 0x1e, 0x74, 0xf0, 0xb0,
		0xc1, 0x8b, 0x16, 0xf9, 0x74, 0xfc, 0x16, 0xc4,
		0x11, 0x18, 0x96, 0x08, 0x25, 0x38, 0x4f, 0x98,
		0x98, 0xbe, 0xb5, 0x61, 0xdf, 0x94, 0x15, 0xcc,
		0x9b, 0x61, 0xef, 0x00, 0x00, 0x80, 0xc0, 0x30,
		0xc0, 0x2c, 0xc0, 0x28, 0xc0, 0x24, 0xc0, 0x14,
		0xc0, 0x0a, 0x00, 0xa3, 0x00, 0x9f, 0x00, 0x6b,
		0x00, 0x6a, 0x00, 0x39, 0x00, 0x38, 0xc0, 0x32,
		0xc0, 0x2e, 0xc0, 0x2a, 0xc0, 0x26, 0xc0, 0x0f,
		0xc0, 0x05, 0x00, 0x9d, 0x00, 0x3d, 0x00, 0x35,
		0xc0, 0x12, 0xc0, 0x08, 0x00, 0x16, 0x00, 0x13,
		0xc0, 0x0d, 0xc0, 0x03, 0x00, 0x0a, 0xc0, 0x2f,
		0xc0, 0x2b, 0xc0, 0x27, 0xc0, 0x23, 0xc0, 0x13,
		0xc0, 0x09, 0x00, 0xa2, 0x00, 0x9e, 0x00, 0x67,
		0x00, 0x40, 0x00, 0x33, 0x00, 0x32, 0xc0, 0x31,
		0xc0, 0x2d, 0xc0, 0x29, 0xc0, 0x25, 0xc0, 0x0e,
		0xc0, 0x04, 0x00, 0x9c, 0x00, 0x3c, 0x00, 0x2f,
		0xc0, 0x11, 0xc0, 0x07, 0xc0, 0x0c, 0xc0, 0x02,
		0x00, 0x05, 0x00, 0x04, 0x00, 0x15, 0x00, 0x12,
		0x00, 0x09, 0x00, 0x14, 0x00, 0x11, 0x00, 0x08,
		0x00, 0x06, 0x00, 0x03, 0x00, 0xff, 0x01, 0x00,
		0x00, 0x6f, 0x00, 0x0b, 0x00, 0x04, 0x03, 0x00,
		0x01, 0x02, 0x00, 0x0a, 0x00, 0x34, 0x00, 0x32,
		0x00, 0x0e, 0x00, 0x0d, 0x00, 0x19, 0x00, 0x0b,
		0x00, 0x0c, 0x00, 0x18, 0x00, 0x09, 0x00, 0x0a,
		0x00, 0x16, 0x00, 0x17, 0x00, 0x08, 0x00, 0x06,
		0x00, 0x07, 0x00, 0x14, 0x00, 0x15, 0x00, 0x04,
		0x00, 0x05, 0x00, 0x12, 0x00, 0x13, 0x00, 0x01,
		0x00, 0x02, 0x00, 0x03, 0x00, 0x0f, 0x00, 0x10,
		0x00, 0x11, 0x00, 0x23, 0x00, 0x00, 0x00, 0x0d,
		0x00, 0x22, 0x00, 0x20, 0x06, 0x01, 0x06, 0x02,
		0x06, 0x03, 0x05, 0x01, 0x05, 0x02, 0x05, 0x03,
		0x04, 0x01, 0x04, 0x02, 0x04, 0x03, 0x03, 0x01,
		0x03, 0x02, 0x03, 0x03, 0x02, 0x01, 0x02, 0x02,
		0x02, 0x03, 0x01, 0x01, 0x00, 0x0f, 0x00, 0x01,
		0x01,
	},
	{
		0x16, 0x03, 0x03, 0x00, 0x30, 0x02, 0x00, 0x00,
		0x2c, 0x03, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0xc0, 0x2f, 0x00, 0x00,
		0x04, 0x00, 0x23, 0x00, 0x00, 0x16, 0x03, 0x03,
		0x02, 0xbe, 0x0b, 0x00, 0x02, 0xba, 0x00, 0x02,
		0xb7, 0x00, 0x02, 0xb4, 0x30, 0x82, 0x02, 0xb0,
		0x30, 0x82, 0x02, 0x19, 0xa0, 0x03, 0x02, 0x01,
		0x02, 0x02, 0x09, 0x00, 0x85, 0xb0, 0xbb, 0xa4,
		0x8a, 0x7f, 0xb8, 0xca, 0x30, 0x0d, 0x06, 0x09,
		0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
		0x05, 0x05, 0x00, 0x30, 0x45, 0x31, 0x0b, 0x30,
		0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
		0x41, 0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03,
		0x55, 0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d,
		0x65, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31,
		0x21, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a,
		0x13, 0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e,
		0x65, 0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69,
		0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c,
		0x74, 0x64, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x30,
		0x30, 0x34, 0x32, 0x34, 0x30, 0x39, 0x30, 0x39,
		0x33, 0x38, 0x5a, 0x17, 0x0d, 0x31, 0x31, 0x30,
		0x34, 0x32, 0x34, 0x30, 0x39, 0x30, 0x39, 0x33,
		0x38, 0x5a, 0x30, 0x45, 0x31, 0x0b, 0x30, 0x09,
		0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x41,
		0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55,
		0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d, 0x65,
		0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x21,
		0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13,
		0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65,
		0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69, 0x74,
		0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c, 0x74,
		0x64, 0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09,
		0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
		0x01, 0x05, 0x00, 0x03, 0x81, 0x8d, 0x00, 0x30,
		0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xbb, 0x79,
		0xd6, 0xf5, 0x17, 0xb5, 0xe5, 0xbf, 0x46, 0x10,
		0xd0, 0xdc, 0x69, 0xbe, 0xe6, 0x2b, 0x07, 0x43,
		0x5a, 0xd0, 0x03, 0x2d, 0x8a, 0x7a, 0x43, 0x85,
		0xb7, 0x14, 0x52, 0xe7, 0xa5, 0x65, 0x4c, 0x2c,
		0x78, 0xb8, 0x23, 0x8c, 0xb5, 0xb4, 0x82, 0xe5,
		0xde, 0x1f, 0x95, 0x3b, 0x7e, 0x62, 0xa5, 0x2c,
		0xa5, 0x33, 0xd6, 0xfe, 0x12, 0x5c, 0x7a, 0x56,
		0xfc, 0xf5, 0x06, 0xbf, 0xfa, 0x58, 0x7b, 0x26,
		0x3f, 0xb5, 0xcd, 0x04, 0xd3, 0xd0, 0xc9, 0x21,
		0x96, 0x4a, 0xc7, 0xf4, 0x54, 0x9f, 0x5a, 0xbf,
		0xef, 0x42, 0x71, 0x00, 0xfe, 0x18, 0x99, 0x07,
		0x7f, 0x7e, 0x88, 0x7d, 0x7d, 0xf1, 0x04, 0x39,
		0xc4, 0xa2, 0x2e, 0xdb, 0x51, 0xc9, 0x7c, 0xe3,
		0xc0, 0x4c, 0x3b, 0x32, 0x66, 0x01, 0xcf, 0xaf,
		0xb1, 0x1d, 0xb8, 0x71, 0x9a, 0x1d, 0xdb, 0xdb,
		0x89, 0x6b, 0xae, 0xda, 0x2d, 0x79, 0x02, 0x03,
		0x01, 0x00, 0x01, 0xa3, 0x81, 0xa7, 0x30, 0x81,
		0xa4, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e,
		0x04, 0x16, 0x04, 0x14, 0xb1, 0xad, 0xe2, 0x85,
		0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69, 0xce, 0x23,
		0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18, 0x88, 0x39,
		0x30, 0x75, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04,
		0x6e, 0x30, 0x6c, 0x80, 0x14, 0xb1, 0xad, 0xe2,
		0x85, 0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69, 0xce,
		0x23, 0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18, 0x88,
		0x39, 0xa1, 0x49, 0xa4, 0x47, 0x30, 0x45, 0x31,
		0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
		0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11,
		0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53,
		0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74,
		0x65, 0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55,
		0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65,
		0x72, 0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64,
		0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79,
		0x20, 0x4c, 0x74, 0x64, 0x82, 0x09, 0x00, 0x85,
		0xb0, 0xbb, 0xa4, 0x8a, 0x7f, 0xb8, 0xca, 0x30,
		0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x05,
		0x30, 0x03, 0x01, 0x01, 0xff, 0x30, 0x0d, 0x06,
		0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
		0x01, 0x05, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00,
		0x08, 0x6c, 0x45, 0x24, 0xc7, 0x6b, 0xb1, 0x59,
		0xab, 0x0c, 0x52, 0xcc, 0xf2, 0xb0, 0x14, 0xd7,
		0x87, 0x9d, 0x7a, 0x64, 0x75, 0xb5, 0x5a, 0x95,
		0x66, 0xe4, 0xc5, 0x2b, 0x8e, 0xae, 0x12, 0x66,
		0x1f, 0xeb, 0x4f, 0x38, 0xb3, 0x6e, 0x60, 0xd3,
		0x92, 0xfd, 0xf7, 0x41, 0x08, 0xb5, 0x25, 0x13,
		0xb1, 0x18, 0x7a, 0x24, 0xfb, 0x30, 0x1d, 0xba,
		0xed, 0x98, 0xb9, 0x17, 0xec, 0xe7, 0xd7, 0x31,
		0x59, 0xdb, 0x95, 0xd3, 0x1d, 0x78, 0xea, 0x50,
		0x56, 0x5c, 0xd5, 0x82, 0x5a, 0x2d, 0x5a, 0x5f,
		0x33, 0xc4, 0xb6, 0xd8, 0xc9, 0x75, 0x90, 0x96,
		0x8c, 0x0f, 0x52, 0x98, 0xb5, 0xcd, 0x98, 0x1f,
		0x89, 0x20, 0x5f, 0xf2, 0xa0, 0x1c, 0xa3, 0x1b,
		0x96, 0x94, 0xdd, 0xa9, 0xfd, 0x57, 0xe9, 0x70,
		0xe8, 0x26, 0x6d, 0x71, 0x99, 0x9b, 0x26, 0x6e,
		0x38, 0x50, 0x29, 0x6c, 0x90, 0xa7, 0xbd, 0xd9,
		0x16, 0x03, 0x03, 0x01, 0x11, 0x0c, 0x00, 0x01,
		0x0d, 0x03, 0x00, 0x19, 0x85, 0x04, 0x01, 0x39,
		0xdc, 0xee, 0x44, 0x17, 0x5e, 0xdb, 0xd7, 0x27,
		0xaf, 0xb6, 0x56, 0xd9, 0xb4, 0x43, 0x5a, 0x99,
		0xcf, 0xaa, 0x31, 0x37, 0x0c, 0x6f, 0x3a, 0xa0,
		0xf8, 0x53, 0xc4, 0x74, 0xd1, 0x91, 0x0a, 0x46,
		0xf5, 0x38, 0x3b, 0x5c, 0x09, 0xd8, 0x97, 0xdc,
		0x4b, 0xaa, 0x70, 0x26, 0x48, 0xf2, 0xd6, 0x0b,
		0x31, 0xc9, 0xf8, 0xd4, 0x98, 0x43, 0xe1, 0x6c,
		0xd5, 0xc7, 0xb2, 0x8e, 0x0b, 0x01, 0xe6, 0xb6,
		0x00, 0x28, 0x80, 0x7b, 0xfc, 0x96, 0x8f, 0x0d,
		0xa2, 0x4f, 0xb0, 0x79, 0xaf, 0xdc, 0x61, 0x28,
		0x63, 0x33, 0x78, 0xf6, 0x31, 0x39, 0xfd, 0x8a,
		0xf4, 0x15, 0x18, 0x11, 0xfe, 0xdb, 0xd5, 0x07,
		0xda, 0x2c, 0xed, 0x49, 0xa0, 0x23, 0xbf, 0xd0,
		0x3a, 0x38, 0x1d, 0x54, 0xae, 0x1c, 0x7b, 0xea,
		0x29, 0xee, 0xd0, 0x38, 0xc1, 0x76, 0xa7, 0x7f,
		0x2a, 0xf4, 0xce, 0x1e, 0xac, 0xcc, 0x94, 0x79,
		0x90, 0x33, 0x04, 0x01, 0x00, 0x80, 0x0d, 0x8e,
		0x79, 0xe6, 0x86, 0xf6, 0xb6, 0xfb, 0x6b, 0x6a,
		0xcc, 0x55, 0xe4, 0x80, 0x4d, 0xc5, 0x0c, 0xc6,
		0xa3, 0x9f, 0x1d, 0x39, 0xd2, 0x98, 0x57, 0x31,
		0xa2, 0x90, 0x73, 0xe8, 0xd2, 0xcd, 0xb0, 0x93,
		0x1a, 0x60, 0x0f, 0x38, 0x02, 0x3b, 0x1b, 0x25,
		0x56, 0xec, 0x44, 0xab, 0xbe, 0x2e, 0x0c, 0xc0,
		0x6e, 0x54, 0x91, 0x50, 0xd6, 0xb1, 0xa2, 0x98,
		0x14, 0xa8, 0x35, 0x62, 0x9d, 0xca, 0xfb, 0x0f,
		0x64, 0x2b, 0x05, 0xa0, 0xa0, 0x57, 0xef, 0xcd,
		0x95, 0x45, 0x13, 0x5a, 0x9b, 0x3d, 0xdb, 0x42,
		0x54, 0x7f, 0xb9, 0x17, 0x08, 0x7f, 0xb2, 0xf0,
		0xb1, 0xc3, 0xdf, 0x67, 0x95, 0xe2, 0x73, 0xf2,
		0x76, 0xa3, 0x97, 0xfd, 0x9c, 0x92, 0x4a, 0xdb,
		0x95, 0x1e, 0x91, 0x95, 0xae, 0x3d, 0xae, 0x58,
		0xb5, 0x03, 0x6f, 0x5c, 0x3a, 0x19, 0xab, 0x92,
		0xa5, 0x09, 0x6b, 0x40, 0x61, 0xb0, 0x16, 0x03,
		0x03, 0x00, 0x04, 0x0e, 0x00, 0x00, 0x00,
	},
	{
		0x16, 0x03, 0x03, 0x00, 0x8a, 0x10, 0x00, 0x00,
		0x86, 0x85, 0x04, 0x01, 0xba, 0xb8, 0xad, 0x69,
		0x20, 0x5e, 0xc1, 0x61, 0xc3, 0x0f, 0xb4, 0x30,
		0x64, 0x66, 0x70, 0x96, 0x33, 0x3c, 0x8e, 0x12,
		0x56, 0xbf, 0x6d, 0xb8, 0x6d, 0xc6, 0xba, 0xea,
		0xfc, 0x38, 0xc0, 0x8b, 0x87, 0xa8, 0xf3, 0x87,
		0xa1, 0xd5, 0xb6, 0xb0, 0x72, 0xc7, 0xd4, 0x19,
		0x56, 0xa0, 0x91, 0xe1, 0x45, 0xc7, 0xf1, 0x7d,
		0xb0, 0x1d, 0x78, 0x18, 0xf6, 0x3d, 0xbf, 0x1a,
		0x23, 0x93, 0x0b, 0x19, 0xb1, 0x00, 0x56, 0xc9,
		0x5e, 0x89, 0xd4, 0x9d, 0xd9, 0x5b, 0xe0, 0xb8,
		0xff, 0x2f, 0x7d, 0x93, 0xae, 0x5b, 0xa5, 0x1f,
		0x1f, 0x2b, 0x09, 0xe5, 0xf6, 0x07, 0x26, 0xa3,
		0xed, 0xcb, 0x6a, 0x1a, 0xd6, 0x14, 0x83, 0x9b,
		0xd3, 0x9d, 0x47, 0x1b, 0xf3, 0x72, 0x5f, 0x69,
		0x21, 0x8f, 0xfa, 0x09, 0x38, 0x1a, 0x6b, 0x91,
		0xcf, 0x19, 0x32, 0x54, 0x58, 0x8e, 0xee, 0xaf,
		0xeb, 0x06, 0x9b, 0x3a, 0x34, 0x16, 0x66, 0x14,
		0x03, 0x03, 0x00, 0x01, 0x01, 0x16, 0x03, 0x03,
		0x00, 0x28, 0xc6, 0x96, 0x67, 0x62, 0xcc, 0x47,
		0x01, 0xb5, 0xbd, 0xb7, 0x24, 0xd3, 0xb6, 0xfd,
		0xb8, 0x46, 0xce, 0x82, 0x6d, 0x31, 0x1f, 0x15,
		0x11, 0x8f, 0xed, 0x62, 0x71, 0x5f, 0xae, 0xb6,
		0xa9, 0x0c, 0x24, 0x1d, 0xe8, 0x26, 0x51, 0xca,
		0x7c, 0x42,
	},
	{
		0x16, 0x03, 0x03, 0x00, 0x72, 0x04, 0x00, 0x00,
		0x6e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x65,
		0xea, 0x8b, 0xfb, 0xef, 0xba, 0xc8, 0x88, 0x94,
		0x44, 0x99, 0x5f, 0x02, 0x68, 0x3a, 0x12, 0x67,
		0x7f, 0xb9, 0x39, 0x71, 0x84, 0xe0, 0x30, 0xe6,
		0x90, 0x6c, 0xcf, 0x32, 0x29, 0x29, 0x5c, 0x5a,
		0x8b, 0x7d, 0xaa, 0x11, 0x28, 0x26, 0xb5, 0xce,
		0xd2, 0x88, 0xd5, 0xb0, 0x5f, 0x94, 0x37, 0xa2,
		0x48, 0xd9, 0x53, 0xb2, 0xab, 0x59, 0x23, 0x3d,
		0x81, 0x6e, 0x64, 0x89, 0xca, 0x1a, 0x84, 0x16,
		0xdf, 0x31, 0x10, 0xde, 0x52, 0x7f, 0x50, 0xf3,
		0xd9, 0x27, 0xa0, 0xe8, 0x34, 0x15, 0x9e, 0x11,
		0xdd, 0xba, 0xce, 0x40, 0x17, 0xf3, 0x67, 0x14,
		0x03, 0x03, 0x00, 0x01, 0x01, 0x16, 0x03, 0x03,
		0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x35, 0xcb, 0x17, 0x66, 0xee, 0xfd,
		0x27, 0xdb, 0xb8, 0xa8, 0x8a, 0xf1, 0x56, 0x67,
		0x89, 0x0d, 0x13, 0xac, 0xe2, 0x31, 0xb9, 0xa2,
		0x26, 0xbb, 0x1c, 0xcf, 0xd1, 0xb2, 0x48, 0x1d,
		0x0d, 0xb1, 0x17, 0x03, 0x03, 0x00, 0x25, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xc0,
		0x89, 0x7c, 0x58, 0x6a, 0x9b, 0x00, 0x05, 0x8c,
		0x7f, 0x28, 0x54, 0x61, 0x44, 0x10, 0xee, 0x85,
		0x26, 0xa8, 0x04, 0xcd, 0xca, 0x85, 0x60, 0xf2,
		0xeb, 0x22, 0xbd, 0x9e, 0x15, 0x03, 0x03, 0x00,
		0x1a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x02, 0x10, 0xe4, 0xe5, 0xf9, 0x85, 0xe3, 0xb0,
		0xec, 0x84, 0x29, 0x91, 0x05, 0x7d, 0x86, 0xe3,
		0x97, 0xeb, 0xb2,
	},
}
