/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* libe-book
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libebook.sourceforge.net
 */

#include <cstdio>

#include "libebook_utils.h"

using std::string;

namespace libebook
{

namespace
{

void checkStream(WPXInputStream *const input)
{
  if (!input || input->atEOS())
    throw EndOfStreamException();
}

struct SeekFailedException {};

}

uint8_t readU8(WPXInputStream *input, bool /* bigEndian */)
{
  checkStream(input);

  unsigned long numBytesRead;
  uint8_t const *p = input->read(sizeof(uint8_t), numBytesRead);

  if (p && numBytesRead == sizeof(uint8_t))
    return *(uint8_t const *)(p);
  throw EndOfStreamException();
}

uint16_t readU16(WPXInputStream *input, bool bigEndian)
{
  checkStream(input);

  unsigned long numBytesRead;
  uint8_t const *p = input->read(sizeof(uint16_t), numBytesRead);

  if (p && numBytesRead == sizeof(uint16_t))
  {
    if (bigEndian)
      return static_cast<uint16_t>((uint16_t)p[1]|((uint16_t)p[0]<<8));
    return static_cast<uint16_t>((uint16_t)p[0]|((uint16_t)p[1]<<8));
  }
  throw EndOfStreamException();
}

uint32_t readU32(WPXInputStream *input, bool bigEndian)
{
  checkStream(input);

  unsigned long numBytesRead;
  uint8_t const *p = input->read(sizeof(uint32_t), numBytesRead);

  if (p && numBytesRead == sizeof(uint32_t))
  {
    if (bigEndian)
      return (uint32_t)p[3]|((uint32_t)p[2]<<8)|((uint32_t)p[1]<<16)|((uint32_t)p[0]<<24);
    return (uint32_t)p[0]|((uint32_t)p[1]<<8)|((uint32_t)p[2]<<16)|((uint32_t)p[3]<<24);
  }
  throw EndOfStreamException();
}

uint64_t readU64(WPXInputStream *input, bool bigEndian)
{
  checkStream(input);

  unsigned long numBytesRead;
  uint8_t const *p = input->read(sizeof(uint64_t), numBytesRead);

  if (p && numBytesRead == sizeof(uint64_t))
  {
    if (bigEndian)
      return (uint64_t)p[7]|((uint64_t)p[6]<<8)|((uint64_t)p[5]<<16)|((uint64_t)p[4]<<24)|((uint64_t)p[3]<<32)|((uint64_t)p[2]<<40)|((uint64_t)p[1]<<48)|((uint64_t)p[0]<<56);
    return (uint64_t)p[0]|((uint64_t)p[1]<<8)|((uint64_t)p[2]<<16)|((uint64_t)p[3]<<24)|((uint64_t)p[4]<<32)|((uint64_t)p[5]<<40)|((uint64_t)p[6]<<48)|((uint64_t)p[7]<<56);
  }
  throw EndOfStreamException();
}

const unsigned char *readNBytes(WPXInputStream *const input, const unsigned long numBytes)
{
  checkStream(input);

  unsigned long readBytes = 0;
  const unsigned char *const s = input->read(numBytes, readBytes);

  if (numBytes != readBytes)
    throw EndOfStreamException();

  return s;
}

string readCString(WPXInputStream *input)
{
  checkStream(input);

  string str;
  unsigned char c = readU8(input);
  while (0 != c)
  {
    str.push_back(c);
    c = readU8(input);
  }

  return str;
}

string readPascalString(WPXInputStream *input)
{
  checkStream(input);

  const unsigned length = readU8(input);
  string str;
  for (unsigned i = 0; length != i; ++i)
    str.push_back(readU8(input));

  return str;
}

void skip(WPXInputStream *input, unsigned long numBytes)
{
  checkStream(input);

  seekRelative(input, static_cast<long>(numBytes));
}

void seek(WPXInputStream *const input, const unsigned long pos)
{
  checkStream(input);

  if (0 != input->seek(static_cast<long>(pos), WPX_SEEK_SET))
    throw SeekFailedException();
}

void seekRelative(WPXInputStream *const input, const long pos)
{
  checkStream(input);

  if (0 != input->seek(pos, WPX_SEEK_CUR))
    throw SeekFailedException();
}

unsigned long getLength(WPXInputStream *const input)
{
  checkStream(input);

  const unsigned long begin = input->tell();
  unsigned long end = begin;

  if (0 == input->seek(0, WPX_SEEK_END))
    end = input->tell();
  else
  {
    // WPX_SEEK_END does not work. Use the harder way.
    while (!input->atEOS())
    {
      readU8(input);
      ++end;
    }
  }

  seek(input, begin);

  return end - begin;
}

uint8_t readU8(const boost::shared_ptr<WPXInputStream> input, bool)
{
  return readU8(input.get());
}

uint16_t readU16(const boost::shared_ptr<WPXInputStream> input, const bool bigEndian)
{
  return readU16(input.get(), bigEndian);
}

uint32_t readU32(const boost::shared_ptr<WPXInputStream> input, const bool bigEndian)
{
  return readU32(input.get(), bigEndian);
}

uint64_t readU64(const boost::shared_ptr<WPXInputStream> input, const bool bigEndian)
{
  return readU64(input.get(), bigEndian);
}

const unsigned char *readNBytes(const boost::shared_ptr<WPXInputStream> input, const unsigned long numBytes)
{
  return readNBytes(input.get(), numBytes);
}

std::string readCString(const boost::shared_ptr<WPXInputStream> input)
{
  return readCString(input.get());
}

std::string readPascalString(const boost::shared_ptr<WPXInputStream> input)
{
  return readPascalString(input.get());
}

void skip(const boost::shared_ptr<WPXInputStream> input, const unsigned long numBytes)
{
  return skip(input.get(), numBytes);
}

void seek(const boost::shared_ptr<WPXInputStream> input, const unsigned long pos)
{
  seek(input.get(), pos);
}

void seekRelative(const boost::shared_ptr<WPXInputStream> input, const long pos)
{
  seekRelative(input.get(), pos);
}

unsigned long getLength(const boost::shared_ptr<WPXInputStream> input)
{
  return getLength(input.get());
}

EndOfStreamException::EndOfStreamException()
{
  EBOOK_DEBUG_MSG(("Throwing EndOfStreamException\n"));
}

}

/* vim:set shiftwidth=2 softtabstop=2 expandtab: */
