/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/stream_test_system.t.cpp
 *
 * @brief [LEVEL: beta] testing @ref diagnostics::unittest::Interactive_Test_Data_Adaptor
 *
 * $Id: stream_test_system.t.cpp,v 1.7 2005/06/23 09:54:26 esdentem Exp $
 *
 * @author Christian Schallhart
 *
 * @todo more tests
 *
 */
#include <diagnostics/unittest.hpp>

#include <diagnostics/unittest/test_system/stream_test_system.hpp>

#include <diagnostics/unittest/test_system_exception.hpp>

#include <diagnostics/util/dummy_test_case.ts.hpp>
#include <diagnostics/unittest/test_system/current_test_logger.hpp>
#include <diagnostics/util/blind_test_run_result.ts.hpp>
#include <diagnostics/util/compilation_mode.ts.hpp>

#include <sstream>

#define TEST_COMPONENT_NAME Stream_Test_System
#define TEST_COMPONENT_NAMESPACE diagnostics::unittest

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;
TEST_NAMESPACE_BEGIN;
TEST_COMPONENT_TEST_NAMESPACE_BEGIN;

/**
 * @brief Testing is_interactive and interactive
 */ 
void incorrect_construction(Test_Data & test_data)
{
    char const * argv[5];
	
	
    // a correct one to begin with
    {
	::std::istringstream in;
	::std::ostringstream out;
	argv[0]="NAME";
	Stream_Test_System s(LEVEL_PROD,
			     1,
			     argv,
			     "test/blub.dat",
			     "test/result_log_file",
			     in,
			     out);
    }

	
    // wrong build level //////////////////////////////////////
    TEST_THROWING_BLOCK_ENTER;
    ::std::istringstream in;
    ::std::ostringstream out;
    argv[0]="NAME";
    Stream_Test_System s(LEVEL_TEST, // <-
			 1,
			 argv,
			 "test/blub.dat",
			 "test/result_log_file",
			 in,
			 out);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    ::std::istringstream in;
    ::std::ostringstream out;
    argv[0]="NAME";
    Stream_Test_System s(LEVEL_SYSTEM, // <-
			 1,
			 argv,
			 "test/blub.dat",
			 "test/result_log_file",
			 in,
			 out);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    // invalid test_data file name ///////////////////////////////////
    TEST_THROWING_BLOCK_ENTER;
    ::std::istringstream in;
    ::std::ostringstream out;
    argv[0]="NAME";
    Stream_Test_System s(LEVEL_AUDIT,
			 1,
			 argv,
			 "A_DIRECTORY_THAT_DOES_NOT_EXIST/AND_A_FILE_YOU_CANNOT_OPEN", // <-
			 "test/result_log_file",
			 in,
			 out);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    // invalid result_log_file file name ///////////////////////////////////
    TEST_THROWING_BLOCK_ENTER;
    ::std::istringstream in;
    ::std::ostringstream out;
    argv[0]="NAME";
    Stream_Test_System s(LEVEL_AUDIT,
			 1,
			 argv,
			 "test/blub.dat",
			 "A_DIRECTORY_THAT_DOES_NOT_EXIST/AND_A_FILE_YOU_CANNOT_OPEN", // <-
			 in,
			 out);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

	
    // a working one
    {
	::std::istringstream in;
	::std::ostringstream out;
	argv[0]="NAME";
	argv[1]="run";
	argv[2]="/*";
	argv[3]="2";
	Stream_Test_System s(LEVEL_AUDIT,
			     4,
			     argv,
			     "test/blub.dat",
			     "test/result_log_file",
			     in,
			     out);
    }
	
    // wronge number of arguments /////////////////////////////////////////

    TEST_THROWING_BLOCK_ENTER;
    ::std::istringstream in;
    ::std::ostringstream out;
    argv[0]="NAME";
    argv[1]="run";
    argv[2]="/*";
    argv[3]="2";
    Stream_Test_System s(LEVEL_AUDIT,
			 2, // <-
			 argv,
			 "test/blub.dat",
			 "test/result_log_file",
			 in,
			 out);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    ::std::istringstream in;
    ::std::ostringstream out;
    argv[0]="NAME";
    argv[1]="run";
    argv[2]="/*";
    argv[3]="2";
    Stream_Test_System s(LEVEL_AUDIT,
			 3, // <-
			 argv,
			 "test/blub.dat",
			 "test/result_log_file",
			 in,
			 out);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    // unknown argument /////////////////////////////////////////////////////

    TEST_THROWING_BLOCK_ENTER;
    ::std::istringstream in;
    ::std::ostringstream out;
    argv[0]="NAME";
    argv[1]="run1"; // <-
    argv[2]="/*";
    argv[3]="2";
    Stream_Test_System s(LEVEL_AUDIT,
			 4,
			 argv,
			 "test/blub.dat",
			 "test/result_log_file",
			 in,
			 out);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    ::std::istringstream in;
    ::std::ostringstream out;
    argv[0]="NAME";
    argv[1]="bla"; // <-
    argv[2]="/*";
    argv[3]="2";
    Stream_Test_System s(LEVEL_AUDIT,
			 4,
			 argv,
			 "test/blub.dat",
			 "test/result_log_file",
			 in,
			 out);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

	
    // invalid mask //////////////////////////////////////////////////

    TEST_THROWING_BLOCK_ENTER;
    ::std::istringstream in;
    ::std::ostringstream out;
    argv[0]="NAME";
    argv[1]="run"; 
    argv[2]="----"; // <-
    argv[3]="2";
    Stream_Test_System s(LEVEL_AUDIT,
			 4,
			 argv,
			 "test/blub.dat",
			 "test/result_log_file",
			 in,
			 out);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    // invalid level for the command 

    TEST_THROWING_BLOCK_ENTER;
    ::std::istringstream in;
    ::std::ostringstream out;
    argv[0]="NAME";
    argv[1]="run"; 
    argv[2]="/*";
    argv[3]="10"; // <-
    Stream_Test_System s(LEVEL_AUDIT,
			 4,
			 argv,
			 "test/blub.dat",
			 "test/result_log_file",
			 in,
			 out);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
	
}



void call_by_commandline_return_value(Test_Data & test_data, 
									  bool const invalid,
									  bool const failure)
{
    using ::diagnostics::unittest::testing::blind_test_run_result;

    char const * argv[5];

    ::std::istringstream in;
    ::std::ostringstream out;
	
    argv[0]="NAME";
    argv[1]="run";
    argv[2]="/*";
    argv[3]="3";
		
    ::std::string const test_data_key(::std::string(invalid ? "invalid " : "valid ")+
									  ::std::string(failure ? "failure " : "success"));

    Stream_Test_System s(LEVEL_DEBUG,
						 4,
						 argv,
						 "test/blub.dat",
						 "test/result_log_file",
						 in,
						 out);
		
    Dummy_Test_Case * const tc1(new Dummy_Test_Case("test_case_1",3,true,true,true,true,invalid, failure));
    Dummy_Test_Case * const tc2(new Dummy_Test_Case("test_case_2",3,true,true,true,true,invalid, failure));
    s.add(tc1);
    s.add(tc2);

	int result;

    TEST_NO_IMPLICIT_LOGGING_ENTER;
	result=s.run();
    TEST_NO_IMPLICIT_LOGGING_EXIT;
    TEST_ASSERT_RELATION(tc1->run_count(),==,1);
    TEST_ASSERT_RELATION(tc2->run_count(),==,1);
    TEST_ASSERT(test_data.compare(DIAGNOSTICS_COMPILATION_MODE_ID+test_data_key, 
								  blind_test_run_result(out.str())));
	TEST_ASSERT_RELATION(result,==,(invalid ? Test_System::TEST_SYSTEM_RESULT_INVALID : 
									(failure ? Test_System::TEST_SYSTEM_RESULT_FAILURE : 0)));
	
}

/**
 * @brief [PROD] Testing the return value of the run() method 
 */ 
void calling_by_commandline_return_value(Test_Data & test_data)
{
    call_by_commandline_return_value(test_data,false,false);
    call_by_commandline_return_value(test_data,false,true);
    call_by_commandline_return_value(test_data,true,false);
    call_by_commandline_return_value(test_data,true,true);
}




void call_by_commandline(Test_Data & test_data, 
			 ::std::string const & arg1,
			 ::std::string const & arg2,
			 ::std::string const & arg3,
			 int const calls1,
			 int const calls2)
{
    using ::diagnostics::unittest::testing::blind_test_run_result;

    char const * argv[5];

    ::std::istringstream in;
    ::std::ostringstream out;
	
    argv[0]="NAME";
    argv[1]=arg1.c_str();
    argv[2]=arg2.c_str();
    argv[3]=arg3.c_str();
		
    ::std::string const test_data_key(arg1+" "+arg2+" "+arg3);

    Stream_Test_System s(LEVEL_DEBUG,  // test case 2 has target level DEBUG
						 4,
						 argv,
						 "test/blub.dat",
						 "test/result_log_file",
						 in,
						 out);
		
    Dummy_Test_Case * const tc1(new Dummy_Test_Case("test_case_1",3,true,true,true,true));
    Dummy_Test_Case * const tc2(new Dummy_Test_Case("test_case_2",3,true,false,true,true));
    s.add(tc1);
    s.add(tc2);

    // one run
    TEST_NO_IMPLICIT_LOGGING_ENTER;
	s.run();
    TEST_NO_IMPLICIT_LOGGING_EXIT;
    TEST_ASSERT_RELATION(tc1->run_count(),==,calls1);
    TEST_ASSERT_RELATION(tc2->run_count(),==,calls2);
    TEST_ASSERT(test_data.compare(DIAGNOSTICS_COMPILATION_MODE_ID+test_data_key, 
				  blind_test_run_result(out.str())));

    // a second run
    TEST_NO_IMPLICIT_LOGGING_ENTER;
    s.run();
    TEST_NO_IMPLICIT_LOGGING_EXIT;
    TEST_ASSERT_RELATION(tc1->run_count(),==,calls1*2);
    TEST_ASSERT_RELATION(tc2->run_count(),==,calls1*2);
    TEST_ASSERT(test_data.compare(DIAGNOSTICS_COMPILATION_MODE_ID+test_data_key+" repeated", 
				  blind_test_run_result(out.str())));
}


/**
 * @brief [PROD] Testing the run() method when controlled by argc/argv
 */ 
void calling_by_commandline(Test_Data & test_data)
{
    // note -- invalid arguments are already handled by 
    // incorrect_construction
    call_by_commandline(test_data,"run","/*","3",1,1);
    call_by_commandline(test_data,"run","/*","1",1,1);
    call_by_commandline(test_data,"list","/*","3",0,0);
    call_by_commandline(test_data,"list","/*","2",0,0);
    call_by_commandline(test_data,"list","/*","1",0,0);
    call_by_commandline(test_data,"list","/*","0",0,0);

    call_by_commandline(test_data,"list","/test_case_1","3",0,0);
    call_by_commandline(test_data,"list","/test_case_1","2",0,0);
    call_by_commandline(test_data,"list","/test_case_1","1",0,0);
    call_by_commandline(test_data,"list","/test_case_1","0",0,0);

    call_by_commandline(test_data,"list","/test_case_2","3",0,0);
    call_by_commandline(test_data,"list","/test_case_2","2",0,0);
    call_by_commandline(test_data,"list","/test_case_2","1",0,0);
    call_by_commandline(test_data,"list","/test_case_2","0",0,0);

    call_by_commandline(test_data,"list","/noop","3",0,0);
}




/**
 * @brief [PROD] Testing the run() method when interactively
 */
void calling_by_interactively(Test_Data & test_data)
{
    using ::diagnostics::unittest::testing::blind_test_run_result;
	
    char const * argv[1];
    argv[0]="NAME";
    {
	Dummy_Test_Case * const tc1(new Dummy_Test_Case("test_case_1",3,true,true,true,true));
	Dummy_Test_Case * const tc2(new Dummy_Test_Case("test_case_2",3,true,false,true,true));
	::std::istringstream in("list /* 3\n"
				"run /* 3\n"
				"interactive 0\n"
				"interactive 1\n"
				"interactive 2\n"
				"run /test_case_1 0\n"
				"run /test_case_2 0\n"
				"run /* 0\n");
	::std::ostringstream out;
	Stream_Test_System s(LEVEL_DEBUG,1,argv,"test/blub.dat","test/result_log_file",in,out);
	s.add(tc1);
	s.add(tc2);
		
	TEST_NO_IMPLICIT_LOGGING_ENTER;
	s.run();
	TEST_NO_IMPLICIT_LOGGING_EXIT;
		
	TEST_ASSERT_RELATION(tc1->run_count(),==,1+1+1);
	TEST_ASSERT_RELATION(tc2->run_count(),==,1+0+0);
	TEST_ASSERT(test_data.compare(DIAGNOSTICS_COMPILATION_MODE_ID+"interactive1",
				      blind_test_run_result(out.str())));
    }


    {
	Dummy_Test_Case * const tc1(new Dummy_Test_Case("test_case_1",3,true,true,true,true));
	Dummy_Test_Case * const tc2(new Dummy_Test_Case("test_case_2",3,true,false,true,true));
	::std::istringstream in("list /* 3\n"
				"quit\n"
				"run /* 3\n");
	::std::ostringstream out;
	Stream_Test_System s(LEVEL_DEBUG,1,argv,"test/blub.dat","test/result_log_file",in,out);
	s.add(tc1);
	s.add(tc2);
		
	TEST_NO_IMPLICIT_LOGGING_ENTER;
	s.run();
	TEST_NO_IMPLICIT_LOGGING_EXIT;
		
	TEST_ASSERT_RELATION(tc1->run_count(),==,0);
	TEST_ASSERT_RELATION(tc2->run_count(),==,0);
	TEST_ASSERT(test_data.compare(DIAGNOSTICS_COMPILATION_MODE_ID+"interactive2", 
				      blind_test_run_result(out.str())));
    }


    {   
	Dummy_Test_Case * const tc1(new Dummy_Test_Case("test_case_1",3,true,true,true,true));
	Dummy_Test_Case * const tc2(new Dummy_Test_Case("test_case_2",3,true,false,true,true));
	::std::istringstream in("list /* 3\n"
				"thiscommanddoesnotexist\n"
				"run /* 3\n");
	::std::ostringstream out;
	Stream_Test_System s(LEVEL_DEBUG,1,argv,"test/blub.dat","test/result_log_file",in,out);
	s.add(tc1);
	s.add(tc2);
		
	TEST_NO_IMPLICIT_LOGGING_ENTER;
	s.run();
	TEST_NO_IMPLICIT_LOGGING_EXIT;
		
	TEST_ASSERT_RELATION(tc1->run_count(),==,1);
	TEST_ASSERT_RELATION(tc2->run_count(),==,1);
	TEST_ASSERT(test_data.compare(DIAGNOSTICS_COMPILATION_MODE_ID+"interactive3", 
				      blind_test_run_result(out.str())));
    }
	

    {   
	Dummy_Test_Case * const tc1(new Dummy_Test_Case("test_case_1",3,true,true,true,true));
	Dummy_Test_Case * const tc2(new Dummy_Test_Case("test_case_2",3,true,false,true,true));
	::std::istringstream in("list /* 3\n"
				"run /* 33\n"
				"run /* 3\n");
	::std::ostringstream out;
	Stream_Test_System s(LEVEL_DEBUG,1,argv,"test/blub.dat","test/result_log_file",in,out);
	s.add(tc1);
	s.add(tc2);
		
	TEST_NO_IMPLICIT_LOGGING_ENTER;
	s.run();
	TEST_NO_IMPLICIT_LOGGING_EXIT;
		
	TEST_ASSERT_RELATION(tc1->run_count(),==,1);
	TEST_ASSERT_RELATION(tc2->run_count(),==,1);
	TEST_ASSERT(test_data.compare(DIAGNOSTICS_COMPILATION_MODE_ID+"interactive4",
				      blind_test_run_result(out.str())));
    }



    {   
	Dummy_Test_Case * const tc1(new Dummy_Test_Case("test_case_1",3,true,true,true,true));
	Dummy_Test_Case * const tc2(new Dummy_Test_Case("test_case_2",3,true,false,true,true));
	::std::istringstream in("list /* 3\n"
				"run /*h 3\n"
				"run /* 3\n");
	::std::ostringstream out;
	Stream_Test_System s(LEVEL_DEBUG,1,argv,"test/blub.dat","test/result_log_file",in,out);
	s.add(tc1);
	s.add(tc2);
		
	TEST_NO_IMPLICIT_LOGGING_ENTER;
	s.run();
	TEST_NO_IMPLICIT_LOGGING_EXIT;
		
	TEST_ASSERT_RELATION(tc1->run_count(),==,1);
	TEST_ASSERT_RELATION(tc2->run_count(),==,1);
	TEST_ASSERT(test_data.compare(DIAGNOSTICS_COMPILATION_MODE_ID+"interactive5", 
				      blind_test_run_result(out.str())));
    }	

}


TEST_COMPONENT_TEST_NAMESPACE_END;
TEST_NAMESPACE_END;
UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

TEST_SUITE_BEGIN;
TEST_NORMAL_CASE(&incorrect_construction,LEVEL_PROD);
TEST_NORMAL_CASE(&calling_by_commandline,LEVEL_PROD);
TEST_NORMAL_CASE(&calling_by_interactively,LEVEL_PROD);
TEST_NORMAL_CASE(&calling_by_commandline_return_value,LEVEL_PROD);
TEST_SUITE_END;

STREAM_TEST_SYSTEM_MAIN;
// vim:ts=4:sw=4
