/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <globus_common.h>
#include "globus_rls_client.h"
#include "misc.h"
#include "auth.h"
#include "conf.h"
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <libgen.h>
#include <sys/types.h>
#include <sys/stat.h>

extern int	errno;
extern int	loglevel;

typedef enum { BOOL, INT, STR, ACLENTRY } opttypes_t;

typedef struct {
  char		*name;
  void		*val;
  union {
    int    i;
    char  *cp;
  }  		defval;
  opttypes_t	type;
  int		flags;
#define CF_CMDLINE	0x1
#define CF_DEPRECATED	0x2
#define CF_PRIVATE	0x4
} OPTION;

int		authentication;
ACL		*acllist = NULL;
int		lrc_bloomfilter_numhash;/* Number of bloom filter hashes*/
int		lrc_bloomfilter_ratio;	/* Ratio of filter size to #lfns*/
char		*lrc_dbname = NULL;
int		lrc_server;
int		update_bf_int;
int		update_factor;
int		update_immediate;
int		update_ll_int;
int		update_retry;
int		update_buftime;
char		*db_pwd = NULL;
char		*db_user = NULL;
int		idletimeout;
int		loglevel;
char    *logtype;
int		maxbackoff;
int		maxconnections;
int		maxfreethreads;
int		maxthreads;
char		*myurl = NULL;
int		myurllen;		/* Includes trailing null byte	*/
static char	myurlbuf[BUFLEN];
char		*odbcini = NULL;
char		*pidfile = NULL;
int		port;
int		result_limit;
int		rli_bloomfilter;
char		*rli_bloomfilter_dir = NULL; /* Dir for RLI to save filters */
char		*rli_dbname = NULL;
int		rli_expire_int;
int		rli_expire_stale;
int		rli_server;
char		*rlscertfile = NULL;
char		*rlskeyfile = NULL;
int		startthreads;
int		timeout;

static OPTION	options[] = {
  { "acl", NULL, {.cp = NULL}, ACLENTRY, 0 },
  { "authentication", &authentication, {.i = GLOBUS_TRUE}, BOOL, 0 },
  { "bloomfilterdir", &rli_bloomfilter_dir, {.cp = NULL}, STR, CF_DEPRECATED },
  { "bloomfilternumhash", &lrc_bloomfilter_numhash, {.i = 0}, INT, CF_DEPRECATED },
  { "bloomfilterratio", &lrc_bloomfilter_ratio, {.i = 0}, INT, CF_DEPRECATED },
  { "dbhost", NULL, {.cp = NULL}, STR, CF_DEPRECATED },
  { "db_pwd", &db_pwd, {.cp = "changethis"}, STR, CF_PRIVATE },
  { "db_user", &db_user, {.cp = "dbperson"}, STR, 0 },
  { "dbpwd", &db_pwd, {.cp = NULL}, STR, CF_DEPRECATED | CF_PRIVATE },
  { "dbuser", &db_user, {.cp = NULL}, STR, CF_DEPRECATED },
  { "idletimeout", &idletimeout, {.i = 900}, INT, 0 },
  { "loglevel", &loglevel, {.i = 0}, INT, 0 },
  { "logtype", &logtype, {.cp = "syslog"}, STR, 0 },
  { "lrc_bloomfilter_numhash", &lrc_bloomfilter_numhash, {.i = 3}, INT, 0 },
  { "lrc_bloomfilter_ratio", &lrc_bloomfilter_ratio, {.i = 10}, INT, 0 },
  { "lrc_buffer_time", &update_buftime, {.i = 0}, INT, CF_DEPRECATED },
  { "lrc_dbname", &lrc_dbname, {.cp = "lrc1000"}, STR, 0 },
  { "lrc_server", &lrc_server, {.i = GLOBUS_FALSE}, BOOL, 0 },
  { "lrc_update_bf", &update_bf_int, {.i = 0}, INT, CF_DEPRECATED },
  { "lrc_update_factor", &update_factor, {.i = 0}, INT, CF_DEPRECATED },
  { "lrc_update_immediate", &update_immediate, {.i = GLOBUS_FALSE}, BOOL, CF_DEPRECATED},
  { "lrc_update_ll", &update_ll_int, {.i = 0}, INT, CF_DEPRECATED },
  { "lrc_update_retry", &update_retry, {.i = 0}, INT, CF_DEPRECATED },
  { "lrcserver", &lrc_server, {.i = GLOBUS_FALSE}, BOOL, CF_DEPRECATED },
  { "maxbackoff", &maxbackoff, {.i = 300}, INT, 0 },
  { "maxconnections", &maxconnections, {.i = 100}, INT, 0 },
  { "maxfreethreads", &maxfreethreads, {.i = 5}, INT, 0 },
  { "maxthreads", &maxthreads, {.i = 30}, INT, 0 },
  { "myurl", &myurl, {.cp = myurlbuf}, STR, 0 },
  { "odbcini", &odbcini, {.cp = NULL}, STR, 0 },
  { "outbuftime", &update_buftime, {.i = 0}, INT, CF_DEPRECATED },
  { "pidfile", &pidfile, {.cp = NULL}, STR, 0 },
  { "port", &port, {.i = GLOBUS_RLS_SERVER_DEFPORT}, INT, 0 },
  { "result_limit", &result_limit, {.i = 0}, INT, 0 },
  { "rli_bloomfilter", &rli_bloomfilter, {.i = GLOBUS_FALSE}, BOOL, 0 },
  { "rli_bloomfilter_dir", &rli_bloomfilter_dir, {.cp = "none"}, STR, 0 },
  { "rli_dbname", &rli_dbname, {.cp = "rli1000"}, STR, 0 },
  { "rli_expire_int", &rli_expire_int, {.i = 28800}, INT, 0 },
  { "rli_expire_stale", &rli_expire_stale, {.i = 86400}, INT, 0 },
  { "rli_server", &rli_server, {.i = GLOBUS_FALSE}, BOOL, 0 },
  { "rlibloomfilter", &rli_bloomfilter, {.i = GLOBUS_FALSE}, BOOL, CF_DEPRECATED },
  { "rliexpireint", &rli_expire_int, {.i = 0}, INT, CF_DEPRECATED },
  { "rliimmediate", &update_immediate, {.i = GLOBUS_FALSE}, BOOL, CF_DEPRECATED },
  { "rliretryint", &update_retry, {.i = 0}, INT, CF_DEPRECATED },
  { "rliserver", &rli_server, {.i = GLOBUS_FALSE}, BOOL, CF_DEPRECATED },
  { "rlistaleint", &rli_expire_stale, {.i = 0}, INT, CF_DEPRECATED },
  { "rliupdateint", &update_ll_int, {.i = 0}, INT, CF_DEPRECATED },
  { "rliupdatesyncfactor", &update_factor, {.i = 0}, INT, CF_DEPRECATED },
  { "rlscertfile", &rlscertfile, {.cp = "/etc/grid-security/hostcert.pem"}, STR, 0 },
  { "rlskeyfile", &rlskeyfile, {.cp = "/etc/grid-security/hostkey.pem"}, STR, 0 },
  { "startthreads", &startthreads, {.i = 3}, INT, 0 },
  { "timeout", &timeout, {.i = -1}, INT, 0 },
  { "update_bf_int", &update_bf_int, {.i = 900}, INT, 0 },
  { "update_buftime", &update_buftime, {.i = 30}, INT, 0 },
  { "update_factor", &update_factor, {.i = 10}, INT, 0 },
  { "update_immediate",&update_immediate,{.i = GLOBUS_FALSE}, BOOL,0},
  { "update_ll_int", &update_ll_int, {.i = 86400}, INT, 0 }, 
  { "update_retry", &update_retry, {.i = 300}, INT, 0 },
  { "wakeupint", NULL, {.i = 300}, INT, CF_DEPRECATED },
};
#define NUMOPTIONS	(sizeof(options) / sizeof(OPTION))

void
conf_read(char *prog, char *fn)

{
  FILE		*f;
  char		buf[BUFLEN];
  char		*kwd;
  char		*val;
  char		*cp;
  int		i;
  char		globuslocation[BUFLEN];
  struct stat	sb;
  static char	odbcinienv[BUFLEN];

  if (loglevel)
    logit(LOG_DEBUG, "conf_read: %s", fn);
  /*
   * Initialize options settable from config file, but don't override
   * values set on command line or dynamically via "set_configuration()".
   */
  for (i = 0; i < NUMOPTIONS; i++) {
    if (options[i].flags & (CF_CMDLINE|CF_DEPRECATED))
      continue;
    switch (options[i].type) {
      case BOOL:/* Same as INT	*/
      case INT: *(int *) options[i].val = options[i].defval.i;
		break;
      case STR:	if (*(char **) options[i].val &&
		    *(char **) options[i].val != options[i].defval.cp)
		  globus_libc_free(*(char **) options[i].val);
		*(char **) options[i].val = options[i].defval.cp;
		break;
      default:	break;
    }
  }
  if (acllist)
    acllist = auth_freeacl(acllist);
  if ((f = fopen(fn, "r")) == NULL) {
    logit(LOG_INFO, "conf_read: Can't open %s: %s", fn, strerror(errno));
    return;
  }
  while (fgets(buf, BUFLEN, f)) {
    for (kwd = buf; isspace((int) *kwd); kwd++);
    if (!*kwd || *kwd == '#')
      continue;
    for (val = kwd; *val && !isspace((int) *val); val++);
    *val++ = '\0';
    while (isspace((int) *val))
      val++;
    for (cp = val; *cp && *cp != '\n' && *cp != '#'; cp++);
    while (isspace((int) cp[-1]))
      cp--;
    *cp = '\0';
    conf_option(kwd, val, 0);
  }
  fclose(f);

  globus_rls_client_certificate(rlscertfile, rlskeyfile);
  putenv("X509_RUN_AS_SERVER=1");

  if ((cp = getenv("GLOBUS_LOCATION")))
    strncpy(globuslocation, cp, BUFLEN);
  else
    *globuslocation = '\0';

  if (odbcini) {
    snprintf(odbcinienv, BUFLEN, "ODBCINI=%s", odbcini);
    putenv(odbcinienv);
  } else if (!getenv("ODBCINI"))	/* Don't override if set */
    if (*globuslocation) {
      snprintf(odbcinienv, BUFLEN, "ODBCINI=%s/var/odbc.ini", globuslocation);
      putenv(odbcinienv);
    }

  if (!pidfile) {
    if (*globuslocation)
      globus_libc_sprintf(buf, "%s/var/%s.pid", globuslocation, prog);
    else
      globus_libc_sprintf(buf, "/var/run/%s.pid", prog);
    pidfile = globus_libc_strdup(buf);
  }

  if (!rli_server)
    rli_bloomfilter = 0;

  /*
   * The RLI can be configured to save bloomfilters to disk.  These are
   * read at start time so the RLI does not have to wait for updates
   * from LRCs when it starts.  If rli_bloomfilter_dir is "none" then this
   * isn't done, if "default" then the default directory is used, else
   * the user specified directory is used.
   */
  if (rli_bloomfilter) {
    if (strcasecmp(rli_bloomfilter_dir, "none") == 0)
      rli_bloomfilter_dir = NULL;
    else if (strcasecmp(rli_bloomfilter_dir, "default") == 0) {
      if (*globuslocation) {
        globus_libc_sprintf(buf, "%s/var/rls_bloomfilters", globuslocation);
        rli_bloomfilter_dir = globus_libc_strdup(buf);
      }
      else {
        logit(LOG_WARNING, "conf_read: GLOBUS_LOCATION must be defined in order to set \'rli_bloomfilter_dir\' to \'default\'");
        rli_bloomfilter_dir = NULL;
      }
    }
    if (rli_bloomfilter_dir) {	/* Create if doesn't exist		*/
      if (stat(rli_bloomfilter_dir, &sb) != 0) {
	if (errno == ENOENT) {
	  if (mkdir(rli_bloomfilter_dir, 0755) != 0) {
	    logit(LOG_WARNING, "conf_read: %s: %s", rli_bloomfilter_dir,
		  strerror(errno));
	    rli_bloomfilter_dir = NULL;
	  }
	} else {
	  logit(LOG_WARNING, "conf_read: %s: %s", rli_bloomfilter_dir,
		strerror(errno));
	  rli_bloomfilter_dir = NULL;
	}
      } else if (!(sb.st_mode & S_IFDIR)) {
	logit(LOG_WARNING,"conf_read: %s not a directory",rli_bloomfilter_dir);
	rli_bloomfilter_dir = NULL;
      }
    }
  }

  if (lrc_bloomfilter_numhash > 8)
    lrc_bloomfilter_numhash = 8;

  if (myurl == myurlbuf) {
    globus_libc_gethostname(buf, BUFLEN);
    globus_libc_sprintf(myurlbuf, "%s://%s:%d",
	authentication ? GLOBUS_RLS_URL_SCHEME : GLOBUS_RLS_URL_SCHEME_NOAUTH,
	buf, port);
  }
  myurllen = strlen(myurl) + 1;

  if (!update_immediate || update_factor < 1)
    update_factor = 1;

  if (maxthreads < 2)
    maxthreads = 2;

  if (timeout >= 0)
    globus_rls_client_set_timeout(timeout);

  if (loglevel > 2)
    for (i = 0; i < NUMOPTIONS; i++) {
      if (options[i].flags & CF_DEPRECATED ||
              options[i].flags & CF_PRIVATE)
	continue;
      switch (options[i].type) {
	case BOOL:
	  logit(LOG_DEBUG, "%24s: %s", options[i].name,
		*(int *) options[i].val ? "true" : "false");
	  break;
	case INT:
	  logit(LOG_DEBUG, "%24s: %d", options[i].name,
		*(int *) options[i].val);
	  break;
	case STR:
	  logit(LOG_DEBUG, "%24s: %s", options[i].name,
	       *(char **) options[i].val ? *(char **) options[i].val : "NULL");
	  break;
	default:
	  break;
      }
    }
}

int
conf_option(char *name, char *val, int cmdline)

{
  int	i;
  int	j;

  for (i = 0; i < NUMOPTIONS; i++) {
    if (strcasecmp(options[i].name, name) != 0)
      continue;
    if (options[i].flags & CF_DEPRECATED) {
      if (options[i].val) {		/* Option was renamed	*/
        for (j = 0; j < NUMOPTIONS; j++)
	  if (i != j)
	    if (options[i].val == options[j].val && !(options[j].flags & CF_DEPRECATED)) {
	      logit(LOG_INFO, "conf_option: %s deprecated, please use %s",
		    options[i].name, options[j].name);
	      i = j;
	      break;
	    }
      } else {
	logit(LOG_INFO, "conf_option: %s deprecated, ignored",options[i].name);
	return GLOBUS_RLS_BADARG;
      }
    }
    if ((options[i].flags & CF_CMDLINE) && !cmdline)
      return GLOBUS_RLS_SUCCESS;	/* Option already set, ignore	*/
    switch (options[i].type) {
      case ACLENTRY:
		acllist = auth_newacl(acllist, val);
		break;
      case BOOL:if (strcasecmp(val, "on") == 0 || strcasecmp(val, "true") == 0)
		  *(int *) options[i].val = GLOBUS_TRUE;
		else
		  *(int *) options[i].val = GLOBUS_FALSE;
		break;
      case INT:	*(int *) options[i].val = atoi(val);
		break;
      case STR:	if (*(char **) options[i].val &&
		    *(char **) options[i].val != options[i].defval.cp)
		  globus_libc_free(*(char **) options[i].val);
		*(char **) options[i].val = globus_libc_strdup(val);
		break;
    }
    if (cmdline)
      options[i].flags |= CF_CMDLINE;
    return GLOBUS_RLS_SUCCESS;
  }
  logit(LOG_WARNING, "conf_option: Unknown option %s", name);
  return GLOBUS_RLS_BADARG;
}

/*
 * Return configuration options.
 */
int
conf_get(int idx, void **sidx, char **name, char **val, char *buf)

{
  ACL	*acl;
  int	all;

  all = 1;
  if (idx == OPT_ALL) {
    for (idx = 0; idx < NUMOPTIONS; idx++)
      if (!(options[idx].flags & CF_DEPRECATED))
	break;
  } else if (idx == OPT_ONE) {
    for (idx = 0; idx < NUMOPTIONS; idx++)
      if (strcmp(options[idx].name, *name) == 0)
	break;
    if (idx < 0) {
      *val = "unknown option";
      return -1;
    } else if (options[idx].flags & CF_DEPRECATED) {
      *val = "deprecated";
      return -1;
    }
    all = 0;
  }
  if (idx < 0 || idx >= NUMOPTIONS)
    return OPT_DONE;

  *name = options[idx].name;
  switch (options[idx].type) {
    case ACLENTRY:
      if (*sidx)
	acl = (ACL *) *sidx;
      else
	acl = acllist;
      if (acl) {
	sprintf(buf, "%s: %s", acl->dn, auth_perms2s(acl->perms));
	*val = buf;
	if (acl->nxt) {
	  *sidx = acl->nxt;
	  if (all)
	    return idx;
	  else
	    return OPT_ONE;
	}
      } else
	*val = "";
      break;
    case BOOL:
      if (*(int *) options[idx].val)
	*val = "true";
      else
	*val = "false";
      break;
    case INT:
      sprintf(buf, "%d", *(int *) options[idx].val);
      *val = buf;
      break;
    case STR:
      *val = *(char **) options[idx].val;
      if (!*val)
	*val = "";
      break;
  }
  while (++idx < NUMOPTIONS)
    if (!(options[idx].flags & CF_DEPRECATED))
      break;
  return idx;
}

