/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     traverse_r.c                                                   */
/*                                                                          */
/*                                                                          */
/* description:                                                             */
/*           recursive mesh traversal routines - common ?d part             */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Univesitaet Bremen                                           */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include "alberta.h"

/*--------------------------------------------------------------------------*/
/*  vertex_of_edge_?d[edge][i], i = 1,2 are the two vertices of edge        */
/*--------------------------------------------------------------------------*/
int vertex_of_edge_2d[3][2] = {{1,2}, {2,0}, {0,1}};
int vertex_of_edge_3d[6][2]={{0,1},{0,2},{0,3},{1,2},{1,3},{2,3}};

/*--------------------------------------------------------------------------*/
/*   child_vertex_3d[el_type][child][i] =                                   */
/*       father's local vertex index of new vertex i                        */
/*       4 stands for the newly generated vertex                            */
/*--------------------------------------------------------------------------*/
int  child_vertex_3d[3][2][4] = {{{0,2,3,4},{1,3,2,4}},
				 {{0,2,3,4},{1,2,3,4}},
				 {{0,2,3,4},{1,2,3,4}}};

/*--------------------------------------------------------------------------*/
/*   child_edge_3d[el_type][child][i] =                                     */
/*       father's local edge index of new edge i                            */
/*       new edge 2 is half of old edge 0,                                  */
/*       new edges 4,5 are really new edges, and value is different:        */
/*         child_edge_3d[][][4,5] = index of same edge in other child	    */
/*--------------------------------------------------------------------------*/
int child_edge_3d[3][2][6] = {{{1,2,0,5,5,4},{4,3,0,5,5,4}},
			      {{1,2,0,5,4,5},{3,4,0,5,4,5}},
			      {{1,2,0,5,4,5},{3,4,0,5,4,5}}};

/*--------------------------------------------------------------------------*/
/*   child_orientation[el_type][child] =                                    */
/*      +1 if orientation is not changed during refinement                  */
/*      -1 if orientation is changed during refinement                      */
/*--------------------------------------------------------------------------*/
S_CHAR child_orientation_3d[3][2] = {{1,1}, {1,-1}, {1,-1}};

/* traverse hierarchical mesh,  call el_fct() for each/some element */

typedef struct traverse_info TRAVERSE_INFO;

struct traverse_info 
{
  MESH     *mesh;
  FLAGS    flag; 
  int      level;
  void     (*el_fct)(const EL_INFO *, void *);
  void     *data;
};

void AI_run_traverse_hooks(MESH *mesh, const EL_INFO *el_info)
{
  FLAGS           acthooks = mesh->active_hooks;
  HOOK_QUEUE_ENUM queue;
  LIST_NODE       *pos;

  queue = 0;
  while (acthooks) {
    if (acthooks & 1) {
      for (pos = mesh->traverse_hooks[queue].next;
	   pos != &mesh->traverse_hooks[queue];
	   pos = pos->next) {
	TRAVERSE_HOOK *hook = LIST_ENTRY(pos, TRAVERSE_HOOK, node);

	hook->function(el_info, hook->data);
      }
    }
    acthooks >>= 1;
    queue ++;
  }
}

static void run_traverse_function(struct traverse_info *trinfo,
				  const EL_INFO *el_info)
{
  AI_run_traverse_hooks(trinfo->mesh, el_info);
  trinfo->el_fct(el_info, trinfo->data);
}

#include "traverse_r_1d.c"
#if DIM_OF_WORLD > 1
#include "traverse_r_2d.c"
#if DIM_OF_WORLD > 2
#include "traverse_r_3d.c"
#endif
#endif


/*--------------------------------------------------------------------------*/
/* common traversal routines for all dimensions                             */
/*--------------------------------------------------------------------------*/

void fill_macro_info(MESH *mesh, const MACRO_EL *mel, EL_INFO *elinfo)
{
  FUNCNAME("fill_macro_info");

  TEST_EXIT(mesh,"No mesh specified!\n");

  switch(mesh->dim) {
  case 0:
    elinfo->mesh     = mesh;
    elinfo->macro_el = mel;
    elinfo->el       = mel->el;
    elinfo->parent   = nil;
    elinfo->level    = 0;

    if (elinfo->fill_flag & FILL_COORDS) {
      int j;

      if (mel->coord[0])
	for (j = 0; j < DIM_OF_WORLD; j++) 
	  elinfo->coord[0][j] = mel->coord[0][j];
      else
	ERROR_EXIT("no mel->coord[0]\n");
    }

    break;
  case 1:
    fill_macro_info_1d(mesh, mel, elinfo);
    break;
#if DIM_OF_WORLD > 1
  case 2:
    fill_macro_info_2d(mesh, mel, elinfo);
    break;
#if DIM_OF_WORLD > 2
  case 3:
    fill_macro_info_3d(mesh, mel, elinfo);
    break;
#endif
#endif
  default:
    ERROR_EXIT("Illegal dim == %d!\n", mesh->dim);
  }
}


void fill_elinfo(int ichild, const EL_INFO *parent_info, EL_INFO *elinfo)
{
  FUNCNAME("fill_elinfo");
  int dim = parent_info->mesh->dim;

  switch(dim) {
  case 1:
    fill_elinfo_1d(ichild, parent_info, elinfo);
    break;
#if DIM_OF_WORLD > 1
  case 2:
    fill_elinfo_2d(ichild, parent_info, elinfo);
    break;
#if DIM_OF_WORLD > 2
  case 3:
    fill_elinfo_3d(ichild, parent_info, elinfo);
    break;
#endif
#endif
  default:
    ERROR_EXIT("Illegal dim == %d!\n", dim);
  }
}

/*--------------------------------------------------------------------------*/
/*   recursive_traverse:                                		    */
/*   -------------------                                		    */
/*   recursively traverse the mesh hierarchy tree       		    */
/*   call the routine traverse_info->el_fct(el_info) with    		    */
/*    - all tree leaves                                 		    */
/*    - all tree leaves at a special level              		    */
/*    - all tree elements in pre-/in-/post-order        		    */
/*   depending on the traverse_info->level variable     		    */
/*--------------------------------------------------------------------------*/

static void recursive_traverse(EL_INFO *elinfo, TRAVERSE_INFO *trinfo)
{
  EL      *el = elinfo->el;
  EL_INFO elinfo_new = {0};
  int     mg_level;

  if (trinfo->flag & CALL_LEAF_EL) 
  {
    if (el->child[0]) 
    {
      fill_elinfo(0, elinfo, &elinfo_new);
      recursive_traverse(&elinfo_new, trinfo);
      fill_elinfo(1, elinfo, &elinfo_new);
      recursive_traverse(&elinfo_new, trinfo);
    }
    else 
    {
      run_traverse_function (trinfo, elinfo);
    }
    return;
  }

  if (trinfo->flag & CALL_LEAF_EL_LEVEL) 
  {
    if (el->child[0]) 
    {
      if ((elinfo->level >= trinfo->level))  return;
      fill_elinfo(0, elinfo, &elinfo_new);
      recursive_traverse(&elinfo_new, trinfo);
      fill_elinfo(1, elinfo, &elinfo_new);
      recursive_traverse(&elinfo_new, trinfo);
    }
    else 
    {
      if (elinfo->level == trinfo->level) {
	run_traverse_function (trinfo, elinfo);
      }
    }
    return;
  }

  if (trinfo->flag & CALL_EL_LEVEL) 
  {
    if (elinfo->level == trinfo->level)
    {
      run_traverse_function (trinfo, elinfo);
    }
    else 
    {
      if (elinfo->level > trinfo->level)
      {
	return;
      }
      else
      {
	if (el->child[0]) 
	{
	  fill_elinfo(0, elinfo, &elinfo_new);
	  recursive_traverse(&elinfo_new, trinfo);
	  fill_elinfo(1, elinfo, &elinfo_new);
	  recursive_traverse(&elinfo_new, trinfo);
	}
      }
    }
    return;
  }

  if (trinfo->flag & CALL_MG_LEVEL) 
  {
    int     dim = trinfo->mesh->dim;

    mg_level = (elinfo->level + dim - 1) / dim;

    if (mg_level > trinfo->level) return;

    if (!(el->child[0])) 
    {
      run_traverse_function (trinfo, elinfo);
      return;
    }
    
    if ((mg_level == trinfo->level) && ((elinfo->level % dim) == 0))
    {
      run_traverse_function (trinfo, elinfo);
      return;
    }

    fill_elinfo(0, elinfo, &elinfo_new);
    recursive_traverse(&elinfo_new, trinfo);

    fill_elinfo(1, elinfo, &elinfo_new);
    recursive_traverse(&elinfo_new, trinfo);

    return;
  }

  if (trinfo->flag & CALL_EVERY_EL_PREORDER)
    run_traverse_function (trinfo, elinfo);

  if (el->child[0]) 
  {
    fill_elinfo(0, elinfo, &elinfo_new);
    recursive_traverse(&elinfo_new, trinfo);

    if (trinfo->flag & CALL_EVERY_EL_INORDER)
      run_traverse_function (trinfo, elinfo);

    fill_elinfo(1, elinfo, &elinfo_new);
    recursive_traverse(&elinfo_new, trinfo);
  }
  else 
  {
    if (trinfo->flag & CALL_EVERY_EL_INORDER)
      run_traverse_function (trinfo, elinfo);
  }

  if (trinfo->flag & CALL_EVERY_EL_POSTORDER)
    run_traverse_function (trinfo, elinfo);

  return;

}

/*--------------------------------------------------------------------------*/
/*   mesh_traverse:                                            		    */
/*   --------------                                     		    */
/*   run through all macro elements and start mesh traversal      	    */
/*   call recursive_traverse() for each macro element's element		    */
/*--------------------------------------------------------------------------*/

void mesh_traverse(MESH *mesh, int level, FLAGS flag, 
		   void (*el_fct)(const EL_INFO *, void *data), void *data)
{
  FUNCNAME("mesh_traverse");
  int           n;
  EL_INFO       elinfo = {0};
  TRAVERSE_INFO traverse_info = {0};

  if (mesh==nil) return;

  elinfo.fill_flag = (flag & FILL_ANY(mesh));
  
  traverse_info.mesh   = mesh;
  traverse_info.level  = level;
  traverse_info.el_fct = el_fct;
  traverse_info.flag   = flag;
  traverse_info.data   = data;
  elinfo.mesh = mesh;

  if (flag & (CALL_LEAF_EL_LEVEL | CALL_EL_LEVEL | CALL_MG_LEVEL))
    TEST_EXIT(level >= 0, "invalid level: %d\n",level);

  if(mesh->parametric && !mesh->parametric->use_reference_mesh)
    flag &= ~FILL_COORDS;

  for(n = 0; n < mesh->n_macro_el; n++) {
    fill_macro_info(mesh, mesh->macro_els + n, &elinfo);

    if(mesh->dim > 0)
      recursive_traverse(&elinfo, &traverse_info);
    else
      run_traverse_function (&traverse_info, &elinfo);
  }

}


/*--------------------------------------------------------------------------*/
/*   test_traverse_fct:                              		            */
/*   ------------------                                 		    */
/*   display information about an element from EL and EL_INFO structures    */
/*--------------------------------------------------------------------------*/

static void test_traverse_fct(const EL_INFO *elinfo, void *data)
{
  FUNCNAME("test_traverse_fct");
  EL *el = elinfo->el;
  int i,j, dim = elinfo->mesh->dim;

  MSG("\n");
  MSG("traversing element %d at %p ---------------------------\n",
      INDEX(el), el);

  print_msg("level:        %3d\n",elinfo->level);

  print_msg("macro_el:     %p\n", elinfo->macro_el);
  
  if (el->child[0]) {
    print_msg("children:      ");
    for (i=0;i<2;i++) 
      if (el->child[i])
	         print_msg(" %3d at %p",INDEX(el->child[i]), el->child[i]);
      else       print_msg(" ---");
    print_msg("\n");
  }

  if (elinfo->fill_flag & FILL_COORDS) {
    print_msg("coords:      ");
    for (i=0; i<N_VERTICES(dim); i++) 
    {
      print_msg("%1d: (", i);
      for (j=0; j<DIM_OF_WORLD; j++)
	print_msg("%10.6lf%s", elinfo->coord[i][j],
		 j < (DIM_OF_WORLD-1) ? ", " : ")\n");
      if (i < (N_VERTICES(dim)-1)) print_msg("             ");
    }
  }

  if (elinfo->fill_flag & FILL_NEIGH) {
#if ALBERTA_DEBUG
    print_msg("neigh index :");
    for (i=0; i<N_NEIGH(dim); i++) 
      if (elinfo->neigh[i]) print_msg(" %3d",INDEX(elinfo->neigh[i]));
      else                  print_msg(" ---");
    print_msg("\n");
#endif
    print_msg("opp_vertex:  ");
    for (i=0; i<N_NEIGH(dim); i++) 
      if (elinfo->neigh[i]) print_msg(" %3d", elinfo->opp_vertex[i]);
      else                  print_msg(" ---");
    print_msg("\n");

    print_msg("neigh:      ");
    for (i=0; i<N_NEIGH(dim); i++) print_msg(" %p",elinfo->neigh[i]);
    print_msg("\n");
  }
   
  if (elinfo->fill_flag & FILL_OPP_COORDS) {
    print_msg("opp_coords:  ");
    for (i=0; i<N_NEIGH(dim); i++) {
      if (elinfo->neigh[i]) {
	print_msg("%1d (ov=%1d): (", i, elinfo->opp_vertex[i]);
	for (j=0; j<DIM_OF_WORLD; j++)
	  print_msg("%10.6lf%s", elinfo->opp_coord[i][j],
		   j < (DIM_OF_WORLD-1) ? ", " : ")\n");
      }
      else {
	print_msg("%1d             : ---\n",i);
      }
      if (i < (N_VERTICES(dim)-1)) print_msg("             ");
    }
  }

  if(elinfo->fill_flag & FILL_PROJECTION) {
    print_msg("projections:\n");
    for(i = 0; i < N_NEIGH(dim) + 1; i++)
      print_msg("   no. %d: %p\n", i, elinfo->projections[i]);
    print_msg("  active projection: %p\n", elinfo->active_projection);
  }

  return;
}

/*--------------------------------------------------------------------------*/
/*   test_traverse:                                  		            */
/*   --------------                                     		    */
/*   display information about all elements from EL and EL_INFO structures  */
/*   by calling traverse with test_traverse_fct().      		    */
/*--------------------------------------------------------------------------*/

void test_traverse(MESH *mesh, int level, FLAGS fill_flag)
{
  FUNCNAME("test_traverse");

  if(!mesh) {
    ERROR("No mesh specified.\n");
    return;
  }

  MSG("with level    : %3d\n", level);
  MSG("with fill_flag:");
  if (fill_flag & FILL_ANY(mesh)) {
    if (fill_flag & FILL_COORDS)     print_msg(" FILL_COORDS");
    if (fill_flag & FILL_BOUND)      print_msg(" FILL_BOUND");
    if (fill_flag & FILL_PROJECTION) print_msg(" FILL_PROJECTION");
    if (fill_flag & FILL_NEIGH)      print_msg(" FILL_NEIGH");
    if (fill_flag & FILL_OPP_COORDS) print_msg(" FILL_OPP_COORDS");
  }
  else print_msg(" none");
  print_msg("\n");

  mesh_traverse(mesh, level, fill_flag, test_traverse_fct, nil);
  MSG("done.\n");
}
