/*
 * @(#)DirMetaDataWriter.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.datastore;

import java.io.File;
import java.io.IOException;

import net.sourceforge.groboutils.codecoverage.v2.IAnalysisModule;


/**
 * Stores meta-data in a directory structure.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:26 $
 * @since     December 15, 2002
 * @see       ZipMetaDataReader
 */
public class DirMetaDataWriter implements IMetaDataWriter
{
    private DirMetaDataIO store;
    
    
    /**
     * Creates a meta-data writer for a specific directory.  This directory
     * should be dedicated to just this writer.
     */
    public DirMetaDataWriter( File basedir )
            throws IOException
    {
        if (basedir == null)
        {
            throw new IllegalArgumentException( "No null args." );
        }
        this.store = new DirMetaDataIO( basedir );
    }
    
    
    /**
     * Returns a mark meta-data writer for a specific class.  If the
     * module has not already been added to the store, it will be added.
     */
    public IClassMetaDataWriter getClassWriter( IAnalysisModule module )
            throws IOException
    {
        if (module == null)
        {
            throw new IllegalArgumentException("no null args");
        }
        checkClose();
        addModule( module );
        return new DirClassMetaDataWriter( module, this.store );
    }
    
    
    /**
     * Closes this writer to prevent further access.
     */
    public void close()
            throws IOException
    {
        checkClose();
        this.store.close();
        this.store = null;
    }
    
    
    /**
     * Conditionally add the module if it isn't already known.
     */
    private void addModule( IAnalysisModule module )
            throws IOException
    {
        /*
        System.out.println(
"********************************************\n"+
"Oi!  There seems to be a bug in this method!\n"+
"********************************************\n"
        );
        */
        
        
        AnalysisModuleSet ams = this.store.getAnalysisModuleSet();
        if (ams.getAnalysisModuleIndex( module ) < 0)
        {
            // add the module
            ams.addAnalysisModule( module );
            this.store.putAnalysisModuleSet( ams );
        }
    }
    
    
    private void checkClose()
            throws IOException
    {
        if (this.store == null)
        {
            throw new IOException( "Writer has already been closed." );
        }
    }
    
    
    // this shouldn't close the DirMetaDataIO instance, but rather the
    // DirMetaDataIO instance should close itself.
    protected void finalize() throws Throwable
    {
        Exception ex = null;
        if (this.store != null)
        {
            ex = new IllegalStateException("Did not close writer.");
        }
        
        super.finalize();
        
        // class-based post condition
        if (ex != null)
        {
            throw ex;
        }
    }
}
