/* nexp_ghost.c
   
   The "ghost" command is implemented here. Used as in:

   ghost with ip 10.116.188.51 and mac 00:00:5e:fc:5c:52 on interface eth0

   Copyright (C) 2007, 2008, 2009 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"
#include "nexp_cmd_parser.h"
#include "util-tcl.h"

#define SNAPLEN 65535

#define GHOST_ID_VARNAME "ghost_id"

#define NEXP_GHOST_NAMELEN (16 + TCL_INTEGER_SPACE)

struct nexp_ghost {
    struct nexp_ghost *next;

    char name[NEXP_GHOST_NAMELEN + 1];

    const char *iface;
    struct addr l2addr;
    struct addr l3addr;
};

/*
 * Linked list of all active ghosts.
 */
static struct nexp_ghost *ghosts;

/*
 * Maintains information needed to be able to read traffic to a ghost,
 * i.e. the PCAP packet capture descriptor and its corresponding file
 * descriptor for select() use, and to send traffic from a ghost, i.e.
 * the libdnet Ethernet handle. There is one of this per *interface*, not
 * one per ghost.
 */
struct ghost_ctrl_blk {
    struct ghost_ctrl_blk *next;

    const char *iface;	    /* the interface name */

    pcap_t *pd;		    /* packet capture descriptor */
    int fd;		    /* file descriptor returned by
			       pcap_get_selectable_fd() */
    eth_t *eth_handle;	    /* libdnet Ethernet handle */
};

/*
 * Linked list of all ghost control blocks.
 */
static struct ghost_ctrl_blk *ctrl_blks;

/*
 * Used by the ghost_main() thread to send ICMP echo replies.
 */
static ip_t *ip_handle;

/*
 * Pipes for inter-thread communications.
 */
static int to_ghost_pipefd[2];
static int from_ghost_pipefd[2];

/*
 * Returns a pointer to the ghost structure that corresponds to the
 * ghost ID passed as a parameter.
 *
 * Returns NULL if no ghost with the specified ID exists.
 */
static struct nexp_ghost *
lookup_ghost(const char *ghost_id)
{
    struct nexp_ghost *g;

    for (g = ghosts; g; g = g->next)
	if (!strcmp(g->name, ghost_id) )
	    break;

    return g;
}

/*
 * Returns true if address "addr" is within IP subnet "network", or false
 * otherwise.
 */
static int
addr_in_network(struct addr *network, struct addr *addr)
{
    struct addr netaddr, bcastaddr;

    /* Sanity checks */
    if (   network->addr_type != addr->addr_type
	|| addr->addr_bits != IP_ADDR_BITS)
	return 0;

    if (network->addr_bits == IP_ADDR_BITS)
	return !addr_cmp(network, addr);

    addr_net(network, &netaddr);	/* obtain network address */
    addr_bcast(network, &bcastaddr);	/* obtain broadcast address */

    return (addr_cmp(&netaddr, addr) <= 0) && (addr_cmp(addr, &bcastaddr) <= 0);
}

/*
 * Callback function for pcap_dispatch(). This function gets called by
 * pcap_dispatch() when there is a new packet that needs to be processed.
 * This function is the one that actually processes the new packet.
 */
static void
ghost_process_pkt(u_char *user, const struct pcap_pkthdr *h,
		  const u_char *bytes)
{
    struct ghost_ctrl_blk *c;
    struct eth_hdr *eth;
    struct nexp_ghost *g;
    struct addr a;

    c = (struct ghost_ctrl_blk *) user;

    eth = (struct eth_hdr *) bytes;

    /*
     * A ghost currently only responds to ARP requests and ICMP echo
     * requests. Everything else we drop.
     */
    if (ntohs(eth->eth_type) == ETH_TYPE_ARP) {
	struct arp_hdr *arp;
	struct arp_ethip *arpip;
	ip_addr_t tmp;

	arp = (struct arp_hdr *) (bytes + ETH_HDR_LEN);

	/* Sanity checks. Should check for size of received pkt too. */
	if (   ntohs(arp->ar_hrd) != ARP_HRD_ETH
	    || ntohs(arp->ar_pro) != ARP_PRO_IP
	    || arp->ar_hln != ETH_ADDR_LEN
	    || arp->ar_pln != IP_ADDR_LEN
	    || ntohs(arp->ar_op) != ARP_OP_REQUEST)
	    return;

	arpip = (struct arp_ethip *) (arp + 1);

	for (g = ghosts; g; g = g->next) {
	    if (strcmp(g->iface, c->iface) )
		/*
		 * ARP request received on interface this ghost is not on,
		 * so we ignore this ARP request.
		 */
		continue;

	    /*
	     * This ghost is on the same interface on which we received
	     * the ARP request; determine if the ARP request is for the
	     * MAC address of this ghost.
	     */

	    addr_pack(&a, ADDR_TYPE_IP, IP_ADDR_BITS, &arpip->ar_tpa, IP_ADDR_LEN);

	    if (!addr_in_network(&g->l3addr, &a) )
		/* ARP request is not for this ghost's IP address */
		continue;

	    /* Prepare ARP reply */
	    memcpy(eth->eth_dst.data, eth->eth_src.data, ETH_ADDR_LEN);
	    memcpy(eth->eth_src.data, g->l2addr.addr_eth.data, ETH_ADDR_LEN);
	    arp->ar_op = htons(ARP_OP_REPLY);
	    memcpy(&tmp, arpip->ar_tpa, IP_ADDR_LEN);
	    memcpy(arpip->ar_tha, arpip->ar_sha, ETH_ADDR_LEN);
	    memcpy(arpip->ar_tpa, arpip->ar_spa, IP_ADDR_LEN);
	    memcpy(arpip->ar_sha, g->l2addr.addr_eth.data, ETH_ADDR_LEN);
	    memcpy(arpip->ar_spa, &tmp, IP_ADDR_LEN);

	    /* Finally send the ARP reply */
	    eth_send(c->eth_handle, eth, h->len);
	}
    } else if (ntohs(eth->eth_type) == ETH_TYPE_IP) {
	struct ip_hdr *ip;
	struct icmp_hdr *icmp;
	struct icmp_msg_echo *icmpecho;
	ip_addr_t ip_src;
	uint16_t icmp_id, icmp_seq, icmp_payload_len, ip_len;
	uint8_t *icmp_payload;

	ip = (struct ip_hdr *) (bytes + ETH_HDR_LEN);
	icmp = (struct icmp_hdr *) ( (uint8_t *) ip + ip->ip_hl*4);

	/*
	 * Sanity checks. Should check for size of received pkt too.
	 * We don't support responding to ICMP echo requests that
	 * span multiple IP packets (and therefore need to be
	 * defragmented.) That's too much work to do using libpcap
	 * for packet capture. If we wanted to do that we'd be much
	 * better off using the traditional socket calls.
	 */
	if (ip->ip_p != IP_PROTO_ICMP
	    || (ntohs(ip->ip_off) & 0x3fff) != 0
	    || icmp->icmp_type != ICMP_ECHO
	    || icmp->icmp_code != 0)
	    return;

	icmpecho = (struct icmp_msg_echo *) (icmp + 1);

	for (g = ghosts; g; g = g->next) {
	    if (strcmp(g->iface, c->iface) )
		/*
		 * ICMP echo request received on interface this ghost is not
		 * on, so we ignore this ICMP echo request.
		 */
		continue;

	    /*
	     * This ghost is on the same interface on which we received the
	     * ICMP echo request; determine if the ICMP echo request was sent
	     * to the IP address of this ghost.
	     */

	    addr_pack(&a, ADDR_TYPE_IP, IP_ADDR_BITS, &ip->ip_dst, IP_ADDR_LEN);

	    if (!addr_in_network(&g->l3addr, &a) )
		/* ICMP echo request was not sent to this ghost's IP address */
		continue;

	    /* Save some things from the initial ICMP echo request */
	    ip_src = ip->ip_src;
	    icmp_id = icmpecho->icmp_id;
	    icmp_seq = icmpecho->icmp_seq;
	    icmp_payload_len = ntohs(ip->ip_len)
			       - ip->ip_hl*4
			       - sizeof(struct icmp_hdr)
			       - sizeof(struct icmp_msg_echo);
	    ip_len =   sizeof(struct ip_hdr)
		     + sizeof(struct icmp_hdr)
		     + sizeof(struct icmp_msg_echo)
		     + icmp_payload_len;
	    icmp_payload = icmpecho->icmp_data;

	    /* Prepare ICMP echo reply */
	    ip->ip_v = 4;
	    ip->ip_hl = sizeof(struct ip_hdr)/4;
	    ip->ip_tos = IP_TOS_DEFAULT;
	    ip->ip_len = htons(ip_len);
	    ip->ip_id = rand();
	    ip->ip_off = htons(IP_DF);
	    ip->ip_ttl = IP_TTL_DEFAULT;
	    ip->ip_p = IP_PROTO_ICMP;
	    ip->ip_sum = 0; /* gets fixed later */
	    ip->ip_src = ip->ip_dst;
	    ip->ip_dst = ip_src;

	    icmp = (struct icmp_hdr *) ( (uint8_t *) ip + ip->ip_hl*4);
	    icmp->icmp_type = ICMP_ECHOREPLY;
	    icmp->icmp_code = 0;
	    icmp->icmp_cksum = 0; /* gets fixed later */

	    icmpecho = (struct icmp_msg_echo *) (icmp + 1);
	    icmpecho->icmp_id = icmp_id;
	    icmpecho->icmp_seq = icmp_seq;
	    memmove(icmpecho->icmp_data, icmp_payload, icmp_payload_len);

	    /* Fix checksums */
	    ip_checksum(ip, ip_len);

	    /* Finally send the ICMP echo reply */
	    ip_send(ip_handle, ip, ip_len);
	}
    }
}

static void
destroy_ghost_ctrl_blk(void)
{
    struct ghost_ctrl_blk *c, *previous_c;
    struct nexp_ghost *g;
    int found;

    /*
     * Determine if there are ghost control blocks for inexistant
     * ghosts, i.e. orphan control blocks.  If there is a ghost control
     * block for an inexistant host we destroy it.
     */
    for (c = ctrl_blks; c; /* "c" is manipulated within the loop */) {
	for (found = 0, g = ghosts; g; g = g->next) {
	    if (!strcmp(c->iface, g->iface) ) {
		found++;
		break;
	    }
	}

	if (found) {
	    /*
	     * Found a ghost for this control block. This means that
	     * this control block is not orphaned and we cannot remove
	     * it.  Just advance to the next control block in the
	     * linked list and keep lookin.
	     */
	    c = c->next;
	    continue;
	}

	/*
	 * This control block is for an inexistant ghost, i.e. the
	 * control block is orphan - remove it.
	 */

	pcap_close(c->pd);
	eth_close(c->eth_handle);
	free( (char *) c->iface);

	/* Remove ghost from linked list of ghosts. */
	if (ctrl_blks == c) {
	    /* Element to remove is at the beginning of the list. */
	    ctrl_blks = c->next;
	    free(c);
	    c = ctrl_blks;
	} else {
	    /*
	     * Find linked list item previous to one we want to remove.
	     */
	    for (previous_c = ctrl_blks;
		 previous_c->next != c;
		 previous_c = previous_c->next);

	    previous_c->next = c->next;
	    free(c);
	    c = previous_c;
	}
    }
}

static void
create_ghost_ctrl_blk(void)
{
    struct nexp_ghost *g;
    int found, retval;
    char errbuf[2*PCAP_ERRBUF_SIZE], pcap_errbuf[PCAP_ERRBUF_SIZE];
    struct bpf_program fcode;
    struct ghost_ctrl_blk *c, *new_c;

    /*
     * Determine if there are ghost control blocks for all the
     * ghosts. If a ghost control block does not exist for a ghost then we
     * create it.
     */
    for (g = ghosts; g; g = g->next) {
	for (found = 0, c = ctrl_blks; c; c = c->next) {
	    if (!strcmp(g->iface, c->iface) ) {
		/* Found a control block for this ghost */
		found++;
		break;
	    }
	}

	if (found)
	    continue;

	/*
	 * This ghost does not have a control block - create one.
	 */

	new_c = xmalloc(sizeof(*new_c) );
	memset(new_c, 0, sizeof(*new_c) );

	/********************************************************
	 ***                     PCAP stuff                   ***
	 ********************************************************/

	*pcap_errbuf = '\0';
	/*
	 * Can't use 0 for the read timeout because this would
	 * cause problems, i.e. select() never returning, on
	 * platforms that do support the read timeout. Setting the timeout to
	 * the lowest possible value (1 msec).
	 */
	new_c->pd = pcap_open_live(g->iface, SNAPLEN,
				   1 /* promisc */, 1, pcap_errbuf);
	if (!new_c->pd) {
	    snprintf(errbuf, sizeof(errbuf),
		     "pcap_open_live() - %s",
		     pcap_errbuf);
	    goto error;
	} else if (*pcap_errbuf)
	    warn("Warning: %s", pcap_errbuf);

	if ( (new_c->fd = pcap_get_selectable_fd(new_c->pd) )
	    == -1) {
	    snprintf(errbuf, sizeof(errbuf),
		     "pcap_get_selectable_fd() returned error");
	    goto error;
	}

	/*
	 * Set things up so we can read from the packet capture descriptor in
	 * real time, i.e. so reads are not buffered.
	 */

	/*
	 * This really isn't needed for Linux but it doesn't hurt.  It is
	 * definitely needed for platforms where select() times out when there
	 * is more than one packet ready to be read for a particular file
	 * descriptor.  See comments in expect_network.c regarding this.
	 */
	if (pcap_setnonblock(new_c->pd, 1, errbuf) == -1) {
	    snprintf(errbuf, sizeof(errbuf),
		     "Can't set non-blocking mode: %s",
		     pcap_geterr(new_c->pd) );
	    goto error;
	}

#if (defined(BSD) || defined(__OpenBSD__) ) && defined(BIOCIMMEDIATE)
	/*
	 * For some of the BSDs this is very important since otherwise we won't
	 * be able to read data as soon as it arrives. Instead, the OS will
	 * buffer it and hand it out once the buffer is full.
	 */
	{
	    int on = 1;

	    if (ioctl(new_c->fd, BIOCIMMEDIATE, &on) < 0)
		warn("ioctl() with BIOCIMMEDIATE returned an error. errno = %d",
		     errno);
	}
#elif __sun__
	/*
	 * Under Solaris, select() keeps waiting until the next packet, because
	 * it is buffered, so we have to set timeout and chunk size to zero
	 */
	{
	    int size_zero = 0;
	    struct timeval time_zero = {0, 0};

	    if (ioctl(new_c->fd, SBIOCSCHUNK, &size_zero) < 0)
		warn("ioctl() with SBIOCSCHUNK returned an error (%d): %s",
		     errno, strerror(errno) );

	    if (ioctl(new_c->fd, SBIOCSTIME, &time_zero) < 0)
		warn("ioctl() with SBIOCSTIME returned an error (%d): %s",
		     errno, strerror(errno) );
	}
#endif

	/*
	 * A ghost only responds to ARP requests and to ICMP echo requests, at
	 * least for now, so the PCAP filter we will use reflects that.
	 */
	if (pcap_compile(new_c->pd, &fcode,
			 "arp[6:2] == 1 or icmp[icmptype] == icmp-echo",
			 1, 0) < 0) {
	    snprintf(errbuf, sizeof(errbuf),
		     "Can't compile filter: %s\n",
		     pcap_geterr(new_c->pd) );
	    goto error;
	}

	if (pcap_setfilter(new_c->pd, &fcode) < 0) {
	    pcap_freecode(&fcode);
	    snprintf(errbuf, sizeof(errbuf),
		     "Can't set filter: %s\n",
		     pcap_geterr(new_c->pd) );
	    goto error;
	}

	/*
	 * pcap(3) says that we can release the memory used by the BPF program
	 * after it has been made the filter program via a call to
	 * pcap_setfilter().
	 */
	pcap_freecode(&fcode);

	/********************************************************
	 ***                   libdnet stuff                  ***
	 ********************************************************/

	new_c->eth_handle = eth_open(g->iface);
	if (!new_c->eth_handle) {
	    snprintf(errbuf, sizeof(errbuf),
		     "Couldn't create handle for interface \"%s\": %s",
		     g->iface, strerror(errno) );
	    goto error;
	}

	new_c->iface = xstrdup(g->iface);

	/* Add new control block to linked list of ghosts. */
	if (!ctrl_blks)
	    ctrl_blks = new_c;
	else {
	    for (c = ctrl_blks; c->next; c = c->next);

	    c->next = new_c;
	}
    }

    /*
     * Tell the main thread that one ghost creation was successful.
     */
    retval = 0;
    write(from_ghost_pipefd[1], &retval, sizeof(retval) );
    return;

error:
    /*
     * An error happened. Just delete the ghost control block that was in the
     * process of getting built and continue running the thread, after
     * communicating to the main thread (via a pipe) that an error happened and
     * the ghost could not be created.
     */

    if (new_c->pd)
	pcap_close(new_c->pd);
    if (new_c->eth_handle)
	eth_close(new_c->eth_handle);
    free( (char *) new_c->iface);
    free(new_c);

    /*
     * Tell the main thread that ghost creation was unsuccessful.
     */
    retval = -1;
    write(from_ghost_pipefd[1], &retval, sizeof(retval) );

    /* Write reason */
    write(from_ghost_pipefd[1], errbuf, strlen(errbuf) + 1);
}

/*
 * The thread that actually does all the work related to the "ghost" command.
 */
static void
ghost_main(ClientData clientData _U_)
{
    int highest_fd, fd, retval, cmd;
    fd_set rfds;
    struct ghost_ctrl_blk *c;

    /*
     * The thread's main loop. We never leave this loop.
     */
    for (;;) {
	/*
	 * We use select() to wait for data from any of the packet capture
	 * descriptors associated with the ghosts.
	 */

	FD_ZERO(&rfds);

	for (highest_fd = 0, c = ctrl_blks; c; c = c->next) {
	    fd = c->fd;

	    FD_SET(fd, &rfds);
	    if (fd > highest_fd)
		highest_fd = fd;
	}

	/*
	 * Arm file descriptor used to receive messages from the
	 * main thread.
	 */
	FD_SET(to_ghost_pipefd[0], &rfds);
	if (to_ghost_pipefd[0] > highest_fd)
	    highest_fd = to_ghost_pipefd[0];

	retval = select(highest_fd + 1, &rfds, NULL, NULL,
			NULL /* no timeout */);
	if (retval < 0)
	    /* No error recovery at all; not elegant. We can do better. */
	    error("select() error. retval = %d, errno = %d\n",
		  retval, errno);

	/*
	 * Data is now available.
	 */

	for (c = ctrl_blks; c; c = c->next) {
	    if (FD_ISSET(c->fd, &rfds) )
		/*
		 * This file descriptor has data that we can read without
		 * blocking...
		 */
		pcap_dispatch(c->pd, 1, &ghost_process_pkt, (u_char *) c);
	}

	if (FD_ISSET(to_ghost_pipefd[0], &rfds) ) {
	    /*
	     * The main thread has sent a command to us through the pipe.
	     */
	    read(to_ghost_pipefd[0], &cmd, sizeof(cmd) );

	    switch (cmd) {
	    case CMD_GHOST_KILL:
		destroy_ghost_ctrl_blk();
		break;
	    case CMD_GHOST_CREATE:
		create_ghost_ctrl_blk();
		break;
	    default:
		/* Nothing */
		;
	    }
	}

    } /* ghost main loop */

    /* Not reached */
}

static int
ghost_kill(const char *ghost_id)
{
    struct nexp_ghost *g, *previous_g;
    int cmd;

    g = lookup_ghost(ghost_id);
    if (!g)
	return -1;

    /* Remove ghost from linked list of ghosts. */
    if (ghosts == g)
	/* Element to remove is at the beginning of the list. */
	ghosts = g->next;
    else {
	/* Find linked list item previous to one we want to remove. */
	for (previous_g = ghosts;
	     previous_g->next != g;
	     previous_g = previous_g->next);

	previous_g->next = g->next;
    }

    free(g);

    /*
     * Tell the ghost thread that a ghost has been killed.
     */
    cmd = CMD_GHOST_KILL;
    write(to_ghost_pipefd[1], &cmd, sizeof(cmd) );

    return 0;
}

/*
 * Creates a new ghost. Ghost creation actually happens in two different
 * places - here in the main thread, and in a "ghost" thread. Here we
 * just create the basic structure to keep track of the new ghost, and
 * later, from within the "ghost" thread, we do the rest.
 */
static int
ghost_create(Tcl_Interp *interp, struct cdb *cdb)
{
    int retval, cmd, thread_status;
    Tcl_ThreadId thread_id;
    struct nexp_ghost *g, *new_ghost;
    static int ghost_id;
    static int inited;
    char errbuf[2*PCAP_ERRBUF_SIZE];

    new_ghost = NULL;

    if (!inited) {
	ip_handle = ip_open();
	if (ip_handle == NULL) {
	    snprintf(errbuf, sizeof(errbuf), "ip_open(): %s", strerror(errno) );
	    goto error;
	}

	/*
	 * Create the ghosts thread. We could do this at startup time but doing
	 * it on demand prevents from having an unneeded thread running around.
	 */
	retval = Tcl_CreateThread(&thread_id, ghost_main, NULL,
				  TCL_THREAD_STACK_DEFAULT,
				  TCL_THREAD_NOFLAGS);
	if (retval != TCL_OK) {
	    snprintf(errbuf, sizeof(errbuf), "Tcl_CreateThread() error");
	    goto error;
	}

	inited++;
    }

    new_ghost = xmalloc(sizeof(*new_ghost) );
    memset(new_ghost, 0, sizeof(*new_ghost) );
    snprintf(new_ghost->name, sizeof(new_ghost->name), "ghost%d", ghost_id++);
    new_ghost->iface = cdb->_ghost.iface;
    new_ghost->l2addr = cdb->_ghost.l2addr;
    new_ghost->l3addr = cdb->_ghost.l3addr;

    /* Add new ghost to linked list of ghosts. */
    if (!ghosts)
	ghosts = new_ghost;
    else {
	for (g = ghosts; g->next; g = g->next);

	g->next = new_ghost;
    }

    /*
     * Tell the ghost thread that there is a new ghost.
     */
    cmd = CMD_GHOST_CREATE;
    write(to_ghost_pipefd[1], &cmd, sizeof(cmd) );

    /*
     * Read status from the ghost thread. We'll block here while the
     * thread does its thing.
     */
    retval = read(from_ghost_pipefd[0], &thread_status, sizeof(thread_status) );
    if (retval == 0) {
	/* EOF */
	snprintf(errbuf, sizeof(errbuf),
		 "read(): got EOF. This should not happen!");
	goto error;
    } else if (retval == -1) {
	/* read() error */
	snprintf(errbuf, sizeof(errbuf), "read(): %s", strerror(errno) );
	goto error;
    } else if (retval != sizeof(thread_status) ) {
	/* We got less than what we asked for */
	snprintf(errbuf, sizeof(errbuf),
		 "read() read less than requested. This should not happen!");
	goto error;
    }

    if (thread_status == 0) {
	/*
	* Tell user of new ghost id.
	*/
	Tcl_SetVar(interp, GHOST_ID_VARNAME, new_ghost->name, 0);
	return 0;
    }

    /* Read error message from ghost thread */
    retval = read(from_ghost_pipefd[0], errbuf, sizeof(errbuf) );

error:
    nexp_error(interp, "%s; ghost creation failed", errbuf);
    if (new_ghost)
	ghost_kill(new_ghost->name);
    return -1;
}

void
ghosts_info(void)
{
    struct nexp_ghost *g;
    int ncblocks, nghosts, i;
    char ipaddr[sizeof("255.255.255.255/255")];
    char macaddr[sizeof("xx:xx:xx:xx:xx:xx")];
    struct ghost_ctrl_blk *c;

    /* Count number of ghost control blocks */
    for (ncblocks = 0, c = ctrl_blks; c; c = c->next, ncblocks++);

    printf("Ghosts:\n");

    for (g = ghosts, nghosts = 0; g; g = g->next, nghosts++);

    printf("  Number of ghosts: %d (%d ghost control %s)\n",
	   nghosts, ncblocks, ncblocks == 1 ? "block" : "blocks");

    for (g = ghosts, i = 0; g; g = g->next, i++) {
	printf("  Ghost #%d:\n", i);
	printf("    Name: %s\n", g->name);
	printf("    Interface: %s\n", g->iface);
	printf("    MAC address: %s\n", addr_ntop(&g->l2addr, macaddr,
						  sizeof(macaddr) ) );
	printf("    IP address: %s\n", addr_ntop(&g->l3addr, ipaddr,
						 sizeof(ipaddr) ) );
    }
}

/*
 * The "ghost" command.
 *
 * Basically an ARP responder.
 *
 * Example:
 *
 * ghost with ip 1.2.3.4 and mac 00:10:c6:e2:39:05 on interface eth1
 *
 * This will cause netexpect to respond to ARP requests for the MAC
 * address of 1.2.3.4 with the MAC 00:10:c6:e2:39:05. It will also
 * respond to simple pings ("simple" means not fragmented.)
 */
static int
NExp_GhostObjCmd(ClientData clientData _U_, Tcl_Interp *interp, int objc,
		 Tcl_Obj *CONST objv[])
{
    char *cmd;
    struct cdb cdb; /* The command description block. Will be filled-in
		       by the command parser. */
    const char *ghost_id;

    cmd = copy_objv(objc, &objv[0]);

    if (cmd_parseargs(interp, cmd, &cdb) != 0) {
	free(cmd);
	return TCL_ERROR;
    }

    free(cmd);

    switch (cdb.code) {
    case CMD_GHOST_CREATE:
	if (ghost_create(interp, &cdb) == -1)
	   return TCL_ERROR;
	break;
    case CMD_GHOST_KILL:
	if (cdb._ghost.name == NULL) {
	    ghost_id = nexp_get_var(interp, GHOST_ID_VARNAME);
	    if (!ghost_id) {
		nexp_error(interp, "Use ghost command to create a ghost "
				   "first.");
		return TCL_ERROR;
	    }

	    cdb._ghost.name = xstrdup(ghost_id);
	}

	if (ghost_kill(cdb._ghost.name) == -1) {
	    nexp_error(interp, "\"%s\" is not a ghost", cdb._ghost.name);
	    return TCL_ERROR;
	}

	free(cdb._ghost.name);
	break;
    case CMD_GHOST_INFO:
	ghosts_info();
	break;
    default:
	nexp_error(interp, "Invalid \"ghost\" subcommand.");
	return TCL_ERROR;
    }

    return TCL_OK;
}

static struct nexp_cmd_data cmd_data[] = {
    {"ghost", NExp_GhostObjCmd, NULL, 0, 0},

    {NULL, NULL, NULL, 0, 0}
};

void
nexp_init_ghost_cmd(Tcl_Interp *interp)
{
    /*
     * Create pipes for communications between main thread and the
     * ghost thread. We do this only once through the entire life
     * of the program.
     */
    if (pipe(to_ghost_pipefd) == -1)
	error("Couldn't open pipe: %s", strerror(errno) );

    if (pipe(from_ghost_pipefd) == -1)
	error("Couldn't open pipe: %s", strerror(errno) );

    nexp_create_commands(interp, cmd_data);
}
