#!/usr/bin/python

"""Main controller for pwrkap."""
# (C) Copyright IBM Corp. 2008-2009
# Licensed under the GPLv2.
import discovery
import lazy_log
import sockmux
import thread
import atexit
import pwrkap_pickle
import sys
import traceback
import time
import cPickle as pickle
import datetime
import pwrkap_train
import http_listener

def save_and_exit():
	"""Tear down control threads and save data."""
	print "Saving power use observations..."
	for dm in discovery.PWRKAP_POWER_DOMAINS:
		dm.exit_control_loop()
	pwrkap_pickle.save_domains(discovery.PWRKAP_POWER_DOMAINS)
	print "...done."

class controller:
	"""Main control loop."""

	def __init__(self, port, log_age, log_size, start_httpd = False):
		"""Create controller."""
		discovery.load_pwrkap_drivers()
		discovery.discover_devices()
		discovery.discover_meters()
		discovery.discover_power_domains()

		if start_httpd:
			self.http_listener = http_listener.http_listener(self)

		self.sockmux = sockmux.sockmux(self, port)
		lazy_log.logger = lazy_log.lazy_log(self.sockmux, log_age, log_size)
		self.pwrkap_events = {
			"dump": self.dump_command,
			"cap": self.cap_command}

	def train(self):
		"""Initialize the system with some data."""
		# Speed things up if we're using fakedomain
		if "fake_cpudomain" in discovery.PWRKAP_DRIVERS:
			pwrkap_train.STABILIZE_TIME = 0
			pwrkap_train.MAX_MEASUREMENTS = 10
		print "Determining power use profile."
		pwrkap_train.train()

	def prepare(self, load_saved = True):
		"""Prepare system for power capping."""
		# Read in what we saved last time (if anything)
		saved_data = None
		if load_saved:
			pwrkap_pickle.load_domains(discovery.PWRKAP_POWER_DOMAINS)
		if saved_data == None:
			self.train()
			pwrkap_pickle.save_domains(discovery.PWRKAP_POWER_DOMAINS)
		else:
			discovery.PWRKAP_POWER_DOMAINS = saved_data

		atexit.register(save_and_exit)

	def run(self):
		"""Run control methods."""
		if len(discovery.PWRKAP_POWER_DOMAINS) < 1:
			print "No power domains found."
			return

		# Try to limit stack consumption.
		try:
			thread.stack_size(65536)
		except:
			pass

		# Start control methods
		for dm in discovery.PWRKAP_POWER_DOMAINS:
			thread.start_new_thread(dm.control_loop, ())

		while True:
			time.sleep(1)

	def run_cli(self):
		# Start command line interface
		data = sys.stdin.readline()
		while len(data) > 0:
			components = data.strip().split()
			self.command(components)
			data = sys.stdin.readline()

	def connect(self, socket, queue):
		"""Connect a new client."""
		try:
			self.do_connect(socket, queue)
		except Exception, e:
			print e
			traceback.print_exc(file=sys.stdout)
			return False
		return True

	def do_connect(self, socket, queue):
		"""Really connect a socket."""
		# Dump the list of power domains.
		inventories = []
		for dm in discovery.PWRKAP_POWER_DOMAINS:
			inventories.append(dm.inventory())
		pickled = pickle.dumps(inventories, protocol = pickle.HIGHEST_PROTOCOL)
		queue.append(pickled)

		# Dump some cached snapshots
		pickles = lazy_log.logger.dump_log()
		for apickle in pickles:
			queue.append(apickle)

		# Display a "data now live" flag
		now = datetime.datetime.utcnow()
		tuple = (now, "live")
		pickled = pickle.dumps(tuple, protocol = pickle.HIGHEST_PROTOCOL)
		queue.append(pickled)

	def command(self, components):
		"""Process commands."""
		try:
			domain_name = components[0]
			command = components[1]
			domain = discovery.find_domain_by_name(domain_name)
			handler = self.pwrkap_events[command]
			handler(domain, components[2:])
		except Exception, e:
			print e
			traceback.print_exc()
			print "Ignoring bogus input '%s'." % components

	def dump_command(self, domain, args):
		"""Handle a dump command."""
		print domain.trans_store.trans_table

	def cap_command(self, domain, args):
		"""Handle a cap command."""
		cap = int(args[0])
		if cap < 1:
			return
		domain.set_cap(cap)

	def dump(self):
		"""Dump current state."""
		print ("devices", discovery.PWRKAP_DEVICES)
		print ("pmeters", discovery.PWRKAP_POWER_METERS)
		print ("emeters", discovery.PWRKAP_ENERGY_METERS)
		print ("devdomains", discovery.PWRKAP_DEVICE_DOMAINS)
		print ("pwrdomains", discovery.PWRKAP_POWER_DOMAINS)

