#ifndef SimTK_SIMBODY_MOBILIZED_BODY_PIN_H_
#define SimTK_SIMBODY_MOBILIZED_BODY_PIN_H_

/* -------------------------------------------------------------------------- *
 *                               Simbody(tm)                                  *
 * -------------------------------------------------------------------------- *
 * This is part of the SimTK biosimulation toolkit originating from           *
 * Simbios, the NIH National Center for Physics-Based Simulation of           *
 * Biological Structures at Stanford, funded under the NIH Roadmap for        *
 * Medical Research, grant U54 GM072970. See https://simtk.org/home/simbody.  *
 *                                                                            *
 * Portions copyright (c) 2007-13 Stanford University and the Authors.        *
 * Authors: Michael Sherman                                                   *
 * Contributors: Paul Mitiguy, Peter Eastman                                  *
 *                                                                            *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may    *
 * not use this file except in compliance with the License. You may obtain a  *
 * copy of the License at http://www.apache.org/licenses/LICENSE-2.0.         *
 *                                                                            *
 * Unless required by applicable law or agreed to in writing, software        *
 * distributed under the License is distributed on an "AS IS" BASIS,          *
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.   *
 * See the License for the specific language governing permissions and        *
 * limitations under the License.                                             *
 * -------------------------------------------------------------------------- */

/** @file
Declares the MobilizedBody::Pin class. **/

#include "simbody/internal/MobilizedBody.h"

namespace SimTK {

/** Provides one rotational mobility about the common z axis of the F and M 
frames of the mobilizer. 

If you want rotation about a different direction, rotate the F and M frames
when you define the mobilized body, so that the z axes are in the desired
direction.

The single generalized coordinate q is the rotation angle in radians, and the
generalized speed u is the rotation rate in radians/time unit, with qdot=u.

Synonyms: Torsion, Revolute.
**/
class SimTK_SIMBODY_EXPORT MobilizedBody::Pin : public MobilizedBody {
public:
    /** Default constructor provides an empty handle that can be assigned to
    reference any %MobilizedBody::Pin. **/
    Pin() {}

    /** Create a %Pin mobilizer between an existing parent (inboard) body P 
    and a new child (outboard) body B created by copying the given \a bodyInfo 
    into a privately-owned Body within the constructed %MobilizedBody object. 
    Specify the mobilizer frames F fixed to parent P and M fixed to child B. 
    @see MobilizedBody for a diagram and explanation of terminology. **/
    Pin(MobilizedBody& parent, const Transform& X_PF,
        const Body& bodyInfo,  const Transform& X_BM, Direction=Forward);

    /** Abbreviated constructor you can use if the mobilizer frames are 
    coincident with the parent and child body frames. **/
    Pin(MobilizedBody& parent, const Body& bodyInfo, Direction=Forward);
    
    // SPECIALIZED INTERFACE FOR PIN MOBILIZER

    /** Set the value that the pin angle should have in the default state. If
    unspecified the initial angle will be zero. **/
    Pin& setDefaultAngle(Real angleInRad) {return setDefaultQ(angleInRad);}
    /** Get the value that the pin angle will have in the default state. **/
    Real getDefaultAngle() const          {return getDefaultQ();}

        // Friendly, mobilizer-specific access to generalized coords and speeds.

    /** Set the pin joint angle (generalized coordinate q) in the given state.
    The angle is in radians. **/
    void setAngle(State& s, Real angleInRad) {setQ(s, angleInRad);}
    /** Get the value that this pin joint's angular coordinate has in the given
    state. The result is in radians. **/
    Real getAngle(const State& s) const {return getQ(s);}

    /** Set the rotation rate (generalized speed u) for this pin joint in the
    given state. The rate is in radians/time unit. **/
    void setRate(State& s, Real rateInRadPerTime) {setU(s, rateInRadPerTime);}
    /** Get the current rotation rate (generalized speed u) that this pin 
    mobilizer has in the given state. The rate is in radians/time unit. **/
    Real getRate(const State& s) const {return getU(s);}

    // Mobility forces are "u-like", that is, one per dof.
    /** Get the generalized force corresponding to this pin mobilizer in the
    given array of mobility forces. **/
    Real getAppliedPinTorque(const State& s, const Vector& mobilityForces) const {
        return getMyPartU(s,mobilityForces);
    }
    /** Add in a torque to the generalized force element corresponding to this
    pin mobilizer in the given array of mobility forces. **/
    void applyPinTorque(const State& s, Real torque, Vector& mobilityForces) const {
        updMyPartU(s,mobilityForces) += torque;
    }

        // STANDARDIZED MOBILIZED BODY INTERFACE


        // access to generalized coordinates q and generalized speeds u
    Pin& setDefaultQ(Real);
    Real getDefaultQ() const;

    Real getQ(const State&) const;
    Real getQDot(const State&) const;
    Real getQDotDot(const State&) const;
    Real getU(const State&) const;
    Real getUDot(const State&) const;

    void setQ(State&, Real) const;
    void setU(State&, Real) const;

    Real getMyPartQ(const State&, const Vector& qlike) const;
    Real getMyPartU(const State&, const Vector& ulike) const;
   
    Real& updMyPartQ(const State&, Vector& qlike) const;
    Real& updMyPartU(const State&, Vector& ulike) const;

        // specialize return type for convenience
    Pin& addBodyDecoration(const Transform& X_BD, const DecorativeGeometry& g)
      { (void)MobilizedBody::addBodyDecoration(X_BD,g); return *this; }
    Pin& addOutboardDecoration(const Transform& X_MD,  const DecorativeGeometry& g)
      { (void)MobilizedBody::addOutboardDecoration(X_MD,g); return *this; }
    Pin& addInboardDecoration (const Transform& X_FD, const DecorativeGeometry& g)
      { (void)MobilizedBody::addInboardDecoration(X_FD,g); return *this; }
    Pin& setDefaultInboardFrame(const Transform& X_PF)
      { (void)MobilizedBody::setDefaultInboardFrame(X_PF); return *this; }
    Pin& setDefaultOutboardFrame(const Transform& X_BM)
      { (void)MobilizedBody::setDefaultOutboardFrame(X_BM); return *this; }

    /** @cond **/ // Don't let doxygen see this
    SimTK_INSERT_DERIVED_HANDLE_DECLARATIONS(Pin, PinImpl, MobilizedBody);
    /** @endcond **/
};

} // namespace SimTK

#endif // SimTK_SIMBODY_MOBILIZED_BODY_PIN_H_



