/*===========================================================================
 *
 *                            PUBLIC DOMAIN NOTICE
 *               National Center for Biotechnology Information
 *
 *  This software/database is a "United States Government Work" under the
 *  terms of the United States Copyright Act.  It was written as part of
 *  the author's official duties as a United States Government employee and
 *  thus cannot be copyrighted.  This software/database is freely available
 *  to the public for use. The National Library of Medicine and the U.S.
 *  Government have not placed any restriction on its use or reproduction.
 *
 *  Although all reasonable efforts have been taken to ensure the accuracy
 *  and reliability of the software and data, the NLM and the U.S.
 *  Government do not and cannot warrant the performance or results that
 *  may be obtained by using this software or data. The NLM and the U.S.
 *  Government disclaim all warranties, express or implied, including
 *  warranties of performance, merchantability or fitness for any particular
 *  purpose.
 *
 *  Please cite the author in any work or product based on this material.
 *
 * ===========================================================================
 *
 */

#ifndef _h_page_map_
#define _h_page_map_

#ifndef _h_vdb_extern_
#include <vdb/extern.h>
#endif

#ifndef _h_klib_defs_
#include <klib/defs.h>
#endif

struct KDataBuffer;
typedef struct PageMap PageMap;
typedef uint32_t row_count_t;
typedef uint32_t elem_count_t;

/* a pessimistic estimate - actual size will *always* be less than or equal */
size_t PageMapGetMaxBufferSize(const PageMap *self);

rc_t PageMapSerialize(const PageMap *self, struct KDataBuffer *buffer, uint64_t byte_offset, uint64_t *actual_bytes);

rc_t PageMapDeserialize(PageMap **lhs, const void *src, uint64_t src_bytes, uint64_t row_count);

rc_t PageMapRelease(const PageMap *self);

rc_t PageMapAddRef(const PageMap *self);

/*  PageMapGetIdxRowInfo
 *
 *  Get row length and starting element number for a row.
 *  This is a potentially expensive operation; the first time
 *  it is called for any page map, the memory used by the page
 *  map nearly doubles and an index is built on the row length
 *  runs and the data runs.  Subsequent calls are O(N log N).
 *  However, asking for the information about the first row
 *  (i.e. idx = 0) never causes allocation or indexing and is
 *  always O(1).
 *
 *  Params:
 *      self: [IN] the page map
 *      idx:  the row number starting from 0
 *      starting_element: [OUT, OPTIONAL]
 *
 *  Returns:
 *      the length of the row
 *      or 0 if not found
 */
uint32_t PageMapGetIdxRowInfo(const PageMap *self, uint32_t idx, uint32_t *starting_element);

rc_t PageMapNew(PageMap **lhs, uint32_t reserve);

rc_t PageMapNewSingle(PageMap **lhs, uint64_t row_count, uint64_t row_length);

rc_t PageMapNewFixedRowLength(PageMap **lhs, uint64_t row_count, uint64_t row_len);

uint32_t PageMapFastRowCount(const PageMap *self);

uint32_t PageMapFixedRowLength(const PageMap *self);

uint32_t PageMapHasSimpleStructure(const PageMap *self);

rc_t PageMapAppendRows(PageMap *self, uint64_t row_length, uint64_t run_length, bool same_data);

#define PageMapAppendRow(SELF, ROW_LENGTH, SAME_DATA) (PageMapAppendRows((SELF), (ROW_LENGTH), 1, SAME_DATA))

/* append some rows of the same data */
#define PageMapAppendSomeRows(SELF, ROW_LENGTH, RUN_LENGTH) (PageMapAppendRows((SELF), (ROW_LENGTH), (RUN_LENGTH), false))

/* concatenate two page maps */
rc_t PageMapAppend(PageMap *self, const PageMap *other);


/*
 -1: error
 0: not same
 1: compatible (i.e. all rows same length)
 else: identical
 */
int PageMapCompare(const PageMap *a, const PageMap *b);
/* same but static columns */
int PageMapCompareStatic(const PageMap *a, const PageMap *b);

enum PageMapIteratorVariants {
    pmivConstant,
    pmivVersion1, /* fixed row length and no repeats */
    pmivFixedRowLength,
    pmivFull /* variable row length and has repeats */
};

typedef struct PageMapIterator PageMapIterator;
struct PageMapIterator {
    row_count_t end_row;
    row_count_t cur_row;
    const elem_count_t *length;
    const elem_count_t *offset;
    const row_count_t  *repeat;
    enum PageMapIteratorVariants variant;
};

rc_t PageMapNewIterator(const PageMap *self, PageMapIterator *lhs, uint64_t first_row, uint64_t num_rows);

static __inline__ bool PageMapIteratorAdvance(PageMapIterator *self, row_count_t rows) {
    if (self->cur_row + rows < self->end_row) {
        self->cur_row += rows;
        return true;
    }
    return false;
}

#define PageMapIteratorNext(SELF) PageMapIteratorAdvance(SELF, 1)

static __inline__ elem_count_t PageMapIteratorDataLength(const PageMapIterator *self) {
    switch (self->variant) {
    default:
        return self->length[0];
    case pmivFull:
        return self->length[self->cur_row];
    }
}

static __inline__ elem_count_t PageMapIteratorDataOffset(const PageMapIterator *self) {
    switch (self->variant) {
    case pmivConstant:
        return 0;
    case pmivVersion1:
        return self->length[0] * self->cur_row;
    default:
        return self->offset[self->cur_row];
    }
}

static __inline__ row_count_t PageMapIteratorRepeatCount(const PageMapIterator *self) {
    switch (self->variant) {
    case pmivConstant:
        return self->end_row - self->cur_row;
    case pmivVersion1:
        return 1;
    default:
        return self->repeat[self->cur_row];
    }
}

elem_count_t PageMapLastLength(const PageMap *cself);
bool PageMapHasRows(const PageMap *self);

#endif /* _h_page_map_ */
